<?php

namespace Tests\Unit;

use App\Models\SalesOrder;
use App\Notifications\UncancellableVeracoreOrderNotification;
use App\Repositories\UserRepository;
use Illuminate\Support\Facades\Notification;
use Modules\Veracore\Entities\VeracoreOrder;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use App\Jobs\UncancellableVeracoreOrderJob;
use App\Models\SalesOrderFulfillment;

class UncancellableVeracoreOrderJobTest extends TestCase
{
    use FastRefreshDatabase;

    private UserRepository $users;

    protected function setUp(): void
    {
        parent::setUp();
        $this->users = app(UserRepository::class);
    }

    public function test_it_can_add_note_to_sales_order()
    {

        Notification::fake();

        /** @var SalesOrderFulfillment $fulfillment */
        $fulfillment = SalesOrderFulfillment::factory()->create();

        // Given
        $job = new UncancellableVeracoreOrderJob(
            fulfillment: $fulfillment,
            currentOrder: [
                'OrdHead' => [
                    'Status' => [
                        'Shipped' => 'true'
                    ],
                ]
            ],
            veracoreOrder: new VeracoreOrder,
            userId: null,
        );

        // When
        $job->handle();

        // Then
        $this->assertDatabaseCount('notes', 1);
        $this->assertDatabaseHas('notes', [
            'note' => $job->getNoteContent(),
            'link_type' => SalesOrder::class,
            'link_id' => $fulfillment->salesOrder->id,
        ]);
    }

    public function test_it_sends_email_notification_when_cancellation_fails()
    {

        Notification::fake();

        // Given
        $fulfillment = SalesOrderFulfillment::factory()->create();
        $job = new UncancellableVeracoreOrderJob(
            fulfillment: $fulfillment,
            currentOrder: [
                'OrdHead' => [
                    'Status' => [
                        'Shipped' => 'true'
                    ],
                ]
            ],
            veracoreOrder: new VeracoreOrder,
            userId: null,
        );

        // When
        $job->handle();

        $admins = $this->users->getAdminUsers();

        // Then
        Notification::assertCount(count($admins));
        foreach ($admins as $admin) {
            Notification::assertSentTo($admin, UncancellableVeracoreOrderNotification::class);
        }
    }
}
