<?php

namespace Tests\Unit;

use App\Actions\InventoryManagement\SplitInventoryMovement;
use App\Data\WarehouseTransferReceiptData;
use App\Data\WarehouseTransferReceiptProductData;
use App\Exceptions\InsufficientStockException;
use App\Exceptions\WarehouseTransfers\NotOpenWarehouseTransferException;
use App\Exceptions\WarehouseTransfers\WarehouseTransferHasNoProductsException;
use App\Exceptions\WarehouseTransfers\WarehouseTransferOpenException;
use App\Managers\WarehouseTransferManager;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\Warehouse;
use App\Models\WarehouseTransfer;
use Carbon\Carbon;
use Illuminate\Contracts\Container\BindingResolutionException;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class SplitInventoryMovementTest extends TestCase
{
    use FastRefreshDatabase;

    /**
     * @throws Throwable
     * @throws BindingResolutionException
     * @throws InsufficientStockException
     * @throws WarehouseTransferHasNoProductsException
     * @throws NotOpenWarehouseTransferException
     * @throws WarehouseTransferOpenException
     */
    public function test_it_can_split_warehouse_transfer_movements()
    {
        $product = Product::factory()->create();
        $fromWarehouse = Warehouse::first();
        $toWarehouse = Warehouse::factory()->create([
            'type' => Warehouse::TYPE_DIRECT,
        ]);

        $wtManager = app(WarehouseTransferManager::class);

        $product->setInitialInventory($fromWarehouse->id, 5);
        $fifoLayer = FifoLayer::first();

        $warehouseTransfer = $wtManager->initiateTransfer([
            'from_warehouse_id' => $fromWarehouse->id,
            'to_warehouse_id' => $toWarehouse->id,
            'products' => [
                [
                    'id' => $product->id,
                    'quantity' => 5,
                ],
            ],
        ]);
        $wtManager->openWarehouseTransfer($warehouseTransfer, []);
        $wtManager->receiveShipment($warehouseTransfer, WarehouseTransferReceiptData::from([
            'receipt_date' => Carbon::now(),
            'products' => WarehouseTransferReceiptProductData::collection([
                WarehouseTransferReceiptProductData::from([
                    'id' => $product->id,
                    'quantity' => 5,
                ])
            ])
        ]));
        $warehouseTransferLine = $warehouseTransfer->warehouseTransferLines->first();
        $warehouseTransferShipmentLine = $warehouseTransferLine->shipmentLine;
        $activeMovement = $warehouseTransferShipmentLine->inventoryMovements()->whereInventoryStatus(InventoryMovement::INVENTORY_STATUS_ACTIVE)->sole();

        $this->assertDatabaseHas(WarehouseTransfer::class, [
            'id' => $warehouseTransfer->id,
            'transfer_status' => WarehouseTransfer::TRANSFER_STATUS_CLOSED,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $fromWarehouse->id,
            'quantity' => -5,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $toWarehouse->id,
            'quantity' => 5,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_IN_TRANSIT
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $toWarehouse->id,
            'quantity' => -5,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_IN_TRANSIT
        ]);

        $splitMovement = app(SplitInventoryMovement::class)($activeMovement, 2);
        $this->assertEquals(-2, $splitMovement->quantity);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $fromWarehouse->id,
            'quantity' => -2,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $toWarehouse->id,
            'quantity' => 2,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_IN_TRANSIT,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $toWarehouse->id,
            'quantity' => -2,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_IN_TRANSIT,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $fromWarehouse->id,
            'quantity' => -3,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $toWarehouse->id,
            'quantity' => 3,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_IN_TRANSIT,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $toWarehouse->id,
            'quantity' => -3,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_IN_TRANSIT,
        ]);
    }
}
