<?php

namespace Tests\Unit;

use App\Jobs\ImportSupplierInventory;
use App\Models\Product;
use App\Models\Supplier;
use App\Models\SupplierInventory;
use App\Models\Warehouse;
use Illuminate\Foundation\Testing\WithFaker;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Queue;
use Tests\TestCase;

class ImportSupplierInventoryTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    protected function setUp(): void
    {
        parent::setUp();
        Queue::fake();
    }

    public function test_it_can_import_fresh_records(): void
    {
        /** @var Supplier $supplier */
        $supplier = Supplier::factory()->create([
            'default_warehouse_id' => Warehouse::factory(),
        ]);
        $products = Product::factory(2)->create();
        $supplier->products()->sync($products->pluck('id'));
        $records = $products->map(function (Product $product) {
            $inStockOrQuantity = $this->faker->randomElement([0, 1]);

            return [
                'product_id' => $product->id,
                'in_stock' => $inStockOrQuantity ? 1 : null,
                'quantity' => ! $inStockOrQuantity ? $this->faker->numberBetween(5, 20) : null,
            ];
        })->toArray();

        $importer = new ImportSupplierInventory($records, ['supplier_id' => $supplier->id]);
        $importer->handle();

        $this->assertDatabaseCount('supplier_inventory', count($products));
    }

    public function test_it_can_update_supplier_inventory_based_on_ids(): void
    {
        /** @var Supplier $supplier */
        $supplier = Supplier::factory()->create([
            'default_warehouse_id' => Warehouse::factory(),
        ]);
        $products = Product::factory(50)->create();
        $supplier->products()->sync($products->pluck('id'));

        $records = $products->map(function (Product $product) {
            $inStockOrQuantity = $this->faker->randomElement([0, 1]);

            return [
                'product_id' => $product->id,
                'in_stock' => $inStockOrQuantity ? 1 : null,
                'quantity' => ! $inStockOrQuantity ? $this->faker->numberBetween(5, 20) : null,
            ];
        })->toArray();

        SupplierInventory::query()->insert(array_map(function ($row) use ($supplier) {
            $row['supplier_id'] = $supplier->id;
            $row['warehouse_id'] = $supplier->default_warehouse_id;

            return $row;
        }, $records));

        // Update some records
        $records = SupplierInventory::query()->take(2)->get()->toArray();
        $records[0]['quantity'] = 30;
        $records[1]['eta'] = now()->addDay();
        $importer = new ImportSupplierInventory($records, ['supplier_id' => $supplier->id]);
        $importer->handle();

        $this->assertDatabaseCount('supplier_inventory', count($products));

        $this->assertDatabaseHas('supplier_inventory', [
            'id' => $records[0]['id'],
            'quantity' => 30,
        ]);
        $this->assertDatabaseMissing('supplier_inventory', [
            'id' => $records[1]['id'],
            'eta' => null,
        ]);
    }

    public function test_it_can_update_supplier_inventory_based_on_sku(): void
    {
        /** @var Supplier $supplier */
        $supplier = Supplier::factory()->create([
            'default_warehouse_id' => Warehouse::factory(),
        ]);
        $products = Product::factory(50)->create();
        $supplier->products()->sync($products->pluck('id'));

        $records = $products->map(function (Product $product) {
            $inStockOrQuantity = $this->faker->randomElement([0, 1]);

            return [
                'product_id' => $product->id,
                'in_stock' => $inStockOrQuantity ? 1 : null,
                'quantity' => ! $inStockOrQuantity ? $this->faker->numberBetween(5, 20) : null,
            ];
        })->toArray();

        SupplierInventory::query()->insert(array_map(function ($row) use ($supplier) {
            $row['supplier_id'] = $supplier->id;
            $row['warehouse_id'] = $supplier->default_warehouse_id;

            return $row;
        }, $records));

        // Update some records
        $records = SupplierInventory::query()->take(2)->get()->toArray();
        $records[0]['quantity'] = 30;
        $records[0]['sku'] = $products[0]->sku;
        $records[1]['sku'] = $products[1]->sku;
        $records[1]['eta'] = now()->addDay();

        unset($records[0]['id'], $records[1]['id'], $records[1]['product_id']);

        $importer = new ImportSupplierInventory($records, ['supplier_id' => $supplier->id]);
        $importer->handle();

        $this->assertDatabaseCount('supplier_inventory', count($products));

        $this->assertDatabaseHas('supplier_inventory', [
            'product_id' => $products[0]['id'],
            'quantity' => 30,
        ]);
        $this->assertDatabaseMissing('supplier_inventory', [
            'product_id' => $products[1]['id'],
            'eta' => null,
        ]);
    }

    public function test_it_can_update_supplier_sku_when_available(): void
    {
        /** @var Supplier $supplier */
        $supplier = Supplier::factory()->create([
            'default_warehouse_id' => Warehouse::factory(),
        ]);
        $products = Product::factory(50)->create();
        $supplier->products()->sync($products->pluck('id'));

        $records = $products->map(function (Product $product) {
            $inStockOrQuantity = $this->faker->randomElement([0, 1]);

            return [
                'product_id' => $product->id,
                'in_stock' => $inStockOrQuantity ? 1 : null,
                'quantity' => ! $inStockOrQuantity ? $this->faker->numberBetween(5, 20) : null,
            ];
        })->toArray();

        SupplierInventory::query()->insert(array_map(function ($row) use ($supplier) {
            $row['supplier_id'] = $supplier->id;
            $row['warehouse_id'] = $supplier->default_warehouse_id;

            return $row;
        }, $records));

        // Update some records
        $supplierSku = $this->faker->word();
        $records = SupplierInventory::query()->take(1)->get()->toArray();
        $records[0]['quantity'] = 30;
        $records[0]['supplier_sku'] = $supplierSku;
        $importer = new ImportSupplierInventory($records, ['supplier_id' => $supplier->id]);
        $importer->handle();

        $this->assertDatabaseCount('supplier_inventory', count($products));

        $this->assertDatabaseHas('supplier_inventory', [
            'product_id' => $products[0]['id'],
            'quantity' => 30,
        ]);
        $this->assertDatabaseHas('supplier_products', [
            'product_id' => $products[0]['id'],
            'supplier_id' => $supplier->id,
            'supplier_sku' => $supplierSku,
        ]);

        // Should still update without product id
        unset($records[0]['product_id']);

        $supplierSku = 'another-supplier-sku';
        $records[0]['supplier_sku'] = $supplierSku;
        $importer = new ImportSupplierInventory($records, ['supplier_id' => $supplier->id]);
        $importer->handle();

        $this->assertDatabaseCount('supplier_inventory', count($products));

        $this->assertDatabaseHas('supplier_inventory', [
            'product_id' => $products[0]['id'],
            'quantity' => 30,
        ]);
        $this->assertDatabaseHas('supplier_products', [
            'product_id' => $products[0]['id'],
            'supplier_id' => $supplier->id,
            'supplier_sku' => $supplierSku,
        ]);
    }

    public function test_it_can_ignore_non_existing_skus_in_import(): void
    {
        /** @var Supplier $supplier */
        $supplier = Supplier::factory()->create([
            'default_warehouse_id' => Warehouse::factory(),
        ]);
        $products = Product::factory(50)->create();
        $supplier->products()->sync($products->pluck('id'));

        $records = $products->map(function (Product $product) {
            $inStockOrQuantity = $this->faker->randomElement([0, 1]);

            return [
                'product_id' => $product->id,
                'in_stock' => $inStockOrQuantity ? 1 : null,
                'quantity' => ! $inStockOrQuantity ? $this->faker->numberBetween(5, 20) : null,
            ];
        })->toArray();

        SupplierInventory::query()->insert(array_map(function ($row) use ($supplier) {
            $row['supplier_id'] = $supplier->id;
            $row['warehouse_id'] = $supplier->default_warehouse_id;

            return $row;
        }, $records));

        // Update some records
        $records = SupplierInventory::query()->take(2)->get()->toArray();
        $records[0]['quantity'] = 30;
        $records[0]['sku'] = $products[0]->sku;
        $records[1]['sku'] = $products[1]->sku;
        $records[1]['eta'] = now()->addDay();

        unset($records[0]['id'], $records[1]['id'], $records[1]['product_id']);

        // Add non-existing sku
        $records[] = [
            'sku' => $records[0]['sku'].'-new-sku',
            'quantity' => 2000,
        ];

        $importer = new ImportSupplierInventory($records, ['supplier_id' => $supplier->id]);
        $importer->handle();

        $this->assertDatabaseCount('supplier_inventory', count($products));

        $this->assertDatabaseHas('supplier_inventory', [
            'product_id' => $products[0]['id'],
            'quantity' => 30,
        ]);
        $this->assertDatabaseMissing('supplier_inventory', [
            'product_id' => $products[1]['id'],
            'eta' => null,
        ]);

        $this->assertDatabaseMissing('supplier_inventory', [
            'quantity' => 2000,
        ]);
    }
}
