<?php

namespace Tests\Unit;

use App\Data\WarehouseTransferReceiptData;
use App\Data\WarehouseTransferReceiptProductData;
use App\Managers\WarehouseTransferManager;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\Warehouse;
use App\Models\WarehouseTransfer;
use App\Repositories\InventoryMovementRepository;
use Carbon\Carbon;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class GetCorrespondingInTransitMovementsTest extends TestCase
{
    use FastRefreshDatabase;

    /**
     * @throws Throwable
     */
    public function test_it_can_get_corresponding_in_transit_movements()
    {
        $product = Product::factory()->create();
        $fromWarehouse = Warehouse::first();
        $toWarehouse = Warehouse::factory()->create([
            'type' => Warehouse::TYPE_DIRECT,
        ]);

        $wtManager = app(WarehouseTransferManager::class);

        $product->setInitialInventory($fromWarehouse->id, 5);

        $warehouseTransfer = $wtManager->initiateTransfer([
            'from_warehouse_id' => $fromWarehouse->id,
            'to_warehouse_id' => $toWarehouse->id,
            'products' => [
                [
                    'id' => $product->id,
                    'quantity' => 5,
                ],
            ],
        ]);
        $warehouseTransferLine = $warehouseTransfer->warehouseTransferLines->first();

        $wtManager->openWarehouseTransfer($warehouseTransfer, []);
        $warehouseTransferShipmentLine = $warehouseTransferLine->shipmentLine;

        $wtManager->receiveShipment($warehouseTransfer, WarehouseTransferReceiptData::from([
            'receipt_date' => Carbon::now(),
            'products' => WarehouseTransferReceiptProductData::collection([
                WarehouseTransferReceiptProductData::from([
                    'id' => $product->id,
                    'quantity' => 5,
                ])
            ])
        ]));

        $this->assertDatabaseHas(WarehouseTransfer::class, [
            'id' => $warehouseTransfer->id,
            'transfer_status' => WarehouseTransfer::TRANSFER_STATUS_CLOSED,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $fromWarehouse->id,
            'quantity' => -5,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $toWarehouse->id,
            'quantity' => 5,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_IN_TRANSIT
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $toWarehouse->id,
            'quantity' => -5,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_IN_TRANSIT
        ]);

        $activeMovement = $warehouseTransferShipmentLine->inventoryMovements()->whereInventoryStatus(InventoryMovement::INVENTORY_STATUS_ACTIVE)->sole();
        $addInTransitMovement = app(InventoryMovementRepository::class)->getCorrespondingAddInTransitMovementForActiveLineMovement($activeMovement);
        app(InventoryMovementRepository::class)->getCorrespondingDeductInTransitMovementForAddInTransitLineMovement($addInTransitMovement);

        // No errors thrown, so tests pass

        $this->assertTrue(true);
    }
}
