<?php

namespace Tests\Unit;

use App\Models\InventoryForecast;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\Setting;
use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Models\User;
use App\Models\Warehouse;
use App\Repositories\InventoryForecastRepository;
use App\Services\InventoryForecasting\ForecastManager;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\Exceptions\LargePurchaseOrderException;
use App\Services\PurchaseOrder\PurchaseOrderValidator;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class ForecastManagerTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    protected function setUp(): void
    {
        parent::setUp();
        Setting::with([])
            ->where('key', Setting::KEY_PO_LEAD_TIME)
            ->update(['value' => 0]);
    }


    /**
     * @throws LargePurchaseOrderException
     * @throws BindingResolutionException
     */
    public function test_it_can_forecast_when_purchases_are_needed(): void
    {
        Queue::fake();
        Sanctum::actingAs(User::factory()->create());

        /** @var Product $product */
        $product = Product::factory()->create();
        /** @var Supplier $supplier */
        $supplier = Supplier::factory()->create(['leadtime' => 0, 'minimum_order_quantity' => 1]);
        $supplier->products()->attach($product->id, ['leadtime' => 0]);

        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::factory()->create();

        // No stock available
        $product->productInventory()->create([
            'inventory_available' => 0,
            'warehouse_id' => $warehouse->id,
        ]);

        $this->postJson('/api/sales-orders', [
            'order_date' => now()->subMonths(2),
            'order_status' => SalesOrder::STATUS_OPEN,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'quantity' => 3,
                    'description' => $this->faker->sentence(),
                    'amount' => 10,
                    'product_id' => $product->id,
                    'warehouse_id' => $warehouse->id,
                ]
            ]
        ])->assertSuccessful();

        $supplierProducts = SupplierProduct::query()
            ->where('product_id', $product->id)
            ->where('supplier_id', $supplier->id)
            ->get();

        $manager = new ForecastManager(new InventoryForecastRepository);
        $manager->forecastForSupplier(
            supplier: $supplier,
            forSpecificProducts: true,
            productIds: $supplierProducts->pluck('product_id')->toArray()
        );

        // Forecast record should exist.
        $this->assertDatabaseCount('inventory_forecasts', 1);
        $this->assertDatabaseHas('inventory_forecasts', [
            'supplier_id' => $supplier->id,
            'leadtime' => 0,
            'minimum_order_quantity' => 1,
        ]);
        $this->assertDatabaseCount('inventory_forecast_items', 1);
        $this->assertDatabaseHas('inventory_forecast_items', [
            'suggested_purchase_quantity' => 6, # 3 to cover backorders and 3 to cover future sales
            'product_id'                  => $product->id,
        ]);

    }

    public function test_it_can_forecast_for_various_warehouses(): void{
        Queue::fake();
        Sanctum::actingAs(User::factory()->create());

        /** @var Product $product */
        $product = Product::factory()->create();
        /** @var Supplier $supplier */
        $supplier = Supplier::factory()->create(['leadtime' => 0, 'minimum_order_quantity' => 1]);
        $supplier->products()->attach($product->id, ['leadtime' => 0]);

        /** @var Warehouse $warehouse */
        $warehouse1 = Warehouse::factory()->create();
        $warehouse2 = Warehouse::factory()->create();

        // No stock available
        $product->productInventory()->create([
            'inventory_available' => 0,
            'warehouse_id' => $warehouse1->id,
        ]);

        $product->productInventory()->create([
            'inventory_available' => 0,
            'warehouse_id' => $warehouse2->id,
        ]);

        $this->postJson('/api/sales-orders', [
            'order_date' => now()->subMonths(2),
            'order_status' => SalesOrder::STATUS_OPEN,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'quantity' => 3,
                    'description' => $this->faker->sentence(),
                    'amount' => 10,
                    'product_id' => $product->id,
                    'warehouse_id' => $warehouse1->id,
                ],
                [
                    'quantity' => 10,
                    'description' => $this->faker->sentence(),
                    'amount' => 10,
                    'product_id' => $product->id,
                    'warehouse_id' => $warehouse2->id,
                ]
            ]
        ])->assertSuccessful();

        $supplierProducts = SupplierProduct::query()
            ->where('product_id', $product->id)
            ->where('supplier_id', $supplier->id)
            ->get();

        $manager = new ForecastManager(new InventoryForecastRepository);
        $manager->forecastForSupplier(
            supplier: $supplier,
            forSpecificProducts: true,
            productIds: $supplierProducts->pluck('product_id')->toArray()
        );

        // Forecast record should exist.
        $this->assertDatabaseCount('inventory_forecasts', 1);
        $this->assertDatabaseHas('inventory_forecasts', [
            'supplier_id' => $supplier->id,
            'leadtime' => 0,
            'minimum_order_quantity' => 1,
        ]);
        $this->assertDatabaseCount('inventory_forecast_items', 1);
        $this->assertDatabaseHas('inventory_forecast_items', [
            'suggested_purchase_quantity' => 26, # 13 to cover backorders and 13 to cover future sales
            'product_id'                  => $product->id,
        ]);
    }

    public function test_it_can_forecast_when_no_purchases_are_needed(): void
    {

        Queue::fake();
        Sanctum::actingAs(User::factory()->create());

        /** @var Product $product */
        $product = Product::factory()->create();
        /** @var Supplier $supplier */
        $supplier = Supplier::factory()->create(['leadtime' => 0, 'minimum_order_quantity' => 1]);
        $supplier->products()->attach($product->id, ['leadtime' => 0]);

        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::factory()->create();

        // Create sales
        $this->postJson('/api/sales-orders', [
            'order_date' => now()->subMonths(2),
            'order_status' => SalesOrder::STATUS_OPEN,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'quantity' => 3,
                    'description' => $this->faker->sentence,
                    'amount' => 10,
                    'product_id' => $product->id,
                    'warehouse_id' => $warehouse->id,
                ]
            ]
        ])->assertSuccessful();


        // Create purchase order
        $this->postJson('/api/purchase-orders', [
            'purchase_order_date' => now(),
            'approval_status' => PurchaseOrderValidator::APPROVAL_STATUS_APPROVED,
            'destination_warehouse_id' => $warehouse->id,
            'supplier_id' => $supplier->id,
            'currency_code' => 'USD',
            'purchase_order_lines' => [
                [
                    'quantity' => 6,
                    'description' => $this->faker->sentence,
                    'amount' => 10,
                    'product_id' => $product->id,
                ]
            ]
        ]);


        $supplierProducts = SupplierProduct::query()
            ->where('product_id', $product->id)
            ->where('supplier_id', $supplier->id)
            ->get();

        $manager = new ForecastManager(new InventoryForecastRepository);
        $manager->forecastForSupplier(
            supplier: $supplier,
            forSpecificProducts: true,
            productIds: $supplierProducts->pluck('product_id')->toArray()
        );

        // Nothing should be recorded
        $this->assertDatabaseCount('inventory_forecast_items', 0);

    }
}
