<?php

namespace Tests\Unit;

use App\Managers\InventoryHealthManager;
use App\Managers\InventoryMovementManager;
use App\Models\BackorderQueue;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\Warehouse;
use App\Repositories\InventoryHealthRepository;
use App\Services\InventoryManagement\BulkInventoryManager;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class FixMissingFulfillmentMovementsTest extends TestCase
{
    use FastRefreshDatabase;

    protected InventoryHealthRepository $health;
    protected InventoryHealthManager $healthManager;
    protected InventoryMovementManager $inventoryMovementManager;

    protected function setUp(): void
    {
        parent::setUp();

        $this->health = app(InventoryHealthRepository::class);
        $this->healthManager = app(InventoryHealthManager::class);
        $this->inventoryMovementManager = app(InventoryMovementManager::class);
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_identify_and_fix_fulfillments_without_movements()
    {
        $product = Product::factory()->create();
        $warehouse = Warehouse::first();

        $product->setInitialInventory($warehouse->id, 10);

        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines(1, [
                'product_id' => $product->id,
                'warehouse_id' => $warehouse->id,
                'quantity' => 1,
            ])
            ->create([
                'order_status' => SalesOrder::STATUS_CLOSED,
                'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_FULFILLED,
            ]);
        $salesOrderLine = $salesOrder->salesOrderLines->first();
        (new BulkInventoryManager())->bulkAllocateNegativeInventoryEvents(collect([$salesOrderLine]));

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => -1,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => 1,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED
        ]);

        app(FulfillmentManager::class)->fulfill($salesOrder, [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'fulfilled_at' => now(),
            'warehouse_id' => $warehouse->id,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrder->salesOrderLines->first()->id,
                    'quantity' => 1,
                ]
            ]
        ], false, false);

        $salesOrderFulfillmentLine = SalesOrderFulfillmentLine::first();

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => -1,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'link_type' => SalesOrderFulfillmentLine::class,
            'link_id' => $salesOrderFulfillmentLine->id,
        ]);

        $fulfillmentMovement = InventoryMovement::where('link_type', SalesOrderFulfillmentLine::class)
            ->where('link_id', $salesOrderFulfillmentLine->id)
            ->first();

        $this->assertEquals(0, $this->health->salesOrderLinesMissingFulfillmentMovementsQuery()->count());

        $fulfillmentMovement->delete();

        $this->assertEquals(1, $this->health->salesOrderLinesMissingFulfillmentMovementsQuery()->count());

        $this->healthManager->createFulfillmentInventoryMovementsForSalesOrderLine($salesOrderLine);

        $this->assertEquals(0, $this->health->salesOrderLinesMissingFulfillmentMovementsQuery()->count());

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => -1,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'link_type' => SalesOrderFulfillmentLine::class,
            'link_id' => $salesOrderFulfillmentLine->id,
        ]);
    }
}
