<?php

namespace Tests\Unit;

use App\Actions\InventoryHealth\FifoLayerInventoryMovementRelocator;
use App\Data\CreateInventoryAdjustmentData;
use App\Data\WarehouseTransferReceiptData;
use App\Data\WarehouseTransferReceiptProductData;
use App\Exceptions\InsufficientStockException;
use App\Exceptions\WarehouseTransfers\NotOpenWarehouseTransferException;
use App\Exceptions\WarehouseTransfers\WarehouseTransferHasNoProductsException;
use App\Exceptions\WarehouseTransfers\WarehouseTransferOpenException;
use App\Http\Requests\StoreInventoryAdjustment;
use App\Managers\InventoryAdjustmentManager;
use App\Managers\WarehouseTransferManager;
use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\Warehouse;
use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferShipment;
use App\Models\WarehouseTransferShipmentLine;
use Carbon\Carbon;
use Illuminate\Contracts\Container\BindingResolutionException;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class FifoLayerInventoryMovementRelocatorTest extends TestCase
{
    use FastRefreshDatabase;

    /**
     * @throws Throwable
     * @throws BindingResolutionException
     * @throws InsufficientStockException
     * @throws WarehouseTransferHasNoProductsException
     * @throws NotOpenWarehouseTransferException
     * @throws WarehouseTransferOpenException
     */
    public function test_it_can_move_warehouse_transfer_movements()
    {
        $product = Product::factory()->create();
        $fromWarehouse = Warehouse::first();
        $toWarehouse = Warehouse::factory()->create([
            'type' => Warehouse::TYPE_DIRECT,
        ]);

        $wtManager = app(WarehouseTransferManager::class);

        $product->setInitialInventory($fromWarehouse->id, 4);
        $fifoLayer = FifoLayer::first();

        $warehouseTransfer = $wtManager->initiateTransfer([
            'from_warehouse_id' => $fromWarehouse->id,
            'to_warehouse_id' => $toWarehouse->id,
            'products' => [
                [
                    'id' => $product->id,
                    'quantity' => 4,
                ],
            ],
        ]);
        $wtManager->openWarehouseTransfer($warehouseTransfer, []);
        $wtManager->receiveShipment($warehouseTransfer, WarehouseTransferReceiptData::from([
            'receipt_date' => Carbon::now(),
            'products' => WarehouseTransferReceiptProductData::collection([
                WarehouseTransferReceiptProductData::from([
                    'id' => $product->id,
                    'quantity' => 2,
                ])
            ])
        ]));
        $wtManager->receiveShipment($warehouseTransfer, WarehouseTransferReceiptData::from([
            'receipt_date' => Carbon::now(),
            'products' => WarehouseTransferReceiptProductData::collection([
                WarehouseTransferReceiptProductData::from([
                    'id' => $product->id,
                    'quantity' => 2,
                ])
            ])
        ]));

        $this->assertDatabaseHas(WarehouseTransfer::class, [
            'id' => $warehouseTransfer->id,
            'transfer_status' => WarehouseTransfer::TRANSFER_STATUS_CLOSED,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $fromWarehouse->id,
            'quantity' => -4,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $toWarehouse->id,
            'quantity' => 4,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_IN_TRANSIT
        ]);

        // There should be two of these
        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $toWarehouse->id,
            'quantity' => -2,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_IN_TRANSIT
        ]);

        $this->assertCount(2, InventoryMovement::query()
            ->where('product_id', $product->id)
            ->where('warehouse_id', $toWarehouse->id)
            ->where('quantity', -2)
            ->where('type', InventoryMovement::TYPE_TRANSFER)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_IN_TRANSIT)
            ->get()
        );

        app(InventoryAdjustmentManager::class)->createAdjustment(CreateInventoryAdjustmentData::from([
            'product_id' => $product->id,
            'quantity' => 10,
            'unit_cost' => 10,
            'adjustment_date' => now(),
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'warehouse_id' => $fromWarehouse->id,
        ]));
        $newFifoLayer = FifoLayer::whereLinkType(InventoryAdjustment::class)->sole();

        $allocationData = app(FifoLayerInventoryMovementRelocator::class)(4, $fifoLayer);
        $this->assertEquals(0, $allocationData->remainingQuantityToAllocate);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $fromWarehouse->id,
            'quantity' => -4,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'layer_id' => $newFifoLayer->id,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $toWarehouse->id,
            'quantity' => 4,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_IN_TRANSIT,
            'layer_id' => $newFifoLayer->id,
        ]);

        // There should be two of these
        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $toWarehouse->id,
            'quantity' => -2,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_IN_TRANSIT,
            'layer_id' => $newFifoLayer->id,
        ]);

        $this->assertCount(2, InventoryMovement::query()
            ->where('product_id', $product->id)
            ->where('warehouse_id', $toWarehouse->id)
            ->where('quantity', -2)
            ->where('type', InventoryMovement::TYPE_TRANSFER)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_IN_TRANSIT)
            ->where('layer_id', $newFifoLayer->id)
            ->get()
        );
    }
}
