<?php

namespace Tests\Unit;

use App\Data\CreateInventoryAdjustmentData;
use App\Exceptions\CantBackorderAlreadyFulfilledQuantityException;
use App\Exceptions\SalesOrder\SalesOrderFulfillmentDispatchException;
use App\Exceptions\SupplierWarehouseCantHaveInventoryMovementsException;
use App\Exceptions\UnableToAllocateToFifoLayersException;
use App\Managers\InventoryAdjustmentManager;
use App\Managers\InventoryHealthManager;
use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLineLayer;
use App\Models\Supplier;
use App\Models\Warehouse;
use App\Repositories\InventoryHealthRepository;
use App\Services\InventoryManagement\BulkInventoryManager;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use Illuminate\Contracts\Container\BindingResolutionException;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class CreateSalesOrderReservationTest extends TestCase
{
    use FastRefreshDatabase;

    protected InventoryHealthRepository $health;
    protected InventoryHealthManager $healthManager;

    protected function setUp(): void
    {
        parent::setUp();

        $this->health = app(InventoryHealthRepository::class);
        $this->healthManager = app(InventoryHealthManager::class);
    }

    /**
     * @throws Throwable
     * @throws SalesOrderFulfillmentDispatchException
     * @throws BindingResolutionException
     */
    public function test_it_can_create_sales_order_reservation_standard()
    {
        $product = Product::factory()->create();
        $warehouse = Warehouse::first();

        $product->setInitialInventory($warehouse->id, 5);
        $fifoLayer = FifoLayer::first();

        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines(1, [
                'product_id' => $product->id,
                'warehouse_id' => $warehouse->id,
                'quantity' => 5,
            ])
            ->create([
                'order_status' => SalesOrder::STATUS_CLOSED,
                'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_FULFILLED,
            ]);
        $salesOrderLine = $salesOrder->salesOrderLines->first();

        $this->healthManager->createSalesOrderReservation($salesOrderLine);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => -5,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 5,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
        ]);

        $this->assertDatabaseHas(SalesOrderLineLayer::class, [
            'sales_order_line_id' => $salesOrderLine->id,
            'layer_id' => $fifoLayer->id,
            'layer_type' => FifoLayer::class,
            'quantity' => 5,
        ]);

        $this->healthManager->deleteSalesOrderReservation($salesOrderLine);

        $this->assertDatabaseMissing(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => -5,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);

        $this->assertDatabaseMissing(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 5,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
        ]);

        $this->assertDatabaseMissing(SalesOrderLineLayer::class, [
            'sales_order_line_id' => $salesOrderLine->id,
            'layer_id' => $fifoLayer->id,
            'layer_type' => FifoLayer::class,
            'quantity' => 5,
        ]);
    }

    /**
     * @throws Throwable
     * @throws UnableToAllocateToFifoLayersException
     */
    public function test_it_can_create_sales_order_reservation_with_partially_externally_fulfilled_line()
    {
        $product = Product::factory()->create();
        $warehouse = Warehouse::first();

        $product->setInitialInventory($warehouse->id, 5);
        $fifoLayer = FifoLayer::first();

        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines(1, [
                'product_id' => $product->id,
                'warehouse_id' => $warehouse->id,
                'quantity' => 5,
                'externally_fulfilled_quantity' => 2,
            ])
            ->create([
                'order_status' => SalesOrder::STATUS_CLOSED,
                'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_FULFILLED,
            ]);
        $salesOrderLine = $salesOrder->salesOrderLines->first();

        $this->healthManager->createSalesOrderReservation($salesOrderLine);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => -3,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 3,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
        ]);

        $this->assertDatabaseHas(SalesOrderLineLayer::class, [
            'sales_order_line_id' => $salesOrderLine->id,
            'layer_id' => $fifoLayer->id,
            'layer_type' => FifoLayer::class,
            'quantity' => 3,
        ]);

        $this->healthManager->deleteSalesOrderReservation($salesOrderLine);

        $this->assertDatabaseMissing(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => -3,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);

        $this->assertDatabaseMissing(InventoryMovement::class, [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 3,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
        ]);

        $this->assertDatabaseMissing(SalesOrderLineLayer::class, [
            'sales_order_line_id' => $salesOrderLine->id,
            'layer_id' => $fifoLayer->id,
            'layer_type' => FifoLayer::class,
            'quantity' => 3,
        ]);
    }

    /**
     * @throws Throwable
     * @throws SalesOrderFulfillmentDispatchException
     * @throws BindingResolutionException
     */
    public function test_it_wont_backorder_already_fulfilled_lines()
    {
        $product = Product::factory()->create();
        $warehouse = Warehouse::first();

        $product->setInitialInventory($warehouse->id, 5);
        $fifoLayer = FifoLayer::first();

        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines(1, [
                'product_id' => $product->id,
                'warehouse_id' => $warehouse->id,
                'quantity' => 5,
            ])
            ->create([
                'order_status' => SalesOrder::STATUS_CLOSED,
                'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_FULFILLED,
            ]);
        $salesOrderLine = $salesOrder->salesOrderLines->first();
        (new BulkInventoryManager())->bulkAllocateNegativeInventoryEvents(collect([$salesOrderLine]));

        app(FulfillmentManager::class)->fulfill($salesOrder, [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'fulfilled_at' => now(),
            'warehouse_id' => $warehouse->id,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrder->salesOrderLines->first()->id,
                    'quantity' => 2,
                ]
            ]
        ], false, false);

        $salesOrderFulfillmentLine = SalesOrderFulfillmentLine::first();

        /*
        |--------------------------------------------------------------------------
        | Remove inventory movements, then use up the remaining inventory, then try to reserve again
        |--------------------------------------------------------------------------
        */

        $salesOrderLine->inventoryMovements()->delete();
        $salesOrderFulfillmentLine->inventoryMovements()->delete();
        $fifoLayer->refresh();
        $fifoLayer->fulfilled_quantity = 0;
        $fifoLayer->save();

        // Create non-movable usage on fifo layer

        $this->assertDatabaseHas(FifoLayer::class, [
            'id' => $fifoLayer->id,
            'available_quantity' => 5,
        ]);

        $inventoryAdjustment = app(InventoryAdjustmentManager::class)->createAdjustment(CreateInventoryAdjustmentData::from([
            'adjustment_type' => InventoryAdjustment::TYPE_DECREASE,
            'adjustment_date' => now(),
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 5,
        ]));

        $this->assertDatabaseHas(FifoLayer::class, [
            'id' => $fifoLayer->id,
            'available_quantity' => 0,
        ]);

        $this->assertThrows(function () use ($salesOrderLine) {
            $this->healthManager->createSalesOrderReservation($salesOrderLine);
        }, CantBackorderAlreadyFulfilledQuantityException::class);

        $inventoryAdjustment->delete();

        $this->healthManager->createSalesOrderReservation($salesOrderLine);
    }

    public function test_it_cant_reserve_for_supplier_warehouses()
    {
        $product = Product::factory()->create();
        $supplier = Supplier::first();
        $warehouse = $supplier->warehouses()->first();

        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines(1, [
                'product_id' => $product->id,
                'warehouse_id' => $warehouse->id,
                'quantity' => 5,
            ])
            ->create([
                'order_status' => SalesOrder::STATUS_CLOSED,
                'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_FULFILLED,
            ]);
        $salesOrderLine = $salesOrder->salesOrderLines->first();

        $this->assertEquals(0, $this->health->salesOrderLinesMissingMovementsQuery(InventoryMovement::INVENTORY_STATUS_ACTIVE)->count());

        $this->assertThrows(function () use ($salesOrderLine) {
            $this->healthManager->createSalesOrderReservation($salesOrderLine);
        }, SupplierWarehouseCantHaveInventoryMovementsException::class);
    }
}
