<?php

namespace Tests\Unit;

use App\Managers\InventoryHealthManager;
use App\Models\BackorderQueue;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineLayer;
use App\Models\Warehouse;
use App\Repositories\InventoryHealthRepository;
use App\Services\InventoryManagement\BulkInventoryManager;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use Illuminate\Contracts\Container\BindingResolutionException;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class BackorderFromFifoLayerTest extends TestCase
{
    use FastRefreshDatabase;

    protected InventoryHealthRepository $health;
    protected InventoryHealthManager $healthManager;

    protected function setUp(): void
    {
        parent::setUp();

        $this->health = app(InventoryHealthRepository::class);
        $this->healthManager = app(InventoryHealthManager::class);
    }

    /**
     * @throws Throwable
     * @throws BindingResolutionException
     */
    public function test_it_can_backorder_from_fifo_layer()
    {
        $product = Product::factory()->create();
        $warehouse = Warehouse::first();

        $product->setInitialInventory($warehouse->id, 5);
        $fifoLayer = FifoLayer::first();

        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines(1, [
                'product_id' => $product->id,
                'warehouse_id' => $warehouse->id,
                'quantity' => 5,
            ])
            ->create([
                'order_status' => SalesOrder::STATUS_OPEN,
            ]);
        $salesOrderLine = $salesOrder->salesOrderLines->first();
        (new BulkInventoryManager())->bulkAllocateNegativeInventoryEvents(collect([$salesOrderLine]));
        $fifoLayer->refresh();

        $this->assertDatabaseHas(InventoryMovement::class, [
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_type' => FifoLayer::class,
        ]);

        $this->assertDatabaseHas(SalesOrderLineLayer::class, [
            'sales_order_line_id' => $salesOrderLine->id,
            'quantity' => 5,
        ]);

        $this->assertDatabaseEmpty(BackorderQueue::class);

        $qtyBackordered = $this->healthManager->backorderFromFifoLayer($fifoLayer, 3 );
        $this->assertEquals(3, $qtyBackordered);

        $this->assertDatabaseHas(BackorderQueue::class, [
            'sales_order_line_id' => $salesOrderLine->id,
            'backordered_quantity' => 3,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'quantity' => -3,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_type' => BackorderQueue::class,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'quantity' => 3,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_type' => BackorderQueue::class,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'quantity' => -2,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_type' => FifoLayer::class,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'quantity' => 2,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_type' => FifoLayer::class,
        ]);

        $this->assertDatabaseHas(SalesOrderLineLayer::class, [
            'sales_order_line_id' => $salesOrderLine->id,
            'quantity' => 2,
        ]);

        app(FulfillmentManager::class)->fulfill($salesOrder, [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'fulfilled_at' => now(),
            'warehouse_id' => $warehouse->id,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrderLine->id,
                    'quantity' => 2,
                ]
            ]
        ], false, false);

        // Can't backorder fulfilled lines
        $qtyBackordered = $this->healthManager->backorderFromFifoLayer($fifoLayer, 2 );
        $this->assertEquals(0, $qtyBackordered);
    }
}
