<?php

namespace Tests\Feature;

use App\Abstractions\AbstractRepository;
use App\Models\Address;
use App\Models\Customer;
use App\Models\FinancialLine;
use App\Models\Payment;
use App\Models\Product;
use App\Models\ProductAttribute;
use App\Models\ProductBrand;
use App\Models\ProductImage;
use App\Models\ProductListing;
use App\Models\ProductPricing;
use App\Models\ReportingDailyFinancial;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineFinancial;
use App\Models\Supplier;
use App\Models\SupplierInventory;
use App\Models\SupplierProduct;
use App\Models\Warehouse;
use Modules\Amazon\Entities\AmazonReport;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Tests\UpsertTestingData;

/**
 * Must cover all inheritors of AbstractRepository
 */
class UpsertTest extends TestCase
{
    use FastRefreshDatabase;

    /**
     * Test upsert functionality for different repositories.
     *
     * The ordering of the tests is important due to the dependencies between the models and the counting of database records.
     */
    public function testUpsertFunctionality()
    {
        // Expecting 4 due to 2 records being seeded
        $this->runUpsertTest(Warehouse::class, UpsertTestingData::warehouseData(), 4);
        // Expecting 3 due to 1 record being seeded
        $this->runUpsertTest(SalesOrder::class, UpsertTestingData::salesOrderData());
        $this->runUpsertTest(Address::class, UpsertTestingData::addressData(), 3);
        $this->runUpsertTest(Customer::class, UpsertTestingData::customerData());
        $this->runUpsertTest(FinancialLine::class, UpsertTestingData::financialLineData());
        $this->runUpsertTest(Payment::class, UpsertTestingData::paymentData());
        $this->runUpsertTest(Product::class, UpsertTestingData::productData());
        $this->runUpsertTest(ProductBrand::class, UpsertTestingData::productBrandData());
        $this->runUpsertTest(ProductImage::class, UpsertTestingData::productImageData());
        $this->runUpsertTest(ProductAttribute::class, UpsertTestingData::productAttributeData());
        $this->runUpsertTest(ProductListing::class, UpsertTestingData::productListingData());
        $this->runUpsertTest(ProductPricing::class, UpsertTestingData::productPricingData());
        $this->runUpsertTest(SalesOrderLine::class, UpsertTestingData::salesOrderLineData());
        $this->runUpsertTest(SalesOrderLineFinancial::class, UpsertTestingData::salesOrderLineFinancialData());
        $this->runUpsertTest(ReportingDailyFinancial::class, UpsertTestingData::reportingDailyFinancialData());
        $this->runUpsertTest(SupplierInventory::class, UpsertTestingData::supplierInventoryData());
        $this->runUpsertTest(SupplierProduct::class, UpsertTestingData::supplierProductData());
        // Expecting 3 due to 1 record being seeded
        $this->runUpsertTest(Supplier::class, UpsertTestingData::supplierData(), 3);
        $this->runUpsertTest(AmazonReport::class, UpsertTestingData::amazonReportData());
    }

    /**
     * Run upsert test for a given repository and model.
     */
    private function runUpsertTest(string $modelClass, array $data, int $expectedCount = 2): void
    {
        $collection = collect($data);
        app(AbstractRepository::class)->save($collection->only(0), $modelClass);
        $this->assertDatabaseCount((new $modelClass())->getTable(), $expectedCount - 1);
        app(AbstractRepository::class)->save($collection, $modelClass);
        $this->assertDatabaseCount((new $modelClass())->getTable(), $expectedCount);
    }
}
