<?php

namespace Feature;

use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\PurchaseInvoice;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderShipment;
use App\Models\PurchaseOrderShipmentReceipt;
use App\Models\SalesCredit;
use App\Models\SalesCreditReturn;
use App\Models\SalesCreditReturnLine;
use App\Models\SalesOrder;
use App\Models\StockTake;
use App\Models\StockTakeItem;
use DB;
use Illuminate\Database\Eloquent\Model;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

/**
 * @group manual
 */
class UpdateTimestampsTest extends TestCase
{
    use FastRefreshDatabase;

   public function test_it_can_update_timestamps()
   {
       // Purchase order line touches purchase order

       $purchaseOrder = PurchaseOrder::factory()->hasPurchaseOrderLines()->create();
       $this->resetUpdatedAt($purchaseOrder);

       $purchaseOrderLine = $purchaseOrder->purchaseOrderLines->first();
       $this->assertEquals(1, $purchaseOrderLine->purchaseOrder->count());

       $purchaseOrderLine->update(['quantity' => ++$purchaseOrderLine->quantity]);
       $this->assertGreaterThanOrEqual($purchaseOrderLine->refresh()->updated_at, $purchaseOrder->refresh()->updated_at);

       // Purchase order line touches purchase order shipment receipt

       $purchaseOrderShipment = PurchaseOrderShipment::factory()
           ->recycle(
               $purchaseOrderLine,
               $purchaseOrder
           )
           ->create();

       $purchaseOrderShipmentReceipt = PurchaseOrderShipmentReceipt::factory()
          ->recycle([
              $purchaseOrderShipment,
              $purchaseOrderLine,
          ])
          ->hasPurchaseOrderShipmentReceiptLines()
           ->create();
       $this->resetUpdatedAt($purchaseOrderShipmentReceipt);

       $this->assertEquals(1, $purchaseOrderLine->purchaseOrderShipmentReceipts()->count());

       $purchaseOrderShipmentReceiptLine = $purchaseOrderShipmentReceipt->purchaseOrderShipmentReceiptLines->first();

       $purchaseOrderLine->update(['quantity' => ++$purchaseOrderLine->quantity]);

       $this->assertGreaterThanOrEqual($purchaseOrderLine->refresh()->updated_at, $purchaseOrderShipmentReceipt->refresh()->updated_at);

       // Purchase order shipment receipt line touches purchase order shipment receipt

       $this->resetUpdatedAt($purchaseOrderShipmentReceipt);
       $purchaseOrderShipmentReceiptLine->update(['quantity' => ++$purchaseOrderShipmentReceiptLine->quantity]);

       $purchaseOrder->refresh();

       $this->assertGreaterThanOrEqual($purchaseOrderShipmentReceiptLine->refresh()->updated_at, $purchaseOrderShipmentReceipt->refresh()->updated_at);
       
       // Purchase order touches purchase order shipment receipt

       $this->resetUpdatedAt($purchaseOrderShipmentReceipt);
       $purchaseOrder->update(['other_date' => now()]);

       $this->assertEquals(1, $purchaseOrder->purchaseOrderShipmentReceipts->count());
       $this->assertGreaterThanOrEqual($purchaseOrder->refresh()->updated_at, $purchaseOrderShipmentReceipt->refresh()->updated_at);

       // Purchase order line touches purchase invoice

       $purchaseInvoice = PurchaseInvoice::factory()
           ->recycle($purchaseOrderLine)
           ->hasPurchaseInvoiceLines()
           ->create();
       $this->assertEquals(1, $purchaseOrderLine->refresh()->purchaseInvoices->count());

       $purchaseInvoiceLine = $purchaseInvoice->purchaseInvoiceLines->first();

       $this->resetUpdatedAt($purchaseInvoice);
       $purchaseOrderLine->update(['quantity' => ++$purchaseOrderLine->quantity]);

       $this->assertGreaterThanOrEqual($purchaseOrderLine->refresh()->updated_at, $purchaseInvoice->refresh()->updated_at);

       // Purchase invoice line touches purchase invoice

       $this->resetUpdatedAt($purchaseInvoice);
       $purchaseInvoiceLine->update(['quantity_invoiced' => ++$purchaseInvoiceLine->quantity_invoiced]);

       $this->assertGreaterThanOrEqual($purchaseInvoiceLine->refresh()->updated_at, $purchaseInvoice->refresh()->updated_at);

       // Inventory movement touches inventory adjustment

       $inventoryAdjustment = InventoryAdjustment::factory()->create();

       $fifoLayer = FifoLayer::factory()
           ->create([
               'link_id' => $inventoryAdjustment->id,
               'link_type' => InventoryAdjustment::class,
           ]);

       $inventoryMovement = InventoryMovement::factory()
           ->create([
                'link_id' => $inventoryAdjustment->id,
                'link_type' => InventoryAdjustment::class,
                'layer_id' => $fifoLayer->id,
                'layer_type' => FifoLayer::class,
           ]);

       $this->assertEquals(1, $inventoryAdjustment->inventoryMovements->count());

       $this->resetUpdatedAt($inventoryAdjustment);
       $inventoryMovement->update(['quantity' => ++$inventoryMovement->quantity]);

       $this->assertGreaterThanOrEqual($inventoryMovement->refresh()->updated_at, $inventoryAdjustment->refresh()->updated_at);

       // Fifo layer touches inventory adjustment

       $this->assertEquals(1, $inventoryAdjustment->fifoLayers->count());

       $this->resetUpdatedAt($inventoryAdjustment);
       $fifoLayer->update(['total_cost' => ++$fifoLayer->total_cost]);

       $this->assertGreaterThanOrEqual($fifoLayer->refresh()->updated_at, $inventoryAdjustment->refresh()->updated_at);

       // Sales order touches sales credit return

       $salesOrder = SalesOrder::factory()
           ->hasSalesOrderLines()
           ->create();
       $salesOrderLine = $salesOrder->salesOrderLines->first();

       $salesCredit = SalesCredit::factory()
           ->hasSalesCreditLines()
           ->recycle([
               $salesOrder,
               $salesOrderLine
           ])
           ->create();
       $salesCreditLine = $salesCredit->salesCreditLines->first();

       $salesCreditReturn = SalesCreditReturn::factory()
           ->hasSalesCreditReturnLines()
           ->recycle([
               $salesCredit,
               $salesCreditLine
           ])
           ->create();
       $salesCreditReturnLine = $salesCreditReturn->salesCreditReturnLines->first();

        $this->assertEquals(1, $salesCredit->refresh()->salesCreditReturns->count());
        $this->assertEquals(1, $salesOrder->refresh()->salesCreditReturns->count());

       $this->resetUpdatedAt($salesCreditReturn);
       $salesOrder->update(['ship_by_date' => now()]);

       $this->assertGreaterThanOrEqual($salesOrder->refresh()->updated_at, $salesCreditReturn->refresh()->updated_at);

       // Inventory movement for sales credit return line touches sales credit return

       $fifoLayer = FifoLayer::factory()
           ->create([
               'link_id' => $salesCreditReturnLine->id,
               'link_type' => SalesCreditReturnLine::class,
              ]);

       $inventoryMovement = InventoryMovement::factory()
              ->create([
                'link_id' => $salesCreditReturnLine->id,
                'link_type' => SalesCreditReturnLine::class,
                'layer_id' => $fifoLayer->id,
                'layer_type' => FifoLayer::class,
                ]);

       $this->assertEquals(1, $salesCreditReturnLine->inventoryMovements->count());

       $this->resetUpdatedAt($salesCreditReturn);
       $inventoryMovement->update(['quantity' => ++$inventoryMovement->quantity]);

       $this->assertGreaterThanOrEqual($inventoryMovement->refresh()->updated_at, $salesCreditReturn->refresh()->updated_at);

       // Fifo layer for sales credit return line touches sales credit return

       $this->assertEquals(1, $salesCreditReturnLine->fifoLayers->count());

        $this->resetUpdatedAt($salesCreditReturn);
        $fifoLayer->update(['total_cost' => ++$fifoLayer->total_cost]);

        $this->assertGreaterThanOrEqual($fifoLayer->refresh()->updated_at, $salesCreditReturn->refresh()->updated_at);

       // Stock take item touches stock take

       $stockTake = StockTake::factory()
           ->hasStockTakeItems()
           ->create();
       $stockTakeItem = $stockTake->stockTakeItems->first();

       $this->resetUpdatedAt($stockTake);
       $stockTakeItem->update(['qty_counted' => ++$stockTakeItem->qty_counted]);

       $this->assertGreaterThanOrEqual($stockTakeItem->refresh()->updated_at, $stockTake->refresh()->updated_at);

       // Inventory movement for stock take item touches stock take item

       $fifoLayer = FifoLayer::factory()
              ->create([
                'link_id' => $stockTakeItem->id,
                'link_type' => StockTakeItem::class,
                ]);

        $inventoryMovement = InventoryMovement::factory()
            ->create([
                'link_id' => $stockTakeItem->id,
                'link_type' => StockTakeItem::class,
                'layer_id' => $fifoLayer->id,
                'layer_type' => FifoLayer::class,
                ]);

        $this->assertEquals(1, $stockTakeItem->inventoryMovements->count());

        $this->resetUpdatedAt($stockTake);
        $inventoryMovement->update(['quantity' => ++$inventoryMovement->quantity]);

        $this->assertGreaterThanOrEqual($inventoryMovement->refresh()->updated_at, $stockTake->refresh()->updated_at);

       // Fifo layer for stock take item touches stock take item

         $this->assertEquals(1, $stockTakeItem->fifoLayers->count());

        $this->resetUpdatedAt($stockTake);
        $fifoLayer->update(['total_cost' => ++$fifoLayer->total_cost]);

        $this->assertGreaterThanOrEqual($fifoLayer->refresh()->updated_at, $stockTake->refresh()->updated_at);
   }

   private function resetUpdatedAt(Model $model)
   {
         $model->updated_at = now()->subYear();
         $model->save();
         $model->refresh();
   }
}
