<?php

namespace Tests\Feature;

use App\Models\ShippingProviderWarehouseMapping;
use App\Models\ShipStation\ShipstationWarehouse;
use App\Models\Warehouse;
use Illuminate\Testing\Fluent\AssertableJson;
use Tests\Support\ShipstationTest;


class ShipstationWarehouseControllerTest extends ShipstationTest
{

    public function test_it_can_download_warehouses(){

        $this->mockEndpoint(
            endpoint: '/warehouses',
            response: [
                ShipstationWarehouse::factory()->make()->json_object
            ]
        );

        $this->postJson('/api/shipstation/warehouses/download')->assertSuccessful();
        $this->assertDatabaseCount('shipstation_warehouses', 1);
    }

    public function test_it_can_map_sku_warehouse_to_shipstation_warehouse(){

        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::factory()->create();
        /** @var ShipstationWarehouse $shipStationWarehouse */
        $shipStationWarehouse = ShipstationWarehouse::factory()->create();

        $this->postJson('/api/shipstation/warehouses/map', [
            'mapping' => [
                [
                    'warehouse_id' => $warehouse->id,
                    'shipstation_warehouse_id' => $shipStationWarehouse->id,
                ]
            ]
        ])->assertSuccessful();

        $this->assertDatabaseHas('shipping_provider_warehouse_mappings', [
            'warehouse_id' => $warehouse->id,
            'provider_id' => $shipStationWarehouse->id,
            'provider_type' => $shipStationWarehouse::class
        ]);
    }

    public function test_it_can_get_shipstation_warehouse_mappings(){

        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::factory()->create();
        /** @var ShipstationWarehouse $shipstationWarehouse */
        $shipstationWarehouse = ShipstationWarehouse::factory()->create();
        /** @var ShippingProviderWarehouseMapping $mapping */
        ShippingProviderWarehouseMapping::query()->create([
            'warehouse_id' => $warehouse->id,
            'provider_id' => $shipstationWarehouse->id,
            'provider_type' => $shipstationWarehouse::class,
        ]);

        $response = $this->getJson('/api/shipstation/warehouses/mappings')
            ->assertSuccessful();

        $response->assertJson(fn(AssertableJson $json) =>
            $json->has('data.0.warehouse.id')
            ->has('data.0.shipstation_warehouse.id')
            ->etc()
        )->assertJsonCount(1, 'data');

    }

    public function test_it_can_unmap_shipstation_warehouse(){

        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::factory()->create();
        /** @var ShipstationWarehouse $shipstationWarehouse */
        $shipstationWarehouse = ShipstationWarehouse::factory()->create();
        /** @var ShippingProviderWarehouseMapping $mapping */
        $mapping = ShippingProviderWarehouseMapping::query()->create([
            'warehouse_id' => $warehouse->id,
            'provider_id' => $shipstationWarehouse->id,
            'provider_type' => $shipstationWarehouse::class,
        ]);

        $this->postJson('/api/shipstation/warehouses/unmap', [
            'ids' => [$mapping->id]
        ])->assertSuccessful();

        $this->assertDatabaseEmpty('shipping_provider_warehouse_mappings');

    }

    public function test_it_can_get_downloaded_shipstation_warehouses(){

        ShipstationWarehouse::factory(10)->create();

        $response = $this->getJson('/api/shipstation/warehouses')
            ->assertSuccessful();

        $response->assertJson(fn(AssertableJson $json) =>
        $json->has('data.0.shipstation_id')
            ->has('data.0.name')
            ->has('data.0.json_object')
            ->etc()
        )->assertJsonCount(10, 'data');

    }

}
