<?php

namespace Tests\Feature;

use App\Http\Requests\StoreInventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\ProductInventory;
use App\Models\SalesOrder;
use App\Models\Setting;
use App\Models\User;
use App\Models\Warehouse;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class SalesOrdersWarehouseGroupingTest extends TestCase
{
    use FastRefreshDatabase;

    public function test_it_can_assign_all_items_in_an_order_based_on_stock_and_warehouse_priority(): void
    {
        Queue::fake();

        Sanctum::actingAs(User::factory()->create());

        Setting::query()->updateOrCreate([
            'key' => Setting::KEY_NEVER_SPLIT_SALES_ORDERS_ACROSS_WAREHOUSES,
        ], [
            'key' => Setting::KEY_NEVER_SPLIT_SALES_ORDERS_ACROSS_WAREHOUSES,
            'value' => 1,
        ]);

        /** @var Product $product1 */
        $product1 = Product::factory()->create();
        /** @var Product $product2 */
        $product2 = Product::factory()->create();

        /** @var Warehouse $warehouse1 */
        $warehouse1 = Warehouse::factory()->create()->withDefaultLocation();
        /** @var Warehouse $warehouse2 */
        $warehouse2 = Warehouse::factory()->create()->withDefaultLocation();

        ProductInventory::query()->insert([
            ['product_id' => $product1->id, 'warehouse_id' => $warehouse1->id],
            ['product_id' => $product1->id, 'warehouse_id' => $warehouse2->id],
            ['product_id' => $product2->id, 'warehouse_id' => $warehouse1->id],
            ['product_id' => $product2->id, 'warehouse_id' => $warehouse2->id],
        ]);

        // Create warehouse priority
        Setting::query()->updateOrCreate(['key' => Setting::KEY_WAREHOUSE_PRIORITY], [
            'key' => Setting::KEY_WAREHOUSE_PRIORITY,
            'value' => json_encode([$warehouse1->id, $warehouse2->id]),
        ]);

        // Product 2 has stock at warehouse 2
        $this->postJson('/api/inventory-adjustments', [
            'product_id' => $product2->id,
            'warehouse_id' => $warehouse2->id,
            'adjustment_date' => now(),
            'quantity' => 5,
            'unit_cost' => 10,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ])->assertSuccessful();

        // Create sales order with product 1 at warehouse 1 and product 2 at warehouse 2
        $this->postJson('/api/sales-orders', [
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_OPEN,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 3,
                    'amount' => 2,
                    'description' => $product1->name,
                ],
                [
                    'product_id' => $product2->id,
                    'quantity' => 4,
                    'amount' => 2,
                    'description' => $product2->name,
                ],
            ],
        ])->assertSuccessful();

        // Since warehouse 2 is the first that has stock and the supplier
        // has 'no-split' setting on, both lines should be at warehouse 2.
        $this->assertDatabaseHas('sales_order_lines', [
            'product_id' => $product1->id,
            'warehouse_id' => $warehouse2->id,
        ]);
        $this->assertDatabaseHas('sales_order_lines', [
            'product_id' => $product2->id,
            'warehouse_id' => $warehouse2->id,
        ]);

        // Inventory movements should also match warehouse selections
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $product1->id,
            'warehouse_id' => $warehouse2->id,
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $product2->id,
            'warehouse_id' => $warehouse2->id,
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);
    }

    public function test_it_wont_assign_all_items_in_an_order_to_same_warehouse_when_not_required(): void
    {
        Queue::fake();

        Sanctum::actingAs(User::factory()->create());

        Setting::query()->updateOrCreate([
            'key' => Setting::KEY_NEVER_SPLIT_SALES_ORDERS_ACROSS_WAREHOUSES,
        ], [
            'key' => Setting::KEY_NEVER_SPLIT_SALES_ORDERS_ACROSS_WAREHOUSES,
            'value' => 0,
        ]);

        /** @var Product $product1 */
        $product1 = Product::factory()->create();
        /** @var Product $product2 */
        $product2 = Product::factory()->create();

        /** @var Warehouse $warehouse1 */
        $warehouse1 = Warehouse::factory()->create()->withDefaultLocation();
        /** @var Warehouse $warehouse2 */
        $warehouse2 = Warehouse::factory()->create()->withDefaultLocation();

        ProductInventory::query()->insert([
            ['product_id' => $product1->id, 'warehouse_id' => $warehouse1->id],
            ['product_id' => $product1->id, 'warehouse_id' => $warehouse2->id],
            ['product_id' => $product2->id, 'warehouse_id' => $warehouse1->id],
            ['product_id' => $product2->id, 'warehouse_id' => $warehouse2->id],
        ]);

        // Create warehouse priority
        Setting::query()->updateOrCreate(['key' => Setting::KEY_WAREHOUSE_PRIORITY], [
            'key' => Setting::KEY_WAREHOUSE_PRIORITY,
            'value' => json_encode([$warehouse1->id, $warehouse2->id]),
        ]);

        // Product 2 has stock at warehouse 2
        $this->postJson('/api/inventory-adjustments', [
            'product_id' => $product2->id,
            'warehouse_id' => $warehouse2->id,
            'adjustment_date' => now(),
            'quantity' => 5,
            'unit_cost' => 10,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ])->assertSuccessful();

        // Create sales order with product 1 at warehouse 1 and product 2 at warehouse 2
        $this->postJson('/api/sales-orders', [
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_OPEN,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 3,
                    'warehouse_id' => $warehouse1->id,
                    'amount' => 2,
                    'description' => $product1->name,
                ],
                [
                    'product_id' => $product2->id,
                    'quantity' => 4,
                    'warehouse_id' => $warehouse2->id,
                    'amount' => 2,
                    'description' => $product2->name,
                ],
            ],
        ])->assertSuccessful();

        $this->assertDatabaseHas('sales_order_lines', [
            'product_id' => $product1->id,
            'warehouse_id' => $warehouse1->id,
        ]);
        $this->assertDatabaseHas('sales_order_lines', [
            'product_id' => $product2->id,
            'warehouse_id' => $warehouse2->id,
        ]);

        // Inventory movements should also match warehouse selections
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $product1->id,
            'warehouse_id' => $warehouse1->id,
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $product2->id,
            'warehouse_id' => $warehouse2->id,
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);
    }

    public function test_it_will_use_priority_warehouse_when_no_warehouse_is_in_stock(): void
    {
        Queue::fake();

        Sanctum::actingAs(User::factory()->create());

        Setting::query()->updateOrCreate([
            'key' => Setting::KEY_NEVER_SPLIT_SALES_ORDERS_ACROSS_WAREHOUSES,
        ], [
            'key' => Setting::KEY_NEVER_SPLIT_SALES_ORDERS_ACROSS_WAREHOUSES,
            'value' => 1,
        ]);

        /** @var Product $product1 */
        $product1 = Product::factory()->create();
        /** @var Product $product2 */
        $product2 = Product::factory()->create();

        /** @var Warehouse $warehouse1 */
        $warehouse1 = Warehouse::factory()->create()->withDefaultLocation();
        /** @var Warehouse $warehouse2 */
        $warehouse2 = Warehouse::factory()->create()->withDefaultLocation();

        ProductInventory::query()->insert([
            ['product_id' => $product1->id, 'warehouse_id' => $warehouse1->id],
            ['product_id' => $product1->id, 'warehouse_id' => $warehouse2->id],
            ['product_id' => $product2->id, 'warehouse_id' => $warehouse1->id],
            ['product_id' => $product2->id, 'warehouse_id' => $warehouse2->id],
        ]);

        /** @var Warehouse $warehouse3 */
        $warehouse3 = Warehouse::factory()->create();

        // Create warehouse priority
        Setting::query()->updateOrCreate(['key' => Setting::KEY_WAREHOUSE_PRIORITY], [
            'key' => Setting::KEY_WAREHOUSE_PRIORITY,
            'value' => json_encode([$warehouse3->id, $warehouse1->id, $warehouse2->id]),
        ]);

        // Create sales order with product 1 at warehouse 1 and product 2 at warehouse 2
        $this->postJson('/api/sales-orders', [
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_OPEN,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 3,
                    'amount' => 2,
                    'description' => $product1->name,
                ],
                [
                    'product_id' => $product2->id,
                    'quantity' => 4,
                    'amount' => 2,
                    'description' => $product2->name,
                ],
            ],
        ])->assertSuccessful();

        $this->assertDatabaseHas('sales_order_lines', [
            'product_id' => $product1->id,
            'warehouse_id' => $warehouse3->id,
        ]);
        $this->assertDatabaseHas('sales_order_lines', [
            'product_id' => $product2->id,
            'warehouse_id' => $warehouse3->id,
        ]);

        // Inventory movements should also match warehouse selections
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $product1->id,
            'warehouse_id' => $warehouse3->id,
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $product2->id,
            'warehouse_id' => $warehouse3->id,
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);
    }
}
