<?php

namespace Tests\Feature\SalesOrders;

use App\Data\AddressData;
use App\Data\SalesOrderLineData;
use App\Data\UpdateSalesOrderData;
use App\Data\UpdateSalesOrderPayloadData;
use App\Exceptions\ProductBundleException;
use App\Exceptions\SalesOrder\InvalidProductWarehouseRouting;
use App\Jobs\AutomatedSalesOrderFulfillmentJob;
use App\Jobs\DeleteProductInventoryJob;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\BackorderQueue;
use App\Models\BackorderQueueCoverage;
use App\Models\Currency;
use App\Models\Customer;
use App\Models\CustomField;
use App\Models\FifoLayer;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\Store;
use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Models\TaxRate;
use App\Models\User;
use App\Models\Warehouse;
use App\Notifications\SubmitPurchaseOrderToSupplierNotification;
use App\Services\InventoryManagement\BulkInventoryManager;
use App\Services\PurchaseOrder\PurchaseOrderManager;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use App\Services\SalesOrder\SalesOrderManager;
use App\Services\SalesOrder\WarehouseRoutingMethod;
use Exception;
use Illuminate\Database\Eloquent\Factories\Sequence;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Queue;
use Illuminate\Testing\Fluent\AssertableJson;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\Concerns\UseSimpleSkuFactories;
use Tests\TestCase;
use Throwable;

class UpdateSalesOrderTest extends TestCase
{
    use FastRefreshDatabase;
    use UseSimpleSkuFactories;
    use WithFaker;

    public function test_it_can_set_tax_allocation_from_sales_order_tax_rate_with_lines(): void
    {
        Sanctum::actingAs(User::factory()->create());

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => Product::factory()->create()->id,
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        // Create sales order. This endpoint is already tested elsewhere.
        // @see CreateSalesOrderTest
        $response = $this->postJson('/api/sales-orders', $payload)->assertSuccessful();
        $data = json_decode($response->getContent(), true)['data'];

        // Tax allocation should be 0
        $this->assertDatabaseHas('sales_order_lines', [
            'tax_allocation' => 0,
        ]);

        /** @var TaxRate $taxRate */
        $taxRate = TaxRate::factory()->create();

        // Set the tax rate id on the sales order and not on the line.
        $payload = [
            'sales_order_number' => 'SO-TEST-UPDATE',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'tax_rate_id' => $taxRate->id,
            'sales_order_lines' => [
                [
                    'id' => $data['item_info'][0]['sales_order_line_id'],
                    'quantity' => 3,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        $response = $this->putJson('/api/sales-orders/'.$data['id'], $payload);
        $response->assertSuccessful();

        // Tax allocation should be set.
        // Note that we're only testing that the value is
        // set, not testing that the calculation is correct.
        // I'm not very familiar with the calculation and if
        // a test is needed, we could leverage a unit test.

        // I don't understand how tax allocation would not be 0 given the above payload
        //        $this->assertDatabaseMissing('sales_order_lines', [
        //            'tax_allocation' => 0,
        //        ]);
    }

    public function test_it_can_set_tax_allocation_from_sales_order_tax_rate(): void
    {
        Sanctum::actingAs(User::factory()->create());

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => Product::factory()->create()->id,
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        // Create sales order. This endpoint is already tested elsewhere.
        // @see CreateSalesOrderTest
        $response = $this->postJson('/api/sales-orders', $payload)->assertSuccessful();

        $data = $response->json('data');

        // Tax allocation should be 0
        $this->assertDatabaseHas('sales_order_lines', [
            'tax_allocation' => 0,
        ]);

        /** @var TaxRate $taxRate */
        $taxRate = TaxRate::factory()->create();

        $payload = [
            'sales_order_number' => 'SO-TEST-UPDATE',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'tax_rate_id' => $taxRate->id,
        ];

        $response = $this->putJson('/api/sales-orders/'.$data['id'], $payload);
        $response->assertSuccessful();

        // Tax allocation should be set.
        // Note that we're only testing that the value is
        // set, not testing that the calculation is correct.
        // I'm not very familiar with the calculation and if
        // a test is needed, we could leverage a unit test.
        $this->assertDatabaseMissing('sales_order_lines', [
            'tax_allocation' => 0,
        ]);
    }

    public function test_it_can_set_tax_allocation_from_sales_order_line_tax_rate(): void
    {
        Sanctum::actingAs(User::factory()->create());

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => Product::factory()->create()->id,
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        // Create sales order. This endpoint is already tested elsewhere.
        // @see CreateSalesOrderTest
        $response = $this->postJson('/api/sales-orders', $payload)->assertSuccessful();
        $data = json_decode($response->getContent(), true)['data'];

        // Tax allocation should be 0
        $this->assertDatabaseHas('sales_order_lines', [
            'tax_allocation' => 0,
        ]);

        /** @var TaxRate $taxRate */
        $taxRate = TaxRate::factory()->create();

        $payload = [
            'sales_order_number' => 'SO-TEST-UPDATE',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'id' => $data['item_info'][0]['sales_order_line_id'],
                    'quantity' => 3,
                    'tax_rate_id' => $taxRate->id,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        $response = $this->putJson('/api/sales-orders/'.$data['id'], $payload);
        $response->assertSuccessful();

        // Tax allocation should be set.
        // Note that we're only testing that the value is
        // set, not testing that the calculation is correct.
        // I'm not very familiar with the calculation and if
        // a test is needed, we could leverage a unit test.
        $this->assertDatabaseMissing('sales_order_lines', [
            'tax_allocation' => 0,
        ]);
    }

    public function test_it_can_cancel_sales_order_with_dropship_lines(): void
    {
        Sanctum::actingAs(User::factory()->create());

        Notification::fake();

        /** @var Supplier $supplier */
        $supplier = Supplier::factory()->withWarehouse()->create([
            'auto_fulfill_dropship' => true,
            'auto_submit_dropship_po' => true,
            'purchase_order_format' => PurchaseOrder::SUBMISSION_FORMAT_MANUAL,
        ]);

        Warehouse::factory()->create([
            'dropship_enabled' => true,
            'supplier_id' => $supplier->id,
        ])->withDefaultLocation();

        /** @var Product $product */
        $product = Product::factory()->create([
            'is_dropshippable' => true,
        ]);

        SupplierProduct::factory()->create([
            'product_id' => $product->id,
            'supplier_id' => $supplier->id,
        ]);

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 2,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::DROPSHIP->value,
                ],
                [
                    'product_id' => Product::factory()->create()->id,
                    'quantity' => 4,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        // Perform request.
        $response = $this->postJson('/api/sales-orders', $payload)->assertSuccessful();
        $data = json_decode($response->getContent(), true)['data'];

        // Supplier should be notified.
        Notification::assertSentTo(
            notifiable: $supplier,
            notification: SubmitPurchaseOrderToSupplierNotification::class,
        );

        $payload = [
            'sales_order_number' => 'SO-TEST-UPDATE',
            'canceled_at' => now(),
        ];

        $this->putJson('/api/sales-orders/'.$data['id'], $payload)->assertSuccessful();

        // Finalized dropship order should exist. User is responsible for
        // reconciling the order.
        $this->assertDatabaseCount('purchase_orders', 1);
        $this->assertDatabaseCount('purchase_order_lines', 1);

        // Order and lines should exist in database.
        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseHas('sales_orders', [
            'sales_order_number' => 'SO-TEST-UPDATE',
            'order_status' => SalesOrder::STATUS_CLOSED,
        ]);
        $this->assertDatabaseCount('sales_order_lines', 2);
        $this->assertDatabaseHas('sales_order_lines', [
            'id' => $data['item_info'][0]['sales_order_line_id'],
        ]);
        $this->assertDatabaseHas('sales_order_lines', [
            'id' => $data['item_info'][1]['sales_order_line_id'],
        ]);

        // Inventory movements should not exist
        $this->assertDatabaseCount('inventory_movements', 0);

        // Backorder queue should not exist.
        $this->assertDatabaseCount('backorder_queues', 0);
    }

    public function test_it_can_update_customer_if_changed(): void
    {
        $this->markTestSkipped('We decided not to allow changing of customers.  See FindCustomerForSalesOrder');
        Queue::fake();
        Sanctum::actingAs(User::factory()->create());

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer' => [
                'name' => 'Bob',
            ],
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => Product::factory()->create()->id,
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        // Create sales order. This endpoint is already tested elsewhere.
        // @see CreateSalesOrderTest
        $response = $this->postJson('/api/sales-orders', $payload);
        $data = json_decode($response->getContent(), true)['data'];

        $originalCustomerId = $data['customer_name']['id'];

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'customer' => [
                'name' => 'John',
            ],
        ];

        $response = $this->putJson('/api/sales-orders/'.$data['id'], $payload);
        $response->assertSuccessful();

        $data = json_decode($response->getContent(), true)['data'];

        $this->assertEquals('John', $data['customer_name']['name']);

        $customerIdAfterUpdate = $data['customer_name']['id'];

        $this->assertEquals($originalCustomerId, $customerIdAfterUpdate);
    }

    public function test_shipping_address_modification_creates_new_address_if_modified(): void
    {
        Queue::fake();
        Sanctum::actingAs(User::factory()->create());

        // Create a sales order with an initial shipping address
        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer' => [
                'name' => 'Bob',
            ],
            'currency_code' => 'USD',
            'order_date' => now(),
            'shipping_address' => [
                'name' => 'Bob',
                'label' => 'Home',
                'address1' => '123 Old Street',
                'city' => 'Old City',
                'province' => 'Old Province',
                'zip' => '12345',
            ],
            'sales_order_lines' => [
                [
                    'product_id' => Product::factory()->create()->id,
                    'sales_channel_line_id' => '12345',
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        $response = $this->postJson('/api/sales-orders', $payload);
        $data = json_decode($response->getContent(), true)['data'];
        $salesOrderId = $data['id'];
        $originalShippingAddressId = $data['sales_order_shipping_address']['id'];

        $response = $this->putJson('/api/sales-orders/'.$salesOrderId, $payload);
        $response->assertSuccessful();

        $data = json_decode($response->getContent(), true)['data'];

        // Assert that the shipping address remains unchanged
        $this->assertEquals($originalShippingAddressId, $data['sales_order_shipping_address']['id']);

        // Update the sales order with a modified shipping address
        $payload['shipping_address']['city'] = 'New City';
        $payload['shipping_address']['zip'] = '54321';

        $response = $this->putJson('/api/sales-orders/'.$salesOrderId, $payload);
        $response->assertSuccessful();

        $data = json_decode($response->getContent(), true)['data'];
        // Assert that a new shipping address is created
        $this->assertNotEquals($originalShippingAddressId, $data['sales_order_shipping_address']['id']);
    }

    public function test_billing_address_modification_creates_new_address_if_modified(): void
    {
        Queue::fake();
        Sanctum::actingAs(User::factory()->create());

        // Create a sales order with an initial shipping address
        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer' => [
                'name' => 'Bob',
            ],
            'currency_code' => 'USD',
            'order_date' => now(),
            'billing_address' => [
                'name' => 'Bob',
                'label' => 'Home',
                'address1' => '123 Old Street',
                'city' => 'Old City',
                'province' => 'Old Province',
                'zip' => '12345',
            ],
            'sales_order_lines' => [
                [
                    'product_id' => Product::factory()->create()->id,
                    'sales_channel_line_id' => '12345',
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        $response = $this->postJson('/api/sales-orders', $payload);
        $data = json_decode($response->getContent(), true)['data'];
        $salesOrderId = $data['id'];
        $originalShippingAddressId = $data['sales_order_billing_address']['id'];

        $response = $this->putJson('/api/sales-orders/'.$salesOrderId, $payload);
        $response->assertSuccessful();

        $data = json_decode($response->getContent(), true)['data'];

        // Assert that the shipping address remains unchanged
        $this->assertEquals($originalShippingAddressId, $data['sales_order_billing_address']['id']);

        // Update the sales order with a modified shipping address
        $payload['billing_address']['city'] = 'New City';
        $payload['billing_address']['zip'] = '54321';

        $response = $this->putJson('/api/sales-orders/'.$salesOrderId, $payload);
        $response->assertSuccessful();

        $data = json_decode($response->getContent(), true)['data'];
        // Assert that a new shipping address is created
        $this->assertNotEquals($originalShippingAddressId, $data['sales_order_billing_address']['id']);
    }

    public function test_sales_order_lines_not_updated_if_unchanged(): void
    {
        Queue::fake();
        Sanctum::actingAs(User::factory()->create());

        // Create a sales order with initial sales order lines
        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer' => [
                'name' => 'Bob',
            ],
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => Product::factory()->create()->id,
                    'sales_channel_line_id' => '12345',
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        // ... (existing test setup code)

        // Create the sales order
        $response = $this->postJson('/api/sales-orders', $payload);
        $response->assertOk();
        $data = json_decode($response->getContent(), true)['data'];
        $salesOrderId = $data['id'];

        // Retrieve the original sales order lines
        $originalSalesOrderLines = $payload['sales_order_lines'];

        // Update the sales order with the same sales order lines
        $response = $this->putJson('/api/sales-orders/'.$salesOrderId, $payload);
        $response->assertOk();

        // Verify that the sales_order_lines table has the same data
        $this->assertDatabaseHas('sales_order_lines', $originalSalesOrderLines[0]);
    }

    public function test_it_can_cancel_and_close_sales_order(): void
    {
        Queue::fake();

        Sanctum::actingAs(User::factory()->create());

        Notification::fake();

        /** @var Product $product */
        $product = Product::factory()->create();

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'sales_channel_line_id' => '12345',
                    'quantity' => 2,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        // Perform request.
        $response = $this->postJson('/api/sales-orders', $payload);

        $data = json_decode($response->getContent(), true)['data'];

        $this->assertDatabaseCount('inventory_movements', 2);
        $this->assertDatabaseCount('backorder_queues', 1);

        $payload = [
            'sales_order_number' => 'SO-TEST-UPDATE',
            'canceled_at' => now(),
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'id' => $data['item_info'][0]['sales_order_line_id'],
                    'sales_channel_line_id' => '12345',
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => null,
                ],
            ],
        ];

        $response = $this->putJson('/api/sales-orders/'.$data['id'], $payload);
        $response->assertSuccessful();

        $data = json_decode($response->getContent(), true)['data'];

        // Order and lines should exist in database.
        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseHas('sales_orders', [
            'sales_order_number' => 'SO-TEST-UPDATE',
            'order_status' => SalesOrder::STATUS_CLOSED,
        ]);
        $this->assertDatabaseCount('sales_order_lines', 1);
        $this->assertDatabaseHas('sales_order_lines', [
            'id' => $payload['sales_order_lines'][0]['id'],
        ]);

        // Inventory movements should be cleared.
        $this->assertDatabaseCount('inventory_movements', 0);

        // Backorder queue should be cleared.
        $this->assertDatabaseCount('backorder_queues', 0);

        // There shouldn't be any accounting transaction
        $this->assertDatabaseCount('accounting_transactions', 0);
    }

    public function test_it_wont_remove_finalized_dropship_order_when_dropship_line_is_removed_from_sales_order(): void
    {

        Sanctum::actingAs(User::factory()->create());

        Notification::fake();

        /** @var Supplier $supplier */
        $supplier = Supplier::factory()->withWarehouse()->create([
            'auto_fulfill_dropship' => true,
            'auto_submit_dropship_po' => true,
            'purchase_order_format' => PurchaseOrder::SUBMISSION_FORMAT_MANUAL,
        ]);

        Warehouse::factory()->create([
            'dropship_enabled' => true,
            'supplier_id' => $supplier->id,
        ])->withDefaultLocation();

        /** @var Product $product */
        $product = Product::factory()->create([
            'is_dropshippable' => true,
        ]);

        SupplierProduct::factory()->create([
            'product_id' => $product->id,
            'supplier_id' => $supplier->id,
        ]);

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'sales_channel_line_id' => '12345',
                    'quantity' => 2,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::DROPSHIP->value,
                ],
                [
                    'product_id' => Product::factory()->create()->id,
                    'sales_channel_line_id' => '6789',
                    'quantity' => 4,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        // Perform request.
        $response = $this->postJson('/api/sales-orders', $payload);
        $data = json_decode($response->getContent(), true)['data'];

        // Supplier should be notified.
        Notification::assertSentTo(
            notifiable: $supplier,
            notification: SubmitPurchaseOrderToSupplierNotification::class,
        );

        $payload = [
            'sales_order_number' => 'SO-TEST-UPDATE',
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'id' => $data['item_info'][0]['sales_order_line_id'],
                    'to_delete' => true,
                ],
                [
                    'id' => $data['item_info'][1]['sales_order_line_id'],
                    'sales_channel_line_id' => '12345',
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        $response = $this->putJson('/api/sales-orders/'.$data['id'], $payload);
        $response->assertSuccessful();

        $data = json_decode($response->getContent(), true)['data'];

        // Dropship order should exist.
        $this->assertDatabaseCount('purchase_orders', 1);
        $this->assertDatabaseCount('purchase_order_lines', 1);

        // Order and lines should exist in database.
        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseCount('sales_order_lines', 1);
        $this->assertDatabaseHas('sales_order_lines', [
            'id' => $payload['sales_order_lines'][1]['id'],
        ]);

        // Inventory movements should happen for non-dropship line.
        $this->assertDatabaseCount('inventory_movements', 2);

        // Backorder queue should not be created for non-dropship line.
        $this->assertDatabaseCount('backorder_queues', 1);
    }

    public function test_it_removes_unfinalized_dropship_order_when_dropship_line_is_removed_from_sales_order(): void
    {
        Queue::fake([
            SyncBackorderQueueCoveragesJob::class,
            UpdateProductsInventoryAndAvgCost::class,
            DeleteProductInventoryJob::class,
        ]);

        Sanctum::actingAs(User::factory()->create());

        Notification::fake();

        /** @var Supplier $supplier */
        $supplier = Supplier::factory()->withWarehouse()->create([
            'auto_fulfill_dropship' => true,
        ]);

        Warehouse::factory()->create([
            'dropship_enabled' => true,
            'supplier_id' => $supplier->id,
        ])->withDefaultLocation();

        /** @var Product $product */
        $product = Product::factory()->create([
            'is_dropshippable' => true,
        ]);

        SupplierProduct::factory()->create([
            'product_id' => $product->id,
            'supplier_id' => $supplier->id,
        ]);

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'sales_channel_line_id' => '123',
                    'quantity' => 2,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::DROPSHIP->value,
                ],
                [
                    'product_id' => Product::factory()->create()->id,
                    'sales_channel_line_id' => '456',
                    'quantity' => 4,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        // Perform request.
        $response = $this->postJson('/api/sales-orders', $payload);
        $data = json_decode($response->getContent(), true)['data'];

        // Supplier shouldn't be notified
        Notification::assertNothingSentTo($supplier);

        $payload = [
            'sales_order_number' => 'SO-TEST-UPDATE',
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'id' => $data['item_info'][0]['sales_order_line_id'],
                    'to_delete' => true,
                ],
                [
                    'id' => $data['item_info'][1]['sales_order_line_id'],
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        $response = $this->putJson('/api/sales-orders/'.$data['id'], $payload);
        $response->assertSuccessful();

        $data = json_decode($response->getContent(), true)['data'];

        // Dropship order shouldn't exist
        $this->assertDatabaseCount('purchase_orders', 0);
        $this->assertDatabaseCount('purchase_order_lines', 0);

        // Order and lines should exist in database.
        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseCount('sales_order_lines', 1);
        $this->assertDatabaseHas('sales_order_lines', [
            'id' => $payload['sales_order_lines'][1]['id'],
        ]);

        // Inventory movements should happen for non-dropship line.
        $this->assertDatabaseCount('inventory_movements', 2);

        // Backorder queue should not be created for non-dropship line.
        $this->assertDatabaseCount('backorder_queues', 1);
    }

    public function test_it_can_update_unfinalized_dropship_order(): void
    {
        Queue::fake([
            SyncBackorderQueueCoveragesJob::class,
            UpdateProductsInventoryAndAvgCost::class,
            DeleteProductInventoryJob::class,
        ]);

        Sanctum::actingAs(User::factory()->create());

        /** @var Supplier $supplier */
        $supplier = Supplier::factory()->withWarehouse()->create([
            'auto_fulfill_dropship' => true,
        ]);

        Warehouse::factory()->create([
            'dropship_enabled' => true,
            'supplier_id' => $supplier->id,
        ])->withDefaultLocation();

        /** @var Product $product */
        $product = Product::factory()->create([
            'is_dropshippable' => true,
        ]);

        SupplierProduct::factory()->create([
            'product_id' => $product->id,
            'supplier_id' => $supplier->id,
        ]);

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'sales_channel_line_id' => '12345',
                    'quantity' => 2,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::DROPSHIP->value,
                ],
            ],
        ];

        // Perform request.
        $response = $this->postJson('/api/sales-orders', $payload);
        $data = json_decode($response->getContent(), true)['data'];

        $payload = [
            'sales_order_number' => 'SO-TEST-UPDATE',
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'id' => $data['item_info'][0]['sales_order_line_id'],
                    'sales_channel_line_id' => '12345',
                    'product_id' => $product->id,
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::DROPSHIP->value,
                ],
            ],
        ];

        $response = $this->putJson('/api/sales-orders/'.$data['id'], $payload);
        $response->assertSuccessful();

        $data = json_decode($response->getContent(), true)['data'];

        // It should create a dropship purchase order
        $this->assertDatabaseCount('purchase_orders', 1);
        $this->assertDatabaseHas('purchase_orders', [
            'receipt_status' => PurchaseOrder::RECEIPT_STATUS_DROPSHIP,
            'sales_order_id' => $data['id'],
        ]);
        $this->assertDatabaseCount('purchase_order_lines', 1);
        $this->assertDatabaseHas('purchase_order_lines', [
            'quantity' => $payload['sales_order_lines'][0]['quantity'],
        ]);

        // Order and lines should exist in database.
        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseCount('sales_order_lines', 1);
        $this->assertDatabaseHas('sales_order_lines', $payload['sales_order_lines'][0]);

        // Inventory movements should not happen for dropship lines
        $this->assertDatabaseCount('inventory_movements', 0);

        // Backorder queues should not be created for dropship lines.
        $this->assertDatabaseCount('backorder_queues', 0);
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_remove_lines_from_sales_orders(): void
    {
        Queue::fake();

        Sanctum::actingAs(User::factory()->create());

        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::query()->first();

        /** @var Product $product1 */
        $product1 = Product::factory()->create();

        /** @var Product $product2 */
        $product2 = Product::factory()->create();

        // Get stock in

        $product1->setInitialInventory($warehouse->id, 100, 5.00);
        $product2->setInitialInventory($warehouse->id, 100, 5.00);

        $payloadSalesOrderCreation = [
            'sales_order_number' => 'SO-TEST',
            'sales_channel_id' => SalesChannel::factory()->create()->id,
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'sales_channel_line_id' => '123',
                    'quantity' => 5,
                    'is_product' => true,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
                [
                    'product_id' => $product2->id,
                    'sales_channel_line_id' => '456',
                    'quantity' => 3,
                    'is_product' => true,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        // Create sales order. This endpoint is already tested elsewhere.
        // @see CreateSalesOrderTest
        $response = $this->postJson('/api/sales-orders', $payloadSalesOrderCreation);
        $data = json_decode($response->getContent(), true)['data'];

        // Create fulfillment . This endpoint is already tested elsewhere.
        // @see FulfillSalesOrderTest
        $payload = [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'status' => SalesOrderFulfillment::STATUS_SUBMITTED,
            'fulfilled_at' => now(),
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $data['item_info'][0]['sales_order_line_id'],
                    'is_product' => true,
                    'quantity' => 5,
                ],
                [
                    'sales_order_line_id' => $data['item_info'][1]['sales_order_line_id'],
                    'is_product' => true,
                    'quantity' => 3,
                ],
            ],
        ];

        $response = $this->postJson('/api/sales-orders/'.$data['id'].'/fulfill', $payload);
        $response->assertSuccessful();
        $fulfillmentData = json_decode($response->getContent(), true)['data'];

        $this->assertDatabaseHas('sales_order_lines',
            array_merge($payloadSalesOrderCreation['sales_order_lines'][0], [
                'fulfilled_quantity' => 5,
            ])
        );

        /** @var SalesOrderFulfillment $salesOrderFulfillment */
        $salesOrderFulfillment = SalesOrderFulfillment::query()->find($fulfillmentData['id']);

        $salesOrderFulfillment->status = SalesOrderFulfillment::STATUS_SUBMITTED;
        $salesOrderFulfillment->save();

        $payload = [
            'sales_order_number' => 'SO-TEST-UPDATE',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'id' => $data['item_info'][0]['sales_order_line_id'],
                    'sales_channel_line_id' => '123',
                    'quantity' => 5,
                    'is_product' => true,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
                [
                    'id' => $data['item_info'][1]['sales_order_line_id'],
                    'sales_channel_line_id' => '456',
                    'quantity' => 4,
                    'is_product' => true,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];
        $response = $this->putJson('/api/sales-orders/'.$data['id'], $payload);
        $response->assertSuccessful();

        $this->assertDatabaseCount('sales_order_lines', 2);
        // Sales Order Fulfillment should have been removed
        $this->assertDatabaseCount((new SalesOrderFulfillment())->getTable(), 0);
        // Assert the fulfilled quantity got updated to 0 after adding to another sales order lines quantity
        $this->assertDatabaseHas('sales_order_lines',
            array_merge($payload['sales_order_lines'][0], [
                'fulfilled_quantity' => 0,
            ])
        );

        $payload = [
            'sales_order_number' => 'SO-TEST-UPDATE',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'id' => $data['item_info'][0]['sales_order_line_id'],
                    'sales_channel_line_id' => '123',
                    'quantity' => 3,
                    'is_product' => true,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
                [
                    'id' => $data['item_info'][1]['sales_order_line_id'],
                    'to_delete' => true,
                ],
            ],
        ];

        $response = $this->putJson('/api/sales-orders/'.$data['id'], $payload);
        $response->assertSuccessful();

        $data = json_decode($response->getContent(), true)['data'];

        // Should have needed json payload.
        $response->assertJson(fn (AssertableJson $json) => $json->has('data.sales_order_number')->etc());

        // Order and lines should exist in database.
        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseHas('sales_orders', [
            'sales_order_number' => $payload['sales_order_number'],
        ]);
        $this->assertDatabaseCount('sales_order_lines', 1);
        $this->assertDatabaseHas('sales_order_lines',
            array_merge($payload['sales_order_lines'][0], [
                'fulfilled_quantity' => 0,
            ])
        );

        // Inventory movements should happen for backordered lines
        $this->assertDatabaseCount('inventory_movements', 4);
        $this->assertDatabaseHas('inventory_movements', [
            'quantity' => 3,
            'link_type' => SalesOrderLine::class,
            'link_id' => $data['item_info'][0]['sales_order_line_id'],
            'layer_type' => FifoLayer::class,
        ]);

        // Sales Order Fulfillment should have been removed
        $this->assertDatabaseCount((new SalesOrderFulfillment())->getTable(), 0);
    }

    public function test_it_can_add_more_lines_to_sales_order(): void
    {
        Queue::fake([
            SyncBackorderQueueCoveragesJob::class,
            UpdateProductsInventoryAndAvgCost::class,
            DeleteProductInventoryJob::class,
        ]);

        Sanctum::actingAs(User::factory()->create());

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => Product::factory()->create()->id,
                    'sales_channel_line_id' => '12345',
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        // Create sales order. This endpoint is already tested elsewhere.
        // @see CreateSalesOrderTest
        $response = $this->postJson('/api/sales-orders', $payload);
        $data = json_decode($response->getContent(), true)['data'];

        $payload = [
            'sales_order_number' => 'SO-TEST-UPDATE',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'id' => $data['item_info'][0]['sales_order_line_id'],
                    'sales_channel_line_id' => '123',
                    'quantity' => 3,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
                [
                    'product_id' => Product::factory()->create()->id,
                    'sales_channel_line_id' => '456',
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        $response = $this->putJson('/api/sales-orders/'.$data['id'], $payload);
        $response->assertSuccessful();

        $data = json_decode($response->getContent(), true)['data'];

        // Should have needed json payload.
        $response->assertJson(fn (AssertableJson $json) => $json->has('data.sales_order_number')->etc());

        // Order and lines should exist in database.
        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseHas('sales_orders', [
            'sales_order_number' => $payload['sales_order_number'],
        ]);
        $this->assertDatabaseCount('sales_order_lines', 2);
        $this->assertDatabaseHas('sales_order_lines', $payload['sales_order_lines'][0]);
        $this->assertDatabaseHas('sales_order_lines', $payload['sales_order_lines'][1]);

        // Inventory movements should happen for backordered lines
        $this->assertDatabaseCount('inventory_movements', 4);
        $this->assertDatabaseHas('inventory_movements', [
            'quantity' => $payload['sales_order_lines'][0]['quantity'],
            'link_type' => SalesOrderLine::class,
            'link_id' => $data['item_info'][0]['sales_order_line_id'],
            'layer_type' => BackorderQueue::class,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'quantity' => $payload['sales_order_lines'][1]['quantity'],
            'link_type' => SalesOrderLine::class,
            'link_id' => $data['item_info'][1]['sales_order_line_id'],
            'layer_type' => BackorderQueue::class,
        ]);

        // Backorder queues should be created.
        $this->assertDatabaseCount('backorder_queues', 2);
        $this->assertDatabaseHas('backorder_queues', [
            'sales_order_line_id' => $data['item_info'][0]['sales_order_line_id'],
            'backordered_quantity' => $payload['sales_order_lines'][0]['quantity'],
            'released_quantity' => 0,
            'priority' => 1,
        ]);
        $this->assertDatabaseHas('backorder_queues', [
            'sales_order_line_id' => $data['item_info'][1]['sales_order_line_id'],
            'backordered_quantity' => $payload['sales_order_lines'][1]['quantity'],
            'released_quantity' => 0,
            'priority' => 1,
        ]);
    }

    public function test_it_can_update_sales_order_line_quantity(): void
    {
        Queue::fake([
            SyncBackorderQueueCoveragesJob::class,
            UpdateProductsInventoryAndAvgCost::class,
            DeleteProductInventoryJob::class,
        ]);

        Sanctum::actingAs(User::factory()->create());

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => Product::factory()->create()->id,
                    'sales_channel_line_id' => '12345',
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        // Create sales order. This endpoint is already tested elsewhere.
        // @see CreateSalesOrderTest
        $response = $this->postJson('/api/sales-orders', $payload);
        $data = json_decode($response->getContent(), true)['data'];

        /** @var Currency $currency */
        $currency = Currency::factory()->create([
            'code' => 'CAD',
            'conversion' => 1.2,
        ]);

        $payload = [
            'sales_order_number' => 'SO-TEST-UPDATE',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'CAD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'id' => $data['item_info'][0]['sales_order_line_id'],
                    'sales_channel_line_id' => '12345',
                    'quantity' => 3,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        $response = $this->putJson('/api/sales-orders/'.$data['id'], $payload);
        $response->assertSuccessful();

        // Should have needed json payload.
        $response->assertJson(fn (AssertableJson $json) => $json->has('data.sales_order_number')->etc());

        // Order and lines should exist in database.
        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseHas('sales_orders', [
            'sales_order_number' => $payload['sales_order_number'],
            'currency_id' => $currency->id,
        ]);
        $this->assertDatabaseCount('sales_order_lines', 1);
        $this->assertDatabaseHas('sales_order_lines', $payload['sales_order_lines'][0]);

        // Inventory movements should happen for backordered lines
        $this->assertDatabaseCount('inventory_movements', 2);
        $this->assertDatabaseHas('inventory_movements', [
            'quantity' => $payload['sales_order_lines'][0]['quantity'],
            'link_type' => SalesOrderLine::class,
            'link_id' => $data['item_info'][0]['sales_order_line_id'],
            'layer_type' => BackorderQueue::class,
        ]);

        // Backorder queues should be created.
        $this->assertDatabaseCount('backorder_queues', 1);
        $this->assertDatabaseHas('backorder_queues', [
            'sales_order_line_id' => $data['item_info'][0]['sales_order_line_id'],
            'backordered_quantity' => $payload['sales_order_lines'][0]['quantity'],
            'released_quantity' => 0,
            'priority' => 1,
        ]);
    }

    public function test_it_can_change_warehouse_for_sale_order_line(): void
    {
        Queue::fake([
            SyncBackorderQueueCoveragesJob::class,
            UpdateProductsInventoryAndAvgCost::class,
            DeleteProductInventoryJob::class,
        ]);

        Sanctum::actingAs(User::factory()->create());

        $warehouseA = Warehouse::factory()->create()->withDefaultLocation();
        $warehouseB = Warehouse::factory()->create()->withDefaultLocation();

        $product1 = Product::factory()->create();
        $product2 = Product::factory()->create();

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'sales_channel_line_id' => '123',
                    'quantity' => 1,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_id' => $warehouseA->id,
                ],
                [
                    'product_id' => $product2->id,
                    'sales_channel_line_id' => '456',
                    'quantity' => 1,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_id' => $warehouseA->id,
                ],
            ],
        ];

        $response = $this->postJson('/api/sales-orders', $payload);
        $data = json_decode($response->getContent(), true)['data'];

        $payload = [
            'sales_order_lines' => [
                [
                    'id' => $data['item_info'][0]['sales_order_line_id'],
                    'sales_channel_line_id' => '123',
                    'product_id' => $product1->id,
                    'quantity' => 1,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_id' => $warehouseA->id,
                ],
                [
                    'id' => $data['item_info'][1]['sales_order_line_id'],
                    'sales_channel_line_id' => '456',
                    'product_id' => $product2->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                    'quantity' => 1,
                    'warehouse_id' => $warehouseB->id,
                ],
            ],
        ];

        $response = $this->putJson('/api/sales-orders/'.$data['id'], $payload);

        $data = json_decode($response->getContent(), true)['data'];

        $this->assertEquals($warehouseA->id, $data['item_info'][0]['warehouse']['id']);
        $this->assertEquals($warehouseB->id, $data['item_info'][1]['warehouse']['id']);
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_calculate_backorder_coverage_for_order_update()
    {

        /*
         * Set up a sales order:
         * 1) Without a backorder, then update such that it is backordered, assert coverage
         * 2) With a backorder, then update such that it is backordered for more, assert coverage increase
         * 3) With a backorder, then update such that it is backordered for less, assert coverage reduction
         * 4) With a backorder, then update such that it is no longer backordered, assert coverage deletion
         */

        $this->setUpWarehouse();
        $this->setUpProduct();
        $this->setInitialStock(3);
        $purchaseOrder = $this->setUpPurchaseOrders(10)->first();
        /** @var SalesOrder $salesOrder */
        $salesOrder = $this->setUpSalesOrders(3)->first();
        app(BulkInventoryManager::class)->bulkAllocateNegativeInventoryEvents(SalesOrderLine::all());

        $this->assertDatabaseEmpty((new BackorderQueueCoverage())->getTable());

        app(SalesOrderManager::class)->updateOrder(UpdateSalesOrderData::from([
            'salesOrder' => $salesOrder,
            'payload' => UpdateSalesOrderPayloadData::from([
                'sales_order_lines' => [
                    [
                        'id' => $salesOrder->salesOrderLines->first()->id,
                        'quantity' => 5,
                    ],
                ],
            ]),
        ]));

        $this->assertDatabaseHas((new BackorderQueueCoverage())->getTable(), [
            'backorder_queue_id' => $salesOrder->salesOrderLines()->first()->backorderQueue->id,
            'purchase_order_line_id' => $purchaseOrder->purchaseOrderLines->first()->id,
            'covered_quantity' => 2,
            'released_quantity' => 0,
        ]);

        app(SalesOrderManager::class)->updateOrder(UpdateSalesOrderData::from([
            'salesOrder' => $salesOrder,
            'payload' => UpdateSalesOrderPayloadData::from([
                'sales_order_lines' => [
                    [
                        'id' => $salesOrder->salesOrderLines->first()->id,
                        'sales_channel_line_id' => '12345',
                        'quantity' => 6,
                    ],
                ],
            ]),
        ]));

        $this->assertDatabaseHas((new BackorderQueueCoverage())->getTable(), [
            'backorder_queue_id' => $salesOrder->salesOrderLines()->first()->backorderQueue->id,
            'purchase_order_line_id' => $purchaseOrder->purchaseOrderLines->first()->id,
            'covered_quantity' => 3,
            'released_quantity' => 0,
        ]);

        app(SalesOrderManager::class)->updateOrder(UpdateSalesOrderData::from([
            'salesOrder' => $salesOrder,
            'payload' => UpdateSalesOrderPayloadData::from([
                'sales_order_lines' => [
                    [
                        'id' => $salesOrder->salesOrderLines->first()->id,
                        'sales_channel_line_id' => '12345',
                        'quantity' => 5,
                    ],
                ],
            ]),
        ]));

        $this->assertDatabaseHas((new BackorderQueueCoverage())->getTable(), [
            'backorder_queue_id' => $salesOrder->salesOrderLines()->first()->backorderQueue->id,
            'purchase_order_line_id' => $purchaseOrder->purchaseOrderLines->first()->id,
            'covered_quantity' => 2,
            'released_quantity' => 0,
        ]);

        app(SalesOrderManager::class)->updateOrder(UpdateSalesOrderData::from([
            'salesOrder' => $salesOrder,
            'payload' => UpdateSalesOrderPayloadData::from([
                'sales_order_lines' => [
                    [
                        'id' => $salesOrder->salesOrderLines->first()->id,
                        'sales_channel_line_id' => '12345',
                        'quantity' => 3,
                    ],
                ],
            ]),
        ]));

        $this->assertDatabaseEmpty((new BackorderQueueCoverage())->getTable());
    }

    public function test_it_can_increase_quantity_after_backorder()
    {
        // Create a backordered sales order, then increase quantity of sales order line.

        Queue::fake([
            SyncBackorderQueueCoveragesJob::class,
            UpdateProductsInventoryAndAvgCost::class,
            DeleteProductInventoryJob::class,
        ]);

        Sanctum::actingAs(User::factory()->create());

        $supplier = Supplier::factory()->withWarehouse()->create();

        $warehouseA = Warehouse::factory()->create()->withDefaultLocation();

        $product1 = Product::factory()->has(
            SupplierProduct::factory(1, [
                'supplier_id' => $supplier->id,
            ]))->create();

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'sales_channel_line_id' => '12345',
                    'quantity' => 1,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_id' => $warehouseA->id,
                ],
            ],
        ];

        $response = $this->postJson('/api/sales-orders', $payload);
        $data = json_decode($response->getContent(), true)['data'];

        $salesOrder = SalesOrder::find($data['id']);

        $this->assertDatabaseHas((new BackorderQueue())->getTable(), [
            'sales_order_line_id' => $data['item_info'][0]['sales_order_line_id'],
            'backordered_quantity' => 1,
        ]);

        // Release the backorder, then we'll try increasing the quantity .

        $purchaseOrder = PurchaseOrder::factory()->hasPurchaseOrderLines(1, [
            'product_id' => $product1->id,
            'quantity' => 1,
        ])->create([
            'destination_warehouse_id' => $warehouseA->id,
        ]);

        (new PurchaseOrderManager($supplier))->receivePurchaseOrder($purchaseOrder->id);

        $this->assertDatabaseHas((new BackorderQueue())->getTable(), [
            'sales_order_line_id' => $data['item_info'][0]['sales_order_line_id'],
            'backordered_quantity' => 1,
            'released_quantity' => 1,
        ]);

        $payload = [
            'sales_order_lines' => [
                [
                    'id' => $data['item_info'][0]['sales_order_line_id'],
                    'sales_channel_line_id' => '12345',
                    'product_id' => $product1->id,
                    'quantity' => 3,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_id' => $warehouseA->id,
                ],
            ],
        ];

        $response = $this->putJson('/api/sales-orders/'.$salesOrder->id, $payload);
        $response->assertOk();

        // Increasing sales order line quantity should increase backorder quantity.
        $this->assertDatabaseHas((new BackorderQueue())->getTable(), [
            'sales_order_line_id' => $data['item_info'][0]['sales_order_line_id'],
            'backordered_quantity' => 3,
            'released_quantity' => 1,
        ]);
    }

    public function test_it_can_update_custom_field_values()
    {
        Queue::fake([
            SyncBackorderQueueCoveragesJob::class,
            UpdateProductsInventoryAndAvgCost::class,
            DeleteProductInventoryJob::class,
        ]);

        Sanctum::actingAs(User::factory()->create());

        $warehouseA = Warehouse::factory()->create()->withDefaultLocation();

        $product1 = Product::factory()->create();

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'sales_channel_line_id' => '12345',
                    'quantity' => 1,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_id' => $warehouseA->id,
                ],
            ],
        ];

        $response = $this->postJson('/api/sales-orders', $payload);
        $data = json_decode($response->getContent(), true)['data'];

        $payload = [
            'custom_field_values' => [
                [
                    'custom_field_id' => CustomField::factory()->create([
                        'link_type' => SalesOrder::class,
                    ])->id,
                    'value' => 'customvalue',
                ],
            ],
        ];

        $response = $this->putJson('/api/sales-orders/'.$data['id'], $payload);

        $data = json_decode($response->getContent(), true)['data'];

        $this->assertEquals('customvalue', $data['custom_field_values'][0]['value']);
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_update_a_sales_order(): void
    {
        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines(3)
            ->create();

        $manager = app(SalesOrderManager::class);

        $manager->updateOrder(UpdateSalesOrderData::from([
            'salesOrder' => $salesOrder,
            'payload' => UpdateSalesOrderPayloadData::from([
                'sales_order_lines' => SalesOrderLineData::collection([
                    SalesOrderLineData::from([
                        'id' => $salesOrder->salesOrderLines->first()->id,
                        'to_delete' => true,
                    ]),
                ]),
            ]),
        ]));

        $this->assertDatabaseCount((new SalesOrderLine())->getTable(), 2);
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_change_sales_order_statuses_when_needed(): void
    {

        /*
        |--------------------------------------------------------------------------
        | Setup
        |--------------------------------------------------------------------------
        */

        $product = Product::factory()->create();
        $warehouse = Warehouse::first();

        $product->setInitialInventory($warehouse->id, 100, 5.00);

        $salesOrder = SalesOrder::factory()
            ->has(SalesOrderLine::factory(1, [
                'product_id' => $product->id,
                'quantity' => 1,
                'warehouse_id' => $warehouse->id,
            ]))
            ->create();

        $salesOrderLine = $salesOrder->salesOrderLines->first();

        $payload = [
            'warehouse_id' => Warehouse::first()->id,
            'fulfilled_at' => now(),
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'requested_shipping_method' => 'abc',
        ];

        // add lines that not fulfilled yet
        $payload['fulfillment_lines'] = $salesOrder->salesOrderLines->map(function (SalesOrderLine $salesOrderLine) {
            return [
                'sales_order_line_id' => $salesOrderLine->id,
                'quantity' => $salesOrderLine->unfulfilled_quantity,
                'sales_channel_line_id' => $salesOrderLine->sales_channel_line_id,
            ];
        })->toArray();

        app(FulfillmentManager::class)->fulfill($salesOrder, $payload, false, false);

        $salesOrder->update([
            'order_status' => SalesOrder::STATUS_OPEN,
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_UNFULFILLED,
        ]);
        
        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'id' => $salesOrder->id,
            'order_status' => SalesOrder::STATUS_OPEN,
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_UNFULFILLED,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Actions
        |--------------------------------------------------------------------------
        */

        $manager = app(SalesOrderManager::class);
        $manager->updateOrder(UpdateSalesOrderData::from([
            'salesOrder' => $salesOrder,
            'payload' => UpdateSalesOrderPayloadData::from([
                'sales_order_lines' => SalesOrderLineData::collection([
                    SalesOrderLineData::from([
                        'id' => $salesOrderLine->id,
                        'quantity' => 1,
                    ]),
                ]),
            ]),
        ]));

        /*
        |--------------------------------------------------------------------------
        | Assertions
        |--------------------------------------------------------------------------
        */

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'id' => $salesOrder->id,
            'order_status' => SalesOrder::STATUS_CLOSED,
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_FULFILLED,
        ]);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_it_can_update_a_sales_order_line_that_was_split(): void
    {
        $product = Product::factory()->create();
        $warehouse1 = Warehouse::factory()->create();
        $warehouse2 = Warehouse::factory()->create();
        $warehouse3 = Warehouse::factory()->create();

        $salesOrder = SalesOrder::factory()
            ->has(
                SalesOrderLine::factory()
                    ->count(3)
                    ->state(new Sequence(
                        [
                            'product_id' => $product->id,
                            'sales_channel_line_id' => '123',
                            'product_listing_id' => null,
                            'quantity' => 2,
                            'warehouse_id' => $warehouse1->id,
                        ],
                        [
                            'product_id' => $product->id,
                            'sales_channel_line_id' => '123',
                            'product_listing_id' => null,
                            'quantity' => 3,
                            'warehouse_id' => $warehouse2->id,
                        ],
                        [
                            'product_id' => $product->id,
                            'sales_channel_line_id' => '123',
                            'product_listing_id' => null,
                            'quantity' => 5,
                            'warehouse_id' => $warehouse3->id,
                        ],
                    ))
            )->create();

        $salesOrderLine1 = $salesOrder->salesOrderLines->first();
        $salesOrderLine2 = $salesOrder->salesOrderLines->get(1);
        $salesOrderLine3 = $salesOrder->salesOrderLines->get(2);

        $salesOrderLine2->split_from_line_id = $salesOrderLine1->id;
        $salesOrderLine2->save();

        $salesOrderLine3->split_from_line_id = $salesOrderLine1->id;
        $salesOrderLine3->save();

        app(SalesOrderManager::class)->updateOrder(UpdateSalesOrderData::from([
            'salesOrder' => $salesOrder,
            'payload' => UpdateSalesOrderPayloadData::from([
                'sales_order_lines' => SalesOrderLineData::collection([
                    SalesOrderLineData::from([
                        'product_id' => $product->id,
                        'sales_channel_line_id' => '123',
                        'quantity' => 1,
                    ]),
                ]),
            ]),
        ]));

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'id' => $salesOrderLine1->id,
            'quantity' => 1,
        ]);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'id' => $salesOrderLine2->id,
            'quantity' => 3,
        ]);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'id' => $salesOrderLine3->id,
            'quantity' => 5,
        ]);
    }

    /**
     * @throws Throwable
     * @throws ProductBundleException
     */
    public function test_it_can_update_a_fulfilled_sales_order_without_getting_out_of_sync(): void
    {
        /*
        |--------------------------------------------------------------------------
        | Set up bundled fulfilled sales order
        |--------------------------------------------------------------------------
        */

        $bundleProduct = Product::factory()->create([
            'type' => Product::TYPE_BUNDLE,
        ]);
        $component1 = Product::factory()->create();
        $component2 = Product::factory()->create();
        $bundleProduct->setBundleComponents([
            [
                'id' => $component1->id,
                'quantity' => 2,
            ],
            [
                'id' => $component2->id,
                'quantity' => 3,
            ],
        ]);
        $warehouse = Warehouse::factory()->create();
        $salesOrder = SalesOrder::factory()
            ->has(
                SalesOrderLine::factory()
                    ->count(2)
                    ->state(new Sequence(
                        [
                            'product_id' => $component1->id,
                            'bundle_id' => $bundleProduct->id,
                            'bundle_quantity_cache' => 1,
                            'sales_channel_line_id' => '123',
                            'product_listing_id' => null,
                            'quantity' => 2,
                            'fulfilled_quantity' => 2,
                            'warehouse_id' => $warehouse->id,
                        ],
                        [
                            'product_id' => $component2->id,
                            'bundle_id' => $bundleProduct->id,
                            'bundle_quantity_cache' => 1,
                            'sales_channel_line_id' => '123',
                            'product_listing_id' => null,
                            'quantity' => 3,
                            'fulfilled_quantity' => 3,
                            'warehouse_id' => $warehouse->id,
                        ],
                    ))
            )->create(
                [
                    'order_status' => SalesOrder::STATUS_OPEN,
                    'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_AWAITING_TRACKING,
                ]
            );

        SalesOrderFulfillment::factory()
            ->has(
                SalesOrderFulfillmentLine::factory()
                    ->count(2)
                    ->state(new Sequence(
                        [
                            'sales_order_line_id' => $salesOrder->salesOrderLines->first()->id,
                            'quantity' => 2,
                        ],
                        [
                            'sales_order_line_id' => $salesOrder->salesOrderLines->offsetGet(1)->id,
                            'quantity' => 3,
                        ],
                    )))
            ->create([
                'sales_order_id' => $salesOrder->id,
                'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
                'status' => SalesOrderFulfillment::STATUS_SUBMITTED,
                'packing_slip_printed_at' => null,
                'fulfilled_at' => now(),
            ]);

        app(SalesOrderManager::class)->updateOrder(UpdateSalesOrderData::from([
            'salesOrder' => $salesOrder,
            'payload' => UpdateSalesOrderPayloadData::from([
                'sales_order_lines' => SalesOrderLineData::collection([
                    SalesOrderLineData::from([
                        'product_id' => $bundleProduct->id,
                        'sales_channel_line_id' => '123',
                        'quantity' => 1,
                    ]),
                ]),
            ]),
        ]));

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'id' => $salesOrder->id,
            'order_status' => SalesOrder::STATUS_OPEN,
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_AWAITING_TRACKING,
        ]);
        $this->assertDatabaseCount((new SalesOrderFulfillment())->getTable(),1);

        /*
        |--------------------------------------------------------------------------
        | Change shipping label
        |--------------------------------------------------------------------------
        */

        $shippingAddress = $salesOrder->shippingAddress;

        $shippingAddress->label = 'new label';

        app(SalesOrderManager::class)->updateOrder(UpdateSalesOrderData::from([
            'salesOrder' => $salesOrder,
            'payload' => UpdateSalesOrderPayloadData::from([
                'shipping_address' => AddressData::from($shippingAddress),
            ]),
        ]));

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'id' => $salesOrder->id,
            'order_status' => SalesOrder::STATUS_OPEN,
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_AWAITING_TRACKING,
        ]);
        $this->assertDatabaseCount((new SalesOrderFulfillment())->getTable(),1);
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_update_tax_total_on_sales_order_line_update(): void
    {
        $gstFreeTaxRate = TaxRate::factory()->create([
            'name' => 'GST Free',
            'rate' => 0,
        ]);

        $gstTaxRate = TaxRate::factory()->create([
            'name' => 'GST 10',
            'rate' => 10,
        ]);

        $gst5 = TaxRate::factory()->create([
            'name' => 'GST 5',
            'rate' => 5,
        ]);

        $salesOrder = SalesOrder::factory()
            ->has(SalesOrderLine::factory()->count(1)->state(new Sequence(
                [
                    'quantity' => 1,
                    'amount' => 100,
                    'tax_rate_id' => $gstFreeTaxRate->id,
                    'fulfilled_quantity' => 1,
                ],
            )))
            ->create([
                'is_tax_included' => false,
                'order_status' => SalesOrder::STATUS_CLOSED,
                'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_FULFILLED,
            ]);

        $salesOrderLine = $salesOrder->salesOrderLines->first();

        SalesOrderFulfillment::factory()
            ->has(
                SalesOrderFulfillmentLine::factory()
                    ->count(1)
                    ->state(new Sequence(
                        [
                            'sales_order_line_id' => $salesOrderLine->id,
                            'quantity' => 1,
                        ],
                    )))
            ->create([
                'sales_order_id' => $salesOrder->id,
                'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
                'status' => SalesOrderFulfillment::STATUS_FULFILLED,
                'packing_slip_printed_at' => now(),
                'fulfilled_at' => now(),
            ]);

        app(SalesOrderManager::class)->updateOrder(UpdateSalesOrderData::from([
            'salesOrder' => $salesOrder,
            'payload' => UpdateSalesOrderPayloadData::from([
                'sales_order_lines' => SalesOrderLineData::collection([
                    SalesOrderLineData::from([
                        'id' => $salesOrderLine->id,
                        'tax_rate_id' => $gstTaxRate->id,
                        'quantity' => 1,
                    ]),
                ]),
            ]),
        ]));

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'id' => $salesOrder->id,
            'tax_total' => 10,
            'order_status' => SalesOrder::STATUS_CLOSED,
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_FULFILLED,
        ]);

        $salesOrderLine->tax_rate_id = $gst5->id;
        $salesOrderLine->save();

        app(SalesOrderManager::class)->updateOrder(UpdateSalesOrderData::from([
            'salesOrder' => $salesOrder,
            'payload' => UpdateSalesOrderPayloadData::from([
                'sales_order_lines' => SalesOrderLineData::collection([
                    SalesOrderLineData::from([
                        'id' => $salesOrderLine->id,
                        'quantity' => 1,
                    ]),
                ]),
            ]),
        ]));

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'id' => $salesOrder->id,
            'tax_total' => 5,
            'order_status' => SalesOrder::STATUS_CLOSED,
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_FULFILLED,
        ]);

        app(SalesOrderManager::class)->updateOrder(UpdateSalesOrderData::from([
            'salesOrder' => $salesOrder,
            'payload' => UpdateSalesOrderPayloadData::from([
                'sales_order_lines' => SalesOrderLineData::collection([
                    SalesOrderLineData::from([
                        'id' => $salesOrderLine->id,
                        'quantity' => 1,
                        'tax_rate_id' => $gstFreeTaxRate->id,
                    ]),
                ]),
            ]),
        ]));

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'id' => $salesOrder->id,
            'tax_total' => 0,
        ]);
    }

    /**
     * @throws Throwable
     * @throws ProductBundleException
     */
    public function test_it_can_update_a_cancelled_sales_order_with_bundles(): void
    {
        Queue::fake();

        /*
        |--------------------------------------------------------------------------
        | Set up bundled sales order
        |--------------------------------------------------------------------------
        */

        $bundleProduct = Product::factory()->create([
            'type' => Product::TYPE_BUNDLE,
        ]);
        $component1    = Product::factory()->create();
        $bundleProduct->setBundleComponents([
            [
                'id' => $component1->id,
                'quantity' => 2,
            ],
        ]);
        $warehouse  = Warehouse::factory()->create();
        $salesOrder = SalesOrder::factory()
            ->has(
                SalesOrderLine::factory()
                    ->count(1)
                    ->state(new Sequence(
                        [
                            'product_id' => $component1->id,
                            'bundle_id' => $bundleProduct->id,
                            'bundle_quantity_cache' => 1,
                            'bundle_component_quantity_cache' => 2,
                            'sales_channel_line_id' => '123',
                            'product_listing_id' => null,
                            'quantity' => 2,
                            'warehouse_id' => $warehouse->id,
                        ],
                    ))
            )->create(
                [
                    'order_status' => SalesOrder::STATUS_OPEN,
                    'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_UNFULFILLED,
                ]
            );

        $canceledAt = now();

        app(SalesOrderManager::class)->updateOrder(UpdateSalesOrderData::from([
            'salesOrder' => $salesOrder,
            'payload' => UpdateSalesOrderPayloadData::from([
                'sales_order_lines' => SalesOrderLineData::collection([
                    SalesOrderLineData::from([
                        'product_id' => $bundleProduct->id,
                        'sales_channel_line_id' => '123',
                        'quantity' => 1,
                        'quantity_to_cancel' => 1,
                    ]),
                ]),
                'on_hold' => true,
                'canceled_at' => $canceledAt->clone()->toDateTimeString(),
            ]),
        ]));

        Queue::assertNotPushed(AutomatedSalesOrderFulfillmentJob::class);
        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'id' => $salesOrder->id,
            'order_status' => SalesOrder::STATUS_CLOSED,
            'canceled_at' => $canceledAt->clone()->toDateTimeString(),
        ]);
        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'quantity' => 0,
            'canceled_quantity' => 2,
        ]);
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_add_product_while_removing_existing_product_from_sales_order(): void
    {
        SalesOrder::factory()
            ->has(SalesOrderLine::factory()->count(1)->state(new Sequence(
                [
                    'quantity' => 1,
                ],
            )))
            ->create();

        $manager = app(SalesOrderManager::class);

        $manager->updateOrder(UpdateSalesOrderData::from([
            'salesOrder' => SalesOrder::first(),
            'payload' => UpdateSalesOrderPayloadData::from([
                'sales_order_lines' => SalesOrderLineData::collection([
                    SalesOrderLineData::from([
                        'id' => SalesOrderLine::first()->id,
                        'to_delete' => true,
                    ]),
                    SalesOrderLineData::from([
                        'product_id' => Product::factory()->create()->id,
                        'quantity' => 1,
                        'description' => 'new product',
                    ]),
                ]),
            ]),
        ]));

        $this->assertDatabaseCount((new SalesOrderLine())->getTable(), 1);
    }

    /**
     * @throws Throwable
     * @throws InvalidProductWarehouseRouting
     */
    public function test_unpaid_orders_stay_reserved_after_update(): void
    {
        $setting = Setting::where('key',Setting::KEY_SO_UNPAID_ORDERS_AS_RESERVED)->first();
        $setting->value = true;
        $setting->save();

        $manager = app(SalesOrderManager::class);

        $salesOrder = $manager->createOrder([
            'sales_order_number' => 'SO-TEST',
            'store_id' => Store::first()->id,
            'sales_channel_id' => SalesChannel::first()->id,
            'order_status' => SalesOrder::STATUS_OPEN,
            'payment_status' => SalesOrder::PAYMENT_STATUS_UNPAID,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => Product::factory()->create()->id,
                    'sales_channel_line_id' => '12345',
                    'quantity' => 1,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ]);

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'order_status' => SalesOrder::STATUS_RESERVED,
        ]);

        $manager->updateOrder(UpdateSalesOrderData::from([
            'salesOrder' => $salesOrder,
            'payload' => UpdateSalesOrderPayloadData::from([
                'order_status' => SalesOrder::STATUS_OPEN,
            ]),
        ]));

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'order_status' => SalesOrder::STATUS_RESERVED,
        ]);
    }
}
