<?php

namespace Tests\Feature\SalesOrders;

use App\Exceptions\InsufficientStockException;
use App\Helpers;
use App\Models\InventoryMovement;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\Warehouse;
use App\Services\InventoryManagement\InventoryDeficiencyActionType;
use App\Services\InventoryManagement\InventoryManager;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use App\Services\SalesOrder\SalesOrderManager;
use Carbon\CarbonImmutable;
use Exception;
use Illuminate\Foundation\Testing\DatabaseTransactions;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Queue;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Tests\TestCaseWithoutTransactions;
use Throwable;

class SyncExternallyFulfilledTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_it_can_sync_externally_fulfilled_to_delete_movements(): void
    {
        Queue::fake();
        /*
         * Fully fulfill sales order
         * Change setting for inventory start date
         * Run SyncExternallyFulfilled
         * Make assertions
         */

        Setting::with([])->updateOrCreate(
            ['key' => Setting::KEY_INVENTORY_START_DATE],
            [
                'description' => 'The start date of inventory count.',
                'type' => Setting::TYPE_TIME,
                'value' => '2022-01-01 00:00:00',
                'default_value' => null,
            ]
        );
        // Clear settings cache
        Setting::$fetchedSettings = null;

        $externallyFulfilledDate = CarbonImmutable::parse('2022-01-05 00:00:00');
        $warehouseId = Warehouse::factory()->create()->id;

        /** @var SalesOrder $salesOrder */
        $salesOrder = SalesOrder::factory()->has(
            SalesOrderLine::factory(2)->state([
                'warehouse_id' => $warehouseId,
                'quantity' => 3,
            ])
        )->create([
            'order_date' => $externallyFulfilledDate,
        ])->first();

        $salesOrder->salesOrderLines->each(/**
         * @throws InsufficientStockException
         */ function (SalesOrderLine $salesOrderLine) {
            $inventoryManager = new InventoryManager(
                $salesOrderLine->warehouse_id,
                $salesOrderLine->product,
            );
            $inventoryManager->takeFromStock($salesOrderLine->quantity, $salesOrderLine, true, null, InventoryDeficiencyActionType::ACTION_TYPE_POSITIVE_ADJUSTMENT);
        });

        app(FulfillmentManager::class)->fullyFulfill($salesOrder, $externallyFulfilledDate);
        $this->assertEquals(8, InventoryMovement::query()->count());

        Setting::with([])->updateOrCreate(
            ['key' => Setting::KEY_INVENTORY_START_DATE],
            [
                'description' => 'The start date of inventory count.',
                'type' => Setting::TYPE_TIME,
                'value' => '2023-01-01 00:00:00',
                'default_value' => null,
            ]
        );

        // Clear settings cache
        Setting::$fetchedSettings = null;

        $this->assertEquals('2023-01-01 00:00:00', Helpers::setting(Setting::KEY_INVENTORY_START_DATE));

        app(SalesOrderManager::class)->syncExternallyFulfilled();

        // Assert that the only inventory movement remaining is the adjustment
        $this->assertEquals(2, InventoryMovement::query()->count());
        $this->assertEquals(InventoryMovement::TYPE_ADJUSTMENT, InventoryMovement::query()->first()->type);

        // Assert that the externally fulfilled quantity got updated
        SalesOrderLine::all()->each(function (SalesOrderLine $salesOrderLine) {
            $this->assertEquals($salesOrderLine->quantity, $salesOrderLine->externally_fulfilled_quantity);
        });
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_it_can_split_a_sales_order_line_that_is_partially_externally_fulfilled(): void
    {
        Queue::fake();
        /*
         * Fully fulfill sales order
         * Change setting for inventory start date
         * Run SyncExternallyFulfilled
         * Make assertions
         */

        Setting::with([])->updateOrCreate(
            ['key' => Setting::KEY_INVENTORY_START_DATE],
            [
                'description' => 'The start date of inventory count.',
                'type' => Setting::TYPE_TIME,
                'value' => '2022-01-01 00:00:00',
                'default_value' => null,
            ]
        );
        // Clear settings cache
        Setting::$fetchedSettings = null;

        $externallyFulfilledDate = CarbonImmutable::parse('2022-01-05 00:00:00');
        $warehouseId = Warehouse::factory()->create()->id;

        /** @var SalesOrder $salesOrder */
        $salesOrder = SalesOrder::factory()->has(
            SalesOrderLine::factory(2)->state([
                'warehouse_id' => $warehouseId,
                'quantity' => 3,
            ])
        )->create([
            'order_date' => $externallyFulfilledDate,
        ])->first();

        $salesOrder->salesOrderLines->each(/**
         * @throws InsufficientStockException
         */ function (SalesOrderLine $salesOrderLine) {
            $inventoryManager = new InventoryManager(
                $salesOrderLine->warehouse_id,
                $salesOrderLine->product,
            );
            $inventoryManager->takeFromStock($salesOrderLine->quantity, $salesOrderLine, true, null, InventoryDeficiencyActionType::ACTION_TYPE_POSITIVE_ADJUSTMENT);
        });

        $splitUnfulfilledSalesOrderLine = $salesOrder->salesOrderLines->first();

        $fulfillmentPayload = [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'warehouse_id' => $splitUnfulfilledSalesOrderLine->warehouse_id,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $splitUnfulfilledSalesOrderLine->id,
                    'quantity' => $splitUnfulfilledSalesOrderLine->quantity - 1,
                ],
            ],
            'fulfilled_at' => $externallyFulfilledDate->toDateTimeString(),
        ];
        app(FulfillmentManager::class)->fulfill($salesOrder, $fulfillmentPayload, false, false);
        $this->assertEquals(7, InventoryMovement::query()->count());

        Setting::with([])->updateOrCreate(
            ['key' => Setting::KEY_INVENTORY_START_DATE],
            [
                'description' => 'The start date of inventory count.',
                'type' => Setting::TYPE_TIME,
                'value' => '2023-01-01 00:00:00',
                'default_value' => null,
            ]
        );

        // Clear settings cache
        Setting::$fetchedSettings = null;

        $this->assertEquals('2023-01-01 00:00:00', Helpers::setting(Setting::KEY_INVENTORY_START_DATE));

        app(SalesOrderManager::class)->syncExternallyFulfilled();

        /** @var SalesOrderLine $splitExternallyFulfilledSalesOrderLine */
        $splitExternallyFulfilledSalesOrderLine = SalesOrderLine::query()->where('split_from_line_id', $splitUnfulfilledSalesOrderLine->id)->first();

        // Assert that the only inventory movement remaining is the adjustment
        $this->assertEquals(2, InventoryMovement::query()->count());
        $this->assertEquals(InventoryMovement::TYPE_ADJUSTMENT, InventoryMovement::query()->first()->type);

        // Assert that the split happened as expected
        $this->assertEquals(0, $splitUnfulfilledSalesOrderLine->externally_fulfilled_quantity);
        $this->assertEquals($splitUnfulfilledSalesOrderLine->quantity - 1, $splitExternallyFulfilledSalesOrderLine->externally_fulfilled_quantity);
    }
}
