<?php

namespace Tests\Feature\SalesOrders;

use App\Exceptions\DropshipWithOpenOrdersException;
use App\Exceptions\SalesOrder\InvalidProductWarehouseRouting;
use App\Http\Requests\StoreInventoryAdjustment;
use App\Jobs\DeleteProductInventoryJob;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\BackorderQueue;
use App\Models\BackorderQueueCoverage;
use App\Models\Customer;
use App\Models\FinancialLine;
use App\Models\FinancialLineType;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Models\User;
use App\Models\Warehouse;
use App\Repositories\SettingRepository;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use App\Services\SalesOrder\SalesOrderManager;
use App\Services\SalesOrder\WarehouseRoutingMethod;
use App\Services\StockTake\OpenStockTakeException;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Queue;
use Illuminate\Testing\Fluent\AssertableJson;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\Concerns\UseSimpleSkuFactories;
use Tests\TestCase;
use Throwable;

class CreateSalesOrderTest extends TestCase
{
    use FastRefreshDatabase;
    use UseSimpleSkuFactories;
    use WithFaker;


    public function test_it_wont_reserve_inventory_for_externally_fulfilled_quantity(): void
    {
        Queue::fake([
            SyncBackorderQueueCoveragesJob::class,
            UpdateProductsInventoryAndAvgCost::class,
            DeleteProductInventoryJob::class,
        ]);

        Sanctum::actingAs(User::factory()->create());

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => Product::factory()->create()->id,
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                    'externally_fulfilled_quantity' => 2,
                ],
            ],
        ];

        // Perform request.
        $response = $this->postJson('/api/sales-orders', $payload);

        // Should be successful.
        $response->assertSuccessful();

        $this->assertDatabaseCount('sales_order_lines', 2);
        $this->assertDatabaseHas('sales_order_lines', [
            'quantity' => 3,
            'externally_fulfilled_quantity' => 0,
        ]);
        $this->assertDatabaseHas('sales_order_lines', [
            'quantity' => 2,
            'externally_fulfilled_quantity' => 2,
        ]);

        $data = json_decode($response->getContent(), true)['data'];

        // Inventory movements should happen for non-externally fulfilled order
        $this->assertDatabaseCount('inventory_movements', 2);
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $payload['sales_order_lines'][0]['product_id'],
            'quantity' => 3,
            'link_type' => SalesOrderLine::class,
            'link_id' => $data['item_info'][0]['sales_order_line_id'],
            'layer_type' => BackorderQueue::class,
        ]);

        // Backorder queues should be created for non-externally fulfilled quantity.
        $this->assertDatabaseCount('backorder_queues', 1);
        $this->assertDatabaseHas('backorder_queues', [
            'sales_order_line_id' => $data['item_info'][0]['sales_order_line_id'],
            'backordered_quantity' => 3,
            'released_quantity' => 0,
            'priority' => 1,
        ]);
    }

    public function test_it_can_create_dropship_sales_order_when_other_lines_are_not_dropship(): void
    {
        Queue::fake([
            SyncBackorderQueueCoveragesJob::class,
            UpdateProductsInventoryAndAvgCost::class,
            DeleteProductInventoryJob::class,
        ]);

        Sanctum::actingAs(User::factory()->create());

        /** @var Supplier $supplier */
        $supplier = Supplier::factory()->withWarehouse()->create([
            'auto_fulfill_dropship' => true,
        ]);

        Warehouse::factory()->create([
            'dropship_enabled' => true,
            'supplier_id' => $supplier->id,
        ])->withDefaultLocation();

        /** @var Product $product */
        $product = Product::factory()->create([
            'is_dropshippable' => true,
        ]);

        SupplierProduct::factory()->create([
            'product_id' => $product->id,
            'supplier_id' => $supplier->id,
        ]);

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 2,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::DROPSHIP->value,
                ],
                [
                    'product_id' => $product->id,
                    'quantity' => 2,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        // Perform request.
        $response = $this->postJson('/api/sales-orders', $payload);

        // Should be successful.
        $response->assertSuccessful();

        $data = json_decode($response->getContent(), true)['data'];

        // It should create a dropship purchase order
        $this->assertDatabaseCount('purchase_orders', 1);
        $this->assertDatabaseHas('purchase_orders', [
            'receipt_status' => PurchaseOrder::RECEIPT_STATUS_DROPSHIP,
            'sales_order_id' => $data['id'],
        ]);
        $this->assertDatabaseCount('purchase_order_lines', 1);
        $this->assertDatabaseHas('purchase_order_lines', [
            'quantity' => $payload['sales_order_lines'][0]['quantity'],
        ]);

        // Order and lines should exist in database.
        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseHas('sales_orders', [
            'sales_order_number' => $payload['sales_order_number'],
        ]);
        $this->assertDatabaseCount('sales_order_lines', 2);
        $this->assertDatabaseHas('sales_order_lines', $payload['sales_order_lines'][0]);
        $this->assertDatabaseHas('sales_order_lines', $payload['sales_order_lines'][1]);

        // Inventory movements should be created for the non-dropship line
        $this->assertDatabaseCount('inventory_movements', 2);

        // Backorder queues should be created for the non-dropship lines.
        $this->assertDatabaseCount('backorder_queues', 1);
    }

    public function test_it_can_create_dropship_sales_order(): void
    {
        Queue::fake([
            SyncBackorderQueueCoveragesJob::class,
            UpdateProductsInventoryAndAvgCost::class,
            DeleteProductInventoryJob::class,
        ]);

        Sanctum::actingAs(User::factory()->create());

        /** @var Supplier $supplier */
        $supplier = Supplier::factory()->withWarehouse()->create([
            'auto_fulfill_dropship' => true,
        ]);

        Warehouse::factory()->create([
            'dropship_enabled' => true,
            'supplier_id' => $supplier->id,
        ])->withDefaultLocation();

        /** @var Product $product */
        $product = Product::factory()->create([
            'is_dropshippable' => true,
        ]);

        SupplierProduct::factory()->create([
            'product_id' => $product->id,
            'supplier_id' => $supplier->id,
        ]);

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 2,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::DROPSHIP->value,
                ],
            ],
        ];

        // Perform request.
        $response = $this->postJson('/api/sales-orders', $payload);

        // Should be successful.
        $response->assertSuccessful();

        $data = json_decode($response->getContent(), true)['data'];

        // It should create a dropship purchase order
        $this->assertDatabaseCount('purchase_orders', 1);
        $this->assertDatabaseHas('purchase_orders', [
            'receipt_status' => PurchaseOrder::RECEIPT_STATUS_DROPSHIP,
            'sales_order_id' => $data['id'],
        ]);
        $this->assertDatabaseCount('purchase_order_lines', 1);
        $this->assertDatabaseHas('purchase_order_lines', [
            'quantity' => $payload['sales_order_lines'][0]['quantity'],
        ]);

        // Order and lines should exist in database.
        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseHas('sales_orders', [
            'sales_order_number' => $payload['sales_order_number'],
        ]);
        $this->assertDatabaseCount('sales_order_lines', 1);
        $this->assertDatabaseHas('sales_order_lines', $payload['sales_order_lines'][0]);

        // Inventory movements should not happen for dropship lines
        $this->assertDatabaseCount('inventory_movements', 0);

        // Backorder queues should not be created for dropship lines.
        $this->assertDatabaseCount('backorder_queues', 0);
    }

    public function test_warehouse_routing_method_of_warehouse_must_have_a_warehouse(): void
    {
        Queue::fake([
            SyncBackorderQueueCoveragesJob::class,
            UpdateProductsInventoryAndAvgCost::class,
            DeleteProductInventoryJob::class,
        ]);

        Sanctum::actingAs(User::factory()->create());

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => Product::factory()->create()->id,
                    'quantity' => 2,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::WAREHOUSE->value,
                ],
            ],
        ];

        // Perform request.
        $response = $this->postJson('/api/sales-orders', $payload);

        // Should be successful.
        $response->assertStatus(422);

        // Order shouldn't be created
        $this->assertDatabaseCount('sales_orders', 0);
    }

    public function test_it_wont_approve_draft_orders(): void
    {
        Queue::fake([
            SyncBackorderQueueCoveragesJob::class,
            UpdateProductsInventoryAndAvgCost::class,
            DeleteProductInventoryJob::class,
        ]);

        Sanctum::actingAs(User::factory()->create());

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_DRAFT,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => Product::factory()->create()->id,
                    'quantity' => 2,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        // Perform request.
        $response = $this->postJson('/api/sales-orders', $payload);

        // Should be successful.
        $response->assertSuccessful();

        // No Inventory movements should happen for draft order
        $this->assertDatabaseCount('inventory_movements', 0);

        // No backorder queues should be created for draft order.
        $this->assertDatabaseCount('backorder_queues', 0);
    }

    public function test_it_can_create_sales_order(): void
    {
        Queue::fake([
            SyncBackorderQueueCoveragesJob::class,
            UpdateProductsInventoryAndAvgCost::class,
            DeleteProductInventoryJob::class,
        ]);

        Sanctum::actingAs(User::factory()->create());

        /** @var FinancialLineType $financialLineType */
        $financialLineType = FinancialLineType::factory()->create();

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => Product::factory()->create()->id,
                    'quantity' => 2,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
                [
                    'product_id' => Product::factory()->create()->id,
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
            'financial_lines' => [
                [
                    'financial_line_type_id' => $financialLineType->id,
                    'description' => 'ABC',
                    'quantity' => 1,
                    'amount' => 4.00,
                    'tax_allocation' => 0.00,
                    'allocate_to_products' => true,
                    'proration_strategy' => 'revenue_based',
                    'allocate_to_id' => null,
                    'allocate_to_type' => null,
                ],
            ],
        ];

        // Perform request.
        $response = $this->postJson('/api/sales-orders', $payload);

        // Should be successful.
        $response->assertSuccessful();

        // Should have needed json payload.
        $response->assertJson(fn (AssertableJson $json) => $json->has('data.sales_order_number')->etc());

        $this->assertDatabaseHas((new FinancialLine())->getTable(), [
            'financial_line_type_id' => $financialLineType->id,
            'description' => 'ABC',
            'quantity' => 1,
            'amount' => 4.00,
            'tax_allocation' => 0.00,
            'allocate_to_products' => true,
            'proration_strategy' => 'revenue_based',
            'allocate_to_id' => null,
            'allocate_to_type' => null,
        ]);

        // Order and lines should exist in database.
        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseHas('sales_orders', [
            'sales_order_number' => $payload['sales_order_number'],
        ]);
        $this->assertDatabaseCount('sales_order_lines', 2);
        $this->assertDatabaseHas('sales_order_lines', $payload['sales_order_lines'][0]);
        $this->assertDatabaseHas('sales_order_lines', $payload['sales_order_lines'][1]);

        $data = json_decode($response->getContent(), true)['data'];

        // Inventory movements should happen for backordered lines
        $this->assertDatabaseCount('inventory_movements', 4); // 2 active, 2 reserved
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $payload['sales_order_lines'][0]['product_id'],
            'quantity' => $payload['sales_order_lines'][0]['quantity'],
            'link_type' => SalesOrderLine::class,
            'link_id' => $data['item_info'][0]['sales_order_line_id'],
            'layer_type' => BackorderQueue::class,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $payload['sales_order_lines'][1]['product_id'],
            'quantity' => $payload['sales_order_lines'][1]['quantity'],
            'link_type' => SalesOrderLine::class,
            'link_id' => $data['item_info'][1]['sales_order_line_id'],
            'layer_type' => BackorderQueue::class,
        ]);

        // Backorder queues should be created.
        $this->assertDatabaseCount('backorder_queues', 2);
        $this->assertDatabaseHas('backorder_queues', [
            'sales_order_line_id' => $data['item_info'][0]['sales_order_line_id'],
            'backordered_quantity' => $payload['sales_order_lines'][0]['quantity'],
            'released_quantity' => 0,
            'priority' => 1,
        ]);
        $this->assertDatabaseHas('backorder_queues', [
            'sales_order_line_id' => $data['item_info'][1]['sales_order_line_id'],
            'backordered_quantity' => $payload['sales_order_lines'][1]['quantity'],
            'released_quantity' => 0,
            'priority' => 1,
        ]);
    }

    /**
     * Lines that are externally fulfilled (Set by sales channel based on fulfillment date < inventory start date):
     * - Should have no warehouse assigned to the lines
     * Orders that are 100% externally fulfilled:
     * - Should have an order status of closed
     * - Should have a fulfillment status of fulfilled
     *
     * @throws OpenStockTakeException
     * @throws InvalidProductWarehouseRouting
     * @throws BindingResolutionException
     * @throws Throwable
     */
    public function test_it_will_handle_orders_externally_fulfilled(): void
    {
        Queue::fake();

        /** @var SalesChannel $salesChannel */
        $salesChannel = SalesChannel::factory()->create();

        /** @var Product $product */
        $product = Product::factory()->create();

        app(SalesOrderManager::class)->createOrder([
            'sales_order_number' => 'Order name',
            'order_status' => SalesOrder::STATUS_OPEN, // This is needed due to SalesOrderManager::requiresApproval() logic
            'sales_channel_id' => $salesChannel->id,
            'order_date' => now()->subDays(2),
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => 'Description',
                    'is_product' => true,
                    'quantity' => 5,
                    'price' => 10,
                    'externally_fulfilled_quantity' => 5,
                ],
            ],
        ]);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'warehouse_id' => null,
            'externally_fulfilled_quantity' => 5,
        ]);

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'sales_order_number' => 'Order name',
            'order_status' => SalesOrder::STATUS_CLOSED,
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_FULFILLED,
        ]);
    }

    /**
     * @throws Throwable
     * @throws InvalidProductWarehouseRouting
     */
    public function test_it_will_handle_sales_order_fulfillment_prior_to_inventory_start_date(): void
    {
        Queue::fake();

        /** @var SalesChannel $salesChannel */
        $salesChannel = SalesChannel::factory()->create();

        /** @var Product $product */
        $product = Product::factory()->create();

        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::query()->first();

        $salesOrder = app(SalesOrderManager::class)->createOrder([
            'sales_order_number' => 'Order name',
            'order_status' => SalesOrder::STATUS_OPEN, // This is needed due to SalesOrderManager::requiresApproval() logic
            'sales_channel_id' => $salesChannel->id,
            'order_date' => SettingRepository::getInventoryStartDate()->subDays(2),
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => 'Description',
                    'is_product' => true,
                    'quantity' => 5,
                    'price' => 10,
                ],
            ],
        ]);

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'inventory_movement_date' => SettingRepository::getInventoryStartDate(),
        ]);

        $fulfillmentManager = app(FulfillmentManager::class);

        // Fulfill fully externally

        $fulfillmentData = [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'fulfilled_at' => SettingRepository::getInventoryStartDate()->subDays(2),
            'fulfilled_shipping_method' => 'USPS',
            'tracking_number' => '123',
            'warehouse_id' => $warehouse->id,
        ];

        $salesOrder->salesOrderLines->each(function (SalesOrderLine $salesOrderLine) use (&$fulfillmentData) {
            $fulfillmentData['fulfillment_lines'][] = [
                'sales_order_line_id' => $salesOrderLine->id,
                'quantity' => $salesOrderLine->quantity,
            ];
        });

        Sanctum::actingAs(User::factory()->create());

        //        $this->postJson(route('inventory-adjustments.store'), [
        //            'product_id' => $product->id,
        //            'adjustment_date' => now()->format('Y-m-d'),
        //            'quantity' => 10,
        //            'warehouse_id' => $warehouse->id,
        //            'reason' => 'test',
        //            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        //            'unit_cost' => 5,
        //        ]);
        $salesOrderLine = $salesOrder->salesOrderLines->first();

        $fulfillmentManager->fulfill($salesOrder, $fulfillmentData, false, false);

        $this->assertDatabaseCount((new InventoryMovement())->getTable(), 0);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'id' => $salesOrderLine->id,
            'warehouse_id' => null,
            'externally_fulfilled_quantity' => $salesOrderLine->quantity,
        ]);
    }

    /**
     * @throws Throwable
     * @throws InvalidProductWarehouseRouting
     */
    public function test_it_can_duplicate_a_sales_order()
    {
        $salesOrder = SalesOrder::factory()->hasSalesOrderLines(3)->create();

        app(SalesOrderManager::class)->duplicate($salesOrder);

        $this->assertDatabaseCount((new SalesOrder())->getTable(), 2);
        $this->assertDatabaseCount((new SalesOrderLine())->getTable(), 6);
    }

    /**
     * @throws Throwable
     * @throws InvalidProductWarehouseRouting
     */
    public function test_it_will_handle_partial_sales_order_fulfillment_prior_to_inventory_start_date(): void
    {
        Queue::fake();

        /** @var SalesChannel $salesChannel */
        $salesChannel = SalesChannel::factory()->create();

        /** @var Product $product */
        $product = Product::factory()->create();

        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::query()->first();

        $salesOrder = app(SalesOrderManager::class)->createOrder([
            'sales_order_number' => 'Order name',
            'order_status' => SalesOrder::STATUS_OPEN, // This is needed due to SalesOrderManager::requiresApproval() logic
            'sales_channel_id' => $salesChannel->id,
            'order_date' => SettingRepository::getInventoryStartDate()->subDays(2),
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => 'Description',
                    'is_product' => true,
                    'quantity' => 5,
                    'price' => 10,
                ],
            ],
        ]);

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'inventory_movement_date' => SettingRepository::getInventoryStartDate(),
        ]);

        $fulfillmentManager = app(FulfillmentManager::class);

        // Fulfill partially externally

        $fulfillmentData = [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'fulfilled_at' => SettingRepository::getInventoryStartDate()->subDays(2),
            'fulfilled_shipping_method' => 'USPS',
            'tracking_number' => '123',
            'warehouse_id' => $warehouse->id,
        ];

        $salesOrder->salesOrderLines->each(function (SalesOrderLine $salesOrderLine) use (&$fulfillmentData) {
            $fulfillmentData['fulfillment_lines'][] = [
                'sales_order_line_id' => $salesOrderLine->id,
                'quantity' => 3,
            ];
        });

        Sanctum::actingAs(User::factory()->create());

        $this->postJson(route('inventory-adjustments.store'), [
            'product_id' => $product->id,
            'adjustment_date' => now()->format('Y-m-d'),
            'quantity' => 2,
            'warehouse_id' => $warehouse->id,
            'reason' => 'test',
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'unit_cost' => 5,
        ]);
        $salesOrderLine = $salesOrder->salesOrderLines->first();

        $fulfillmentManager->fulfill($salesOrder, $fulfillmentData, false, false);

        $this->assertDatabaseCount((new InventoryMovement())->getTable(), 3);
        $this->assertDatabaseCount((new SalesOrderFulfillment())->getTable(), 0);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'id' => $salesOrderLine->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 2,
            'externally_fulfilled_quantity' => 0,
        ]);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'warehouse_id' => null,
            'quantity' => 3,
            'externally_fulfilled_quantity' => 3,
        ]);
    }

    /**
     * @throws InvalidProductWarehouseRouting
     * @throws Throwable
     */
    public function test_it_can_calculate_backorder_coverage_for_order_creation(): void
    {
        $this->setUpWarehouse();
        $this->setUpProduct();
        $purchaseOrder = $this->setUpPurchaseOrders(3)->first();

        /** @var SalesOrder $salesOrder */
        $salesOrder = app(SalesOrderManager::class)->createOrder([
            'order_status' => 'open',
            'sales_order_lines' => [
                [
                    'description' => 'description',
                    'amount' => 2.00,
                    'quantity' => 5,
                    'product_id' => $this->product->id,
                    'warehouse_id' => $this->warehouse->id,
                ],
            ],
        ]);

        $this->assertDatabaseHas((new BackorderQueueCoverage())->getTable(), [
            'backorder_queue_id' => $salesOrder->salesOrderLines->first()->backorderQueue->id,
            'purchase_order_line_id' => $purchaseOrder->purchaseOrderLines->first()->id,
            'covered_quantity' => 3,
            'released_quantity' => 0,
        ]);
    }

    /**
     * @throws DropshipWithOpenOrdersException
     * @throws InvalidProductWarehouseRouting
     * @throws Throwable
     */
    public function test_it_wont_reserve_inventory_for_amazon_fba_warehouse(): void
    {
        $this->setUpProduct();
        $this->setUpWarehouse();
        $this->warehouse->type = Warehouse::TYPE_AMAZON_FBA;
        $this->warehouse->save();
        $this->product->setInitialInventory($this->warehouse->id, 10);

        app(SalesOrderManager::class)->createOrder([
            'sales_order_number' => 'Order name',
            'order_status' => SalesOrder::STATUS_OPEN, // This is needed due to SalesOrderManager::requiresApproval() logic
            'sales_channel_id' => SalesChannel::first()->id,
            'order_date' => SettingRepository::getInventoryStartDate()->subDays(2),
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $this->product->id,
                    'warehouse_id' => $this->warehouse->id,
                    'description' => 'Description',
                    'is_product' => true,
                    'quantity' => 5,
                    'price' => 10,
                ],
            ],
        ]);

        // Orders using MFN should not reserve inventory
        $this->assertDatabaseMissing((new InventoryMovement())->getTable(), [
            'link_type' => SalesOrderLine::class,
        ]);
    }

    /**
     * @throws Throwable
     * @throws InvalidProductWarehouseRouting
     */
    public function test_it_ignores_routing_for_warehouses_with_auto_routing_enabled_set_to_false(): void
    {
        $warehouse1 = Warehouse::first();
        $warehouse2 = Warehouse::factory()->create([
            'type' => 'direct',
            'auto_routing_enabled' => false,
        ]);

        $autoSplitSetting = Setting::where('key', 'auto_split_sales_order_line_across_warehouses')->first()->value;
        $this->assertEquals(1, $autoSplitSetting);

        $priorityWarehouses = Setting::where('key', 'warehouse_priority')->first()->value;
        $this->assertEquals("[$warehouse1->id]", $priorityWarehouses);

        $product = Product::factory()->create();
        $product->setInitialInventory($warehouse2->id, 10);

        app(SalesOrderManager::class)->createOrder([
            'sales_order_number' => 'Order name',
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => SettingRepository::getInventoryStartDate()->addDays(2),
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => 'Description',
                    'is_product' => true,
                    'quantity' => 5,
                    'price' => 10,
                ],
            ],
        ]);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'warehouse_id' => $warehouse1->id,
        ]);
    }

    /*
     * TODO: Manually clear pre inventory start date movements from curvy, then change start date to be november 2nd 2021.
     *  Move all dates to minimum november 2nd midnight sydney time.
     * TODO: Test impact of command on existing data *****
     *  Test where no_audit_trail is used for a sales order line and a warehouse is set.  Convert to externally_fulfilled_quantity and no warehouse set
     * TODO Post MVP: Need to test what happens when inventory start date is changed.
     * TODO Post MVP: For the sync fulfillment flow, If a warehouse is selected for fulfillment that is
     *  different from the sales order line, then adjust the sales order line warehouse id as well.  Potentially split if the quantity fulfilled
     *  < sales order line quantity.  Add a warning message to User that "The sales order line warehouse will change to X"
     * TODO Post MVP: In the sync fulfillment flow, Allow user to create a fulfillment for partial quantity, resulting in a split
     * TODO Post MVP: Test if a backordered sales order from shopify that was was fulfilled in shopify (hence out of sync) becomes
     *  In Stock, and, there is only 1 warehouse, then automatically create the fulfillment and change status of order to (Closed/Fulfilled)
     * TODO Post MVP: Manual test, if a sales order has no fulfillable lines due to it being fulfilled by the sales channel, then
     *  Don't show Fulfill button (Right now it shows it but the lines themselves are not fulfillable)
     * TODO: Post MVP: If you can deduce the warehouse through the shopify location -> warehouse relationship (if only 1 warehouse feeds into the location)
     *  then automatically create fulfillment (both on order creation and backorder release).
     */
}
