<?php

namespace Tests\Feature;

use App\Models\Customer;
use App\Models\Product;
use App\Models\SalesCreditLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\TaxRate;
use App\Models\User;
use App\Models\Warehouse;
use App\Services\InventoryManagement\BulkInventoryManager;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\Concerns\UseSimpleSkuFactories;
use Tests\TestCase;

class SalesCreditManagerTest extends TestCase
{
    use FastRefreshDatabase;
    use UseSimpleSkuFactories;
    use WithFaker;

    public function test_it_can_get_unit_cost(): void
    {
        Queue::fake();
        Sanctum::actingAs(User::factory()->create());

        $product = Product::factory()->create();
        $warehouse = Warehouse::factory()->create();

        $product->setInitialInventory($warehouse->id, 2, 5.00);

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_id' => $warehouse->id,
                ],
            ],
        ];

        // Perform request.
        $response = $this->postJson('/api/sales-orders', $payload)->assertSuccessful();

        $salesOrderId = $response->json('data.id');

        $payload = [
            'sales_order_id' => $salesOrderId,
            'notes' => 'damaged',
            'sales_credit_lines' => [
                [
                    'product_id' => $product->id,
                    'sales_order_line_id' => SalesOrderLine::first()->id,
                    'quantity' => 3,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                ],
            ],
        ];

        $this->postJson('/api/sales-credits', $payload)->assertSuccessful();

        $salesCreditLine = SalesCreditLine::first();

        $this->assertEquals(5.00, $salesCreditLine->unit_cost);
    }

    public function test_it_can_set_tax_rate_to_null(): void
    {
        Queue::fake();
        Sanctum::actingAs(User::factory()->create());

        $product = Product::factory()->create();
        /** @var Customer $customer */
        $customer = Customer::factory()->create();

        /** @var TaxRate $taxRate */
        $taxRate = TaxRate::factory()->create();

        $payload = [
            'notes' => 'damaged',
            'customer_id' => $customer->id,
            'sales_credit_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 3,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'tax_rate_id' => $taxRate->id,
                ],
            ],
        ];

        $response = $this->postJson('/api/sales-credits', $payload)->assertSuccessful();
        $this->assertDatabaseHas('sales_credit_lines', [
            'tax_rate_id' => $taxRate->id,
            'id' => $response->json('data.item_info.0.id'),
        ]);
        // Update tax rate to null
        $this->putJson('/api/sales-credits/'.$response->json('data.id'), [
            'sales_credit_lines' => [
                [
                    'id' => $response->json('data.item_info.0.id'),
                    'tax_rate_id' => null,
                ],
            ],
        ])->assertSuccessful();

        $this->assertDatabaseHas('sales_credit_lines', [
            'tax_rate_id' => null,
            'id' => $response->json('data.item_info.0.id'),
            'tax_allocation' => 0,
        ]);
    }

    /**
     * @throws BindingResolutionException
     */
    public function test_it_can_set_sales_credit_line_product_id(): void
    {
        $this->setUpWarehouse();
        $this->setUpProduct();
        /** @var SalesOrder $salesOrder */
        $salesOrder = $this->setUpSalesOrders(1)->first();
        (new BulkInventoryManager())->bulkAllocateNegativeInventoryEvents(SalesOrderLine::all());
        Sanctum::actingAs(User::factory()->create());

        $this->postJson(route('sales-credits.store'), [
            'sales_order_id' => $salesOrder->id,
            'sales_credit_lines' => [
                [
                    'sales_order_line_id' => $salesOrder->salesOrderLines->first()->id,
                    'quantity' => 1,
                    'amount' => 10,
                    'description' => 'test',
                ],
            ],
        ])->assertSuccessful();

        $this->assertDatabaseHas((new SalesCreditLine())->getTable(), [
            'product_id' => $this->product->id,
        ]);
    }
}
