<?php

namespace Tests\Feature;

use App\Data\FinancialSummaryQueryData;
use App\Enums\FinancialMetricEnum;
use App\Enums\SortEnum;
use App\Managers\ReportManager;
use App\Managers\WarehouseTransferManager;
use App\Models\Product;
use App\Models\ReportingDailyFinancial;
use App\Models\Warehouse;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\Sequence;
use Illuminate\Support\Facades\Cache;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class ReportManagerTest extends TestCase
{
    use FastRefreshDatabase;

    /**
     * @throws Throwable
     */
    public function test_it_can_report_on_inventory_valuation()
    {
        $reportManager = app(ReportManager::class);

        // Initial inventory
        $product = Product::factory()->create();

        $defaultWarehouse = Warehouse::first();
        $secondaryWarehouse = Warehouse::factory()->create();

        $product->setInitialInventory(1, 10, 10.0);

        $this->assertEquals([
            'total_quantity' => 10,
            'total_valuation' => 100.00,
            'average_cost' => 10.0,
            'active_quantity' => 10,
            'active_valuation' => 100.00,
            'reserved_quantity' => 0,
            'reserved_valuation' => 0.00,
            'in_transit_quantity' => 0,
            'in_transit_valuation' => 0.00,
        ], $reportManager->valuationSummary());

        Cache::flush();

        $warehouseTransferManager = app(WarehouseTransferManager::class);

        $warehouseTransfer = $warehouseTransferManager->initiateTransfer([
            'transfer_date' => Carbon::now()->toDateTimeString(),
            'from_warehouse_id' => $defaultWarehouse->id,
            'to_warehouse_id' => $secondaryWarehouse->id,
            'products' => [
                [
                    'id' => $product->id,
                    'quantity' => 2,
                ],
            ],
        ]);

        $warehouseTransferManager->openWarehouseTransfer($warehouseTransfer, []);

        $this->assertEquals([
            'total_quantity' => 10,
            'total_valuation' => 100.00,
            'average_cost' => 10.0,
            'active_quantity' => 8,
            'active_valuation' => 80.00,
            'reserved_quantity' => 0,
            'reserved_valuation' => 0.00,
            'in_transit_quantity' => 2,
            'in_transit_valuation' => 20.00,
        ], $reportManager->valuationSummary());
    }

    public function test_it_can_filter_financials_summaries(): void
    {
        $product1 = Product::factory()->withSupplierProduct()->withBrand()->create(['sku' => 'SKU1']);
        $product2 = Product::factory()->withSupplierProduct()->withBrand()->create(['sku' => 'SKU2']);
        $product3 = Product::factory()->withSupplierProduct()->withBrand()->create(['sku' => 'SKU3']);

        ReportingDailyFinancial::factory()
            ->state(new Sequence(
                [
                    'quantity' => '1',
                    'reportable_type' => Product::class,
                    'reportable_id' => $product1->id,
                    'date' => '2021-01-01'
                ],
                [
                    'quantity' => '2',
                    'reportable_type' => Product::class,
                    'reportable_id' => $product1->id,
                    'date' => '2021-01-02'
                ],
                [
                    'quantity' => '1',
                    'reportable_type' => Product::class,
                    'reportable_id' => $product1->id,
                    'date' => '2021-01-03'
                ],
                [
                    'quantity' => '10',
                    'reportable_type' => Product::class,
                    'reportable_id' => $product2->id,
                    'date' => '2021-01-01'
                ],
                [
                    'quantity' => '20',
                    'reportable_type' => Product::class,
                    'reportable_id' => $product2->id,
                    'date' => '2021-01-02'
                ],
                [
                    'quantity' => '15',
                    'reportable_type' => Product::class,
                    'reportable_id' => $product2->id,
                    'date' => '2021-01-03'
                ],
                [
                    'quantity' => '5',
                    'reportable_type' => Product::class,
                    'reportable_id' => $product3->id,
                    'date' => '2021-01-01'
                ],
                [
                    'quantity' => '4',
                    'reportable_type' => Product::class,
                    'reportable_id' => $product3->id,
                    'date' => '2021-01-02'
                ],
                [
                    'quantity' => '3',
                    'reportable_type' => Product::class,
                    'reportable_id' => $product3->id,
                    'date' => '2021-01-03'
                ],
            ))
            ->count(9)
            ->create();

        $manager = app(ReportManager::class);

        $reportData = $manager->getFinancialsBySku(FinancialSummaryQueryData::from([
            'interval' => 'day',
            'metric' => FinancialMetricEnum::UNITS_SOLD,
            'sort' => SortEnum::DESC,
        ]))['data'];

        // Assert getting all items
        $this->assertCount(3, $reportData);
        $this->assertEquals('SKU2', $reportData[0]['SKU']);
        $this->assertEquals('SKU3', $reportData[1]['SKU']);
        $this->assertEquals('SKU1', $reportData[2]['SKU']);

        $reportData = $manager->getFinancialsBySku(FinancialSummaryQueryData::from([
            'interval' => 'day',
            'metric' => FinancialMetricEnum::UNITS_SOLD,
            'limit' => 2,
            'sort' => SortEnum::DESC,
        ]))['data'];

        // Assert getting top 2
        $this->assertCount(2, $reportData);
        $this->assertEquals('SKU2', $reportData[0]['SKU']);
        $this->assertEquals('SKU3', $reportData[1]['SKU']);

        $reportData = $manager->getFinancialsBySku(FinancialSummaryQueryData::from([
            'interval' => 'day',
            'metric' => FinancialMetricEnum::UNITS_SOLD,
            'limit' => 2,
            'sort' => SortEnum::ASC,
        ]))['data'];

        // Assert getting bottom 2
        $this->assertCount(2, $reportData);
        $this->assertEquals('SKU1', $reportData[0]['SKU']);
        $this->assertEquals('SKU3', $reportData[1]['SKU']);
    }
}
