<?php

namespace Tests\Feature;

use App\Data\CreateInventoryAdjustmentData;
use App\Exceptions\PurchaseOrder\NotOpenPurchaseOrderException;
use App\Exceptions\PurchaseOrder\ReceivePurchaseOrderLineException;
use App\Helpers;
use App\Http\Requests\StoreInventoryAdjustment;
use App\Managers\InventoryAdjustmentManager;
use App\Models\AccountingTransaction;
use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineLayer;
use App\Models\Setting;
use App\Models\Supplier;
use App\Models\User;
use App\Models\Warehouse;
use App\Repositories\Accounting\AccountingTransactionRepository;
use App\Repositories\SalesOrderLineFinancialsRepository;
use App\Repositories\SettingRepository;
use App\Services\Accounting\AccountingTransactionManager;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use App\Services\InventoryManagement\BulkInventoryManager;
use App\Services\PurchaseOrder\PurchaseOrderValidator;
use App\Services\PurchaseOrder\ShipmentManager;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use Illuminate\Support\Facades\Redis;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class PurchaseOrderShipmentReceiptTest extends TestCase
{

    use FastRefreshDatabase;

    /**
     * @throws Throwable
     * @throws ReceivePurchaseOrderLineException
     * @throws NotOpenPurchaseOrderException
     */
    public function test_it_can_create_negative_adjustment_for_negative_receipt_quantity(): void{

        // Create purchase order
        /** @var PurchaseOrder $purchaseOrder */
        $purchaseOrder = PurchaseOrder::factory()
            ->withLines(quantity: 10)
            ->receiveAll()
            ->create();

        // Should be fully received
        $this->assertTrue($purchaseOrder->fully_received);

        $this->assertDatabaseCount('inventory_movements', 1);
        $this->assertDatabaseHas('inventory_movements', [
            'quantity' => $purchaseOrder->purchaseOrderLines->sum('quantity'),
            'type' => 'purchase_receipt',
        ]);

        $this->assertDatabaseCount('fifo_layers', 1);
        $this->assertDatabaseHas('fifo_layers', [
            'original_quantity' => $purchaseOrder->purchaseOrderLines->sum('quantity'),
            'fulfilled_quantity' => 0
        ]);
        $this->assertDatabaseEmpty('inventory_adjustments');

        // Receive negative quantity
        $manager = new ShipmentManager();
        $manager->receiveShipment([
            'purchase_order_id' => $purchaseOrder->id,
            'received_at' => now()->toIso8601ZuluString(),
            'receipt_lines' => $purchaseOrder->purchaseOrderLines->map(function ($line) {
                return [
                    'quantity' => -3,
                    'purchase_order_line_id' => $line->id,
                ];
            })->toArray(),
        ]);

        // PO should be partially received
        $this->assertFalse($purchaseOrder->fully_received);
        $this->assertTrue($purchaseOrder->partially_received);

        // Assert negative adjustment created
        $this->assertDatabaseCount('inventory_adjustments', 1);

        $this->assertDatabaseHas('inventory_adjustments', [
            'quantity' => -3,
            'link_type' => PurchaseOrderLine::class,
        ]);

        // Received quantity cache on purchase order line should be updated.
        $this->assertEquals(7, $purchaseOrder->purchaseOrderLines->first()->received_quantity);

        // Receive positive quantity
        $manager->receiveShipment([
            'purchase_order_id' => $purchaseOrder->id,
            'received_at' => now()->toIso8601ZuluString(),
            'receipt_lines' => $purchaseOrder->purchaseOrderLines->map(function ($line) {
                return [
                    'quantity' => 3,
                    'purchase_order_line_id' => $line->id,
                ];
            })->toArray(),
        ]);

        // PO should be fully received
        $this->assertTrue($purchaseOrder->fully_received);

        // Received quantity cache on purchase order line should be updated.
        $this->assertEquals(10, $purchaseOrder->purchaseOrderLines->first()->received_quantity);
    }

    /**
     * Here we are testing the negative receipt of a partially received purchase order.
     * The fifo layer chosen for the negative receipt should be the same as the original positive receipt.
     * However, it is possible that the positive receipt fifo layer is completely used, in which case, it should
     * move some usages of the original positive receipt to another fifo layer (if available)
     *
     * @throws Throwable
     * @throws ReceivePurchaseOrderLineException
     * @throws NotOpenPurchaseOrderException
     */
    public function test_it_can_create_negative_adjustment_for_negative_receipt_quantity_on_partially_received_po(): void{

        app(SettingRepository::class)->set(Setting::KEY_ACCOUNTING_ENABLED, true);

        // Create purchase order
        /** @var PurchaseOrder $purchaseOrder */
        $purchaseOrder = PurchaseOrder::factory()
            ->withLines(quantity: 10)
            ->receiveQuantity(quantity: 5)
            ->create();

        $originalFifoLayer = FifoLayer::first();

        $purchaseOrderLine = $purchaseOrder->purchaseOrderLines->first();
        $purchaseOrderLineShipmentReceipt = $purchaseOrderLine->purchaseOrderShipmentLines->first()->purchaseOrderShipmentReceiptLines->first();
        $purchaseOrderLineShipmentReceiptFifoLayer = $purchaseOrderLineShipmentReceipt->fifoLayers()->first();
        $product = $purchaseOrderLine->product;

        // Should be partially received
        $this->assertTrue($purchaseOrder->partially_received);
        $this->assertFalse($purchaseOrder->fully_received);

        $this->assertDatabaseCount('inventory_movements', 1);
        $this->assertDatabaseHas('inventory_movements', [
            'quantity' => 5,
            'type' => 'purchase_receipt',
        ]);

        $this->assertDatabaseCount('fifo_layers', 1);
        $this->assertDatabaseHas('fifo_layers', [
            'original_quantity' => 5,
            'fulfilled_quantity' => 0
        ]);
        $this->assertDatabaseEmpty('inventory_adjustments');

        // Fully use FIFO layer
        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines(1, ['product_id' => $product->id, 'quantity' => 5, 'warehouse_id' => $purchaseOrder->destination_warehouse_id])
            ->create();
        $salesOrderLine = $salesOrder->salesOrderLines()->first();
        app(BulkInventoryManager::class)->bulkAllocateNegativeInventoryEvents(collect([$salesOrderLine]));

        $this->assertDatabaseHas(SalesOrderLineLayer::class, [
            'quantity' => 5,
            'sales_order_line_id' => $salesOrderLine->id,
            'layer_type' => FifoLayer::class,
            'layer_id' => $purchaseOrderLineShipmentReceiptFifoLayer->id,
        ]);

        app(FulfillmentManager::class)->fulfill($salesOrder, [
            'fulfilled_at' => now(),
            'warehouse_id' => $purchaseOrder->destination_warehouse_id,
            'fulfillment_lines' => $salesOrder->salesOrderLines->map(function ($line) {
                return [
                    'quantity' => $line->quantity,
                    'sales_order_line_id' => $line->id,
                ];
            })->toArray(),
        ], false, false);

        (new SalesOrderLineFinancialManager())->calculate();
        app(AccountingTransactionManager::class)->sync();

        $this->assertEmpty(app(SalesOrderLineFinancialsRepository::class)->getInvalidatedLines());

        $inventoryAdjustment = app(InventoryAdjustmentManager::class)->createAdjustment(CreateInventoryAdjustmentData::from([
            'product_id' => $product->id,
            'quantity' => 10,
            'unit_cost' => 10,
            'adjustment_date' => now(),
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'warehouse_id' => $purchaseOrder->destination_warehouse_id,
        ]));
        $alternativeFifoLayer = $inventoryAdjustment->fifoLayers->first();

        // Receive negative quantity
        $manager = new ShipmentManager();
        $manager->receiveShipment([
            'purchase_order_id' => $purchaseOrder->id,
            'received_at' => now()->toIso8601ZuluString(),
            'receipt_lines' => $purchaseOrder->purchaseOrderLines->map(function ($line) {
                return [
                    'quantity' => -3,
                    'purchase_order_line_id' => $line->id,
                ];
            })->toArray(),
        ]);
        $purchaseOrderLineInventoryAdjustment = $purchaseOrderLine->adjustments()->first();

        // PO should be partially received
        $this->assertTrue($purchaseOrder->partially_received);
        $this->assertFalse($purchaseOrder->fully_received);

        // Purchase order line received quantity should be updated.
        $this->assertEquals(2, $purchaseOrder->purchaseOrderLines->first()->received_quantity);

        // Assert negative adjustment created
        $this->assertDatabaseHas('inventory_adjustments', [
            'quantity' => -3,
            'link_type' => PurchaseOrderLine::class,
        ]);

        $this->assertDatabaseHas('inventory_movements', [
            'quantity' => -3,
            'link_type' => InventoryAdjustment::class,
            'link_id' => $purchaseOrderLineInventoryAdjustment->id,
            'layer_type' => FifoLayer::class,
            'layer_id' => $purchaseOrderLineShipmentReceiptFifoLayer->id,
        ]);

        // Need Quantity (3) of Sales order line movements should now be moved to the alternative fifo layer
        $this->assertDatabaseHas('inventory_movements', [
            'quantity' => -3,
            'inventory_status' => 'active',
            'link_type' => SalesOrderLine::class,
            'layer_type' => FifoLayer::class,
            'layer_id' => $alternativeFifoLayer->id,
        ]);

        $this->assertDatabaseHas('inventory_movements', [
            'quantity' => 3,
            'inventory_status' => 'reserved',
            'link_type' => SalesOrderLine::class,
            'layer_type' => FifoLayer::class,
            'layer_id' => $alternativeFifoLayer->id,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'quantity' => -3,
            'inventory_status' => 'reserved',
            'link_type' => SalesOrderFulfillmentLine::class,
            'layer_type' => FifoLayer::class,
            'layer_id' => $alternativeFifoLayer->id,
        ]);
        $this->assertDatabaseHas(SalesOrderLineLayer::class, [
            'quantity' => 3,
            'sales_order_line_id' => $salesOrderLine->id,
            'layer_type' => FifoLayer::class,
            'layer_id' => $alternativeFifoLayer->id,
        ]);

        // Quantity 2 remain with original
        $this->assertDatabaseHas('inventory_movements', [
            'quantity' => -2,
            'inventory_status' => 'active',
            'link_type' => SalesOrderLine::class,
            'layer_type' => FifoLayer::class,
            'layer_id' => $originalFifoLayer->id,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'quantity' => 2,
            'inventory_status' => 'reserved',
            'link_type' => SalesOrderLine::class,
            'layer_type' => FifoLayer::class,
            'layer_id' => $originalFifoLayer->id,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'quantity' => -2,
            'inventory_status' => 'reserved',
            'link_type' => SalesOrderFulfillmentLine::class,
            'layer_type' => FifoLayer::class,
            'layer_id' => $originalFifoLayer->id,
        ]);
        $this->assertDatabaseHas(SalesOrderLineLayer::class, [
            'quantity' => 2,
            'sales_order_line_id' => $salesOrderLine->id,
            'layer_type' => FifoLayer::class,
            'layer_id' => $originalFifoLayer->id,
        ]);

        // COGS should be recalculated
        $this->assertNotEmpty(app(SalesOrderLineFinancialsRepository::class)->getInvalidatedLines());
        // Accounting transaction should be updated
        $this->assertTrue(app(AccountingTransactionRepository::class)->getTransactionsNeedingUpdate(100, 0)->contains($salesOrder->accountingTransaction));

        // Receive positive quantity
        $manager->receiveShipment([
            'purchase_order_id' => $purchaseOrder->id,
            'received_at' => now()->toIso8601ZuluString(),
            'receipt_lines' => $purchaseOrder->purchaseOrderLines->map(function ($line) {
                return [
                    'quantity' => 3,
                    'purchase_order_line_id' => $line->id,
                ];
            })->toArray(),
        ]);

        // PO should be partially received still as before
        $this->assertTrue($purchaseOrder->partially_received);

        // Purchase order line received quantity should be updated.
        $this->assertEquals(5, $purchaseOrder->purchaseOrderLines->first()->received_quantity);
    }

    public function test_negative_receipts_take_from_same_fifo_layer_as_original_positive_receipt(): void{


        Sanctum::actingAs(User::factory()->create());

        $quantity = 10;

        /** @var Product $product */
        $product = Product::factory()->create();

        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::factory()->create()->withDefaultLocation();

        /** @var Supplier $supplier */
        $supplier = Supplier::factory()->create();
        $supplier->products()->attach($product->id);

        // Create adjustment with fifo layers.
        $this->postJson('/api/inventory-adjustments', [
            'product_id' => $product->id,
            'quantity' => $quantity,
            'adjustment_date' => now(),
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'reason' => 'Initial stock',
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 10,
        ])->assertSuccessful();

        // Create purchase order and receive it.
        $response = $this->postJson('/api/purchase-orders', [
            'supplier_id' => $supplier->id,
            'purchase_order_date' => now(),
            'destination_warehouse_id' => $warehouse->id,
            'currency_code' => 'USD',
            'approval_status' => PurchaseOrderValidator::APPROVAL_STATUS_APPROVED,
            'purchase_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => $quantity,
                    'amount' => 10,
                ]
            ]
        ])->assertSuccessful();

        // Receive the purchase order.
        $this->postJson("/api/purchase-order-shipments/receive", [
            'received_at' => now(),
            'receipt_lines' => [
                [
                    'purchase_order_line_id' => $response->json('data.items.0.id'),
                    'quantity' => $quantity,
                ]
            ]
        ])->assertSuccessful();

        $purchaseOrder = PurchaseOrder::query()->findOrFail($response->json('data.id'));

        // Receive negative quantity
        $manager = new ShipmentManager();
        $purchaseOrder = $manager->receiveShipment([
            'purchase_order_id' => $purchaseOrder->id,
            'received_at' => now()->toIso8601ZuluString(),
            'receipt_lines' => $purchaseOrder->purchaseOrderLines->map(function ($line) {
                return [
                    'quantity' => -3,
                    'purchase_order_line_id' => $line->id,
                ];
            })->toArray(),
        ]);

        // PO should be partially received
        $this->assertFalse($purchaseOrder->fully_received);
        $this->assertTrue($purchaseOrder->partially_received);

        // Purchase order line received quantity should be updated.
        $this->assertEquals(7, $purchaseOrder->purchaseOrderLines->first()->received_quantity);

        // Assert negative adjustment created (1 extra for initial stock)
        $this->assertDatabaseCount('inventory_adjustments', 2);

        $this->assertDatabaseHas('inventory_adjustments', [
            'quantity' => -3,
            'link_type' => PurchaseOrderLine::class,
        ]);

        $receiptAdjustment = $purchaseOrder->purchaseOrderLines->pluck('adjustments')->flatten();
        $receiptFifoLayer = FifoLayer::query()
            ->where('link_type', PurchaseOrderShipmentReceiptLine::class)
            ->firstOrFail();

        // Negative adjustment should be linked to the same fifo layer as the original positive receipt.
        $this->assertDatabaseHas('inventory_movements', [
            'quantity' => -3,
            'link_type' => InventoryAdjustment::class,
            'link_id' => $receiptAdjustment->first()->id,
            'layer_type' => FifoLayer::class,
            'layer_id' => $receiptFifoLayer->id,
        ]);

        // Receive positive quantity
        $manager->receiveShipment([
            'purchase_order_id' => $purchaseOrder->id,
            'received_at' => now()->toIso8601ZuluString(),
            'receipt_lines' => $purchaseOrder->purchaseOrderLines->map(function ($line) {
                return [
                    'quantity' => 3,
                    'purchase_order_line_id' => $line->id,
                ];
            })->toArray(),
        ]);

        // PO should be fully received
        $this->assertTrue($purchaseOrder->fully_received);

        // Purchase order line received quantity should be updated.
        $this->assertEquals(10, $purchaseOrder->purchaseOrderLines->first()->received_quantity);

    }

}
