<?php

namespace Tests\Feature;

use App\Data\CreateStockTakeData;
use App\Data\StockTakeItemData;
use App\Data\UpdateStockTakeData;
use App\Exceptions\PurchaseOrder\NotOpenPurchaseOrderException;
use App\Exceptions\PurchaseOrder\ReceivePurchaseOrderLineException;
use App\Http\Requests\StoreInventoryAdjustment;
use App\Jobs\BackorderPurchasingJob;
use App\Models\AccountingTransaction;
use App\Models\AccountingTransactionLine;
use App\Models\BackorderQueue;
use App\Models\BackorderQueueRelease;
use App\Models\Currency;
use App\Models\Customer;
use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\ProductBrand;
use App\Models\ProductInventory;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\PurchaseOrderShipment;
use App\Models\PurchaseOrderShipmentReceipt;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineFinancial;
use App\Models\StockTake;
use App\Models\StockTakeItem;
use App\Models\Supplier;
use App\Models\SupplierPricingTier;
use App\Models\SupplierProduct;
use App\Models\SupplierProductPricing;
use App\Models\User;
use App\Models\Warehouse;
use App\Repositories\FifoLayerRepository;
use App\Repositories\PurchaseOrderRepository;
use App\Services\Accounting\AccountingTransactionManager;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use App\Services\InventoryAdjustment\InventoryAdjustmentService;
use App\Services\InventoryManagement\BackorderManager;
use App\Services\InventoryManagement\BulkInventoryManager;
use App\Services\PurchaseOrder\PurchaseOrderManager;
use App\Services\PurchaseOrder\PurchaseOrderValidator;
use App\Services\PurchaseOrder\ShipmentManager;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use App\Services\SalesOrder\SalesOrderManager;
use App\Services\StockTake\StockTakeManager;
use Carbon\Carbon;
use Database\Factories\FactoryDataRecycler;
use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Database\Eloquent\Factories\Sequence;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Tests\TestCase;
use Throwable;

class PurchaseOrderManagerTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    private function resetUpdatedAt(AccountingTransaction $model): void
    {
        $model->refresh();
        $model->updated_at = Carbon::now()->subDay();
        $model->save();
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_create_backorder_purchase_order(): void
    {
        Queue::fake();
        Mail::fake();

        /** @var ProductBrand $brand1 */
        $brand1 = ProductBrand::factory()->create();

        /** @var ProductBrand $brand2 */
        $brand2 = ProductBrand::factory()->create();

        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::query()->first();

        /** @var Warehouse $warehouse2 */
        $warehouse2 = Warehouse::factory()->create();

        /** @var Warehouse $supplierWarehouse */
        $supplierWarehouse = Warehouse::factory()->create([
            'name' => 'SupplierWarehouse',
            'type' => Warehouse::TYPE_SUPPLIER,
        ]);

        /** @var Supplier $supplier */
        $supplier = Supplier::factory()->create([
            'name' => 'BackorderSupplier',
            'auto_generate_backorder_po' => true,
            'default_warehouse_id' => $supplierWarehouse->id,
            'default_pricing_tier_id' => SupplierPricingTier::default()->id,
            'auto_submit_backorder_po' => true,
            'auto_split_backorder_po_by_brand' => true,
            'auto_receive_backorder_po' => true,
        ]);

        /** @var Product $product */
        $product = Product::factory()->has(
            SupplierProduct::factory()->has(
                SupplierProductPricing::factory(null, [
                    'price' => 4.99,
                    'supplier_pricing_tier_id' => SupplierPricingTier::default()->id,
                ])
            )->state([
                'supplier_id' => $supplier->id,
                'is_default' => true,
            ])
        )->create([
            'unit_cost' => 4.99,
            'brand_id' => $brand1->id,
        ]);

        /** @var Product $product2 */
        $product2 = Product::factory()->has(
            SupplierProduct::factory()->has(
                SupplierProductPricing::factory(null, [
                    'price' => 5.99,
                    'supplier_pricing_tier_id' => SupplierPricingTier::default()->id,
                ])
            )->state([
                'supplier_id' => $supplier->id,
                'is_default' => true,
            ])
        )->create([
            'unit_cost' => 4.99,
            'brand_id' => $brand1->id,
        ]);

        /** @var Product $product2 */
        $product2 = Product::factory()->has(
            SupplierProduct::factory()->has(
                SupplierProductPricing::factory(null, [
                    'price' => 5.99,
                    'supplier_pricing_tier_id' => SupplierPricingTier::default()->id,
                ])
            )->state([
                'supplier_id' => $supplier->id,
                'is_default' => true,
            ])
        )->create([
            'unit_cost' => 5.99,
            'brand_id' => $brand1->id,
        ]);

        /** @var Product $product3 */
        $product3 = Product::factory()->has(
            SupplierProduct::factory()->has(
                SupplierProductPricing::factory(null, [
                    'price' => 1.99,
                    'supplier_pricing_tier_id' => SupplierPricingTier::default()->id,
                ])
            )->state([
                'supplier_id' => $supplier->id,
                'is_default' => true,
            ])
        )->create([
            'unit_cost' => 1.99,
            'brand_id' => $brand2->id,
        ]);

        Sanctum::actingAs(User::factory()->create());

        $this->postJson('/api/sales-orders', [
            'sales_order_number' => 'BackorderSalesOrder',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'order_date' => Carbon::now()->format('Y-m-d H:i:s'),
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => $product->name,
                    'warehouse_id' => $warehouse->id,
                    'quantity' => 5,
                    'amount' => 6.00,
                ],
                [
                    'product_id' => $product2->id,
                    'description' => $product2->name,
                    'warehouse_id' => $warehouse2->id,
                    'quantity' => 3,
                    'amount' => 7.00,
                ],
                [
                    'product_id' => $product3->id,
                    'description' => $product3->name,
                    'warehouse_id' => $warehouse2->id,
                    'quantity' => 4,
                    'amount' => 4.00,
                ],
            ],
        ])->assertOk();

        SalesOrder::query()->first();

        $this->assertCount(1, app(PurchaseOrderRepository::class)->suppliersEligibleForBackorderPurchasing());

        (new PurchaseOrderManager($supplier))->processBackorderPurchasing();

        $this->assertDatabaseHas((new PurchaseOrder())->getTable(), [
            'supplier_id' => $supplier->id,
            'destination_warehouse_id' => $warehouse->id,
            'order_status' => PurchaseOrder::STATUS_CLOSED,
            'receipt_status' => PurchaseOrder::RECEIPT_STATUS_RECEIVED,
            'currency_id' => Currency::default()->id,
        ]);

        $this->assertEquals(
            2,
            PurchaseOrder::query()
                ->where('supplier_id', $supplier->id)
                ->where('destination_warehouse_id', $warehouse2->id)
                ->where('order_status', PurchaseOrder::STATUS_CLOSED)
                ->where('receipt_status', PurchaseOrder::RECEIPT_STATUS_RECEIVED)
                ->where('currency_id', Currency::default()->id)
                ->count()
        );

        $this->assertDatabaseHas((new PurchaseOrderLine())->getTable(), [
            'product_id' => $product->id,
            'description' => $product->name,
            'quantity' => 5,
            'amount' => 4.99,
        ]);

        $this->assertDatabaseHas((new PurchaseOrderLine())->getTable(), [
            'product_id' => $product2->id,
            'description' => $product2->name,
            'quantity' => 3,
            'amount' => 5.99,
        ]);

        $this->assertDatabaseHas((new PurchaseOrderLine())->getTable(), [
            'product_id' => $product3->id,
            'description' => $product3->name,
            'quantity' => 4,
            'amount' => 1.99,
        ]);

        $this->assertDatabaseHas((new PurchaseOrderLine())->getTable(), [
            'product_id' => $product2->id,
            'description' => $product2->name,
            'quantity' => 3,
            'amount' => 5.99,
        ]);

        $this->assertDatabaseHas((new PurchaseOrderLine())->getTable(), [
            'product_id' => $product3->id,
            'description' => $product3->name,
            'quantity' => 4,
            'amount' => 1.99,
        ]);

        $this->assertDatabaseCount((new PurchaseOrderShipment())->getTable(), 3);
        $this->assertDatabaseCount((new PurchaseOrderShipmentReceipt())->getTable(), 3);
        $this->assertDatabaseCount((new PurchaseOrderShipmentReceiptLine())->getTable(), 3);
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 5,
            'type' => InventoryMovement::TYPE_PURCHASE_RECEIPT,
        ]);

        // All backorders covered, so no supplier eligible
        $this->assertCount(0, app(PurchaseOrderRepository::class)->suppliersEligibleForBackorderPurchasing());
    }

    /**
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function test_scheduling_backorder_purchasing(): void
    {
        Supplier::factory()->create([
            'name' => 'BackorderSupplier',
            'auto_generate_backorder_po' => true,
            'backorder_po_schedule' => [
                [
                    'time' => '04:10',
                    'days' => ['monday', 'tuesday'],
                ],
                [
                    'time' => '11:00',
                    'days' => ['wednesday'],
                ],
            ],
        ]);

        $schedule = $this->app->get(Schedule::class);

        $this->assertCount(3, collect($schedule->events())->filter(function ($event) {
            return $event->description === BackorderPurchasingJob::class;
        }));
    }

    /**
     * @throws Throwable
     * @throws NotOpenPurchaseOrderException
     * @throws ReceivePurchaseOrderLineException
     */
    public function test_it_can_clear_backorder_queue_released_quantity_on_receipt_deletion(): void
    {
        Queue::fake();

        // Create direct warehouses
        $warehouse = Warehouse::factory(1)->direct()->create();

        // Create suppliers
        $supplier = Supplier::factory(1)->create();

        // Create products, all associated with suppliers
        $products = Product::factory(1)
            ->withProductPricing()
            ->withSupplierProduct(
                (new FactoryDataRecycler([
                    $supplier,
                ]))
            )
            ->create();

        // Create sales orders
        SalesOrder::factory(1)->withLines(
            1,
            (new FactoryDataRecycler())
                ->addRecycledData($products, null, true)
                ->addRecycledData($warehouse)
        )->open()
            ->create([
                'sales_channel_id' => SalesChannel::query()->first()->id,
                'order_date' => $this->faker->dateTimeThisYear(),
            ]);

        SalesOrderLine::query()->first()->update([
            'quantity' => 20,
        ]);

        (new BulkInventoryManager())->bulkAllocateNegativeInventoryEvents(SalesOrderLine::all());

        $purchaseOrder = PurchaseOrder::factory(1)
            ->withLines(
                1,
                (new FactoryDataRecycler())->addRecycledData($products, null, false)
            )
            /*
             * This is Laravel's built-in recycle method.  We need to explore the differences in functionality between
             * my FactoryDataRecycler class and Laravel's recycle method.
             */
            //->recycle($suppliers)
            ->factoryDataRecycler(
                (new FactoryDataRecycler())
                    ->addRecycledData($warehouse, 'destination_warehouse_id')
                    ->addRecycledData($supplier)
            )
            ->receiveAll() // not working?
            ->create()->first();

        /** @var PurchaseOrderShipmentReceipt $purchaseOrderShipmentReceipt */
        $purchaseOrderShipmentReceipt = PurchaseOrderShipmentReceipt::query()->first();

        $this->assertDatabaseHas((new BackorderQueueRelease())->getTable(), [
            'link_type' => PurchaseOrderShipmentReceiptLine::class,
        ]);
        $purchaseOrderShipmentReceipt->delete();
        $this->assertDatabaseEmpty((new BackorderQueueRelease())->getTable());
        $this->assertEquals(
            0,
            BackorderQueue::query()
                ->where('released_quantity', '>', 0)
                ->count()
        );
    }

    /**
     * @throws Throwable
     * @throws NotOpenPurchaseOrderException
     * @throws ReceivePurchaseOrderLineException
     */
    public function test_it_updates_backorder_queue_cache_properly_on_unreceived_purchase_order_deletion(): void
    {
        Queue::fake();

        // Create direct warehouses
        $warehouse = Warehouse::factory(1)->direct()->create();

        // Create suppliers
        $supplier = Supplier::factory(1)->create();

        // Create products, all associated with suppliers
        $products = Product::factory(1)
            ->withProductPricing()
            ->withSupplierProduct(
                (new FactoryDataRecycler([
                    $supplier,
                ]))
            )
            ->create();

        // Create sales orders
        SalesOrder::factory(1)->withLines(
            1,
            (new FactoryDataRecycler())
                ->addRecycledData($products, null, true)
                ->addRecycledData($warehouse)
        )->open()
            ->create([
                'sales_channel_id' => SalesChannel::query()->first()->id,
                'order_date' => $this->faker->dateTimeThisYear(),
            ]);

        SalesOrderLine::query()->first()->update([
            'quantity' => 3,
        ]);

        (new BulkInventoryManager())->bulkAllocateNegativeInventoryEvents(SalesOrderLine::all());

        $purchaseOrder = PurchaseOrder::factory(1)
            ->withLines(
                1,
                (new FactoryDataRecycler())->addRecycledData($products, null, false),
                2
            )
            /*
             * This is Laravel's built-in recycle method.  We need to explore the differences in functionality between
             * my FactoryDataRecycler class and Laravel's recycle method.
             */
            //->recycle($suppliers)
            ->factoryDataRecycler(
                (new FactoryDataRecycler())
                    ->addRecycledData($warehouse, 'destination_warehouse_id')
                    ->addRecycledData($supplier)
            )
            ->receiveAll()
            ->create()->first();

        $purchaseOrderLine = $purchaseOrder->purchaseOrderLines()->first();

        (new BackorderManager())->withGlobalCoverage()->coverBackorderQueues([$purchaseOrderLine->id]);

        $this->assertDatabaseHas((new BackorderQueueRelease())->getTable(), [
            'link_type' => PurchaseOrderShipmentReceiptLine::class,
        ]);

        $this->assertDatabaseHas((new BackorderQueue())->getTable(), [
            'released_quantity' => 2,
        ]);

        $purchaseOrder2 = PurchaseOrder::factory(1)
            ->withLines(
                1,
                (new FactoryDataRecycler())->addRecycledData($products, null, false),
                1
            )
            /*
             * This is Laravel's built-in recycle method.  We need to explore the differences in functionality between
             * my FactoryDataRecycler class and Laravel's recycle method.
             */
            //->recycle($suppliers)
            ->factoryDataRecycler(
                (new FactoryDataRecycler())
                    ->addRecycledData($warehouse, 'destination_warehouse_id')
                    ->addRecycledData($supplier)
            )
            ->receiveAll()
            ->create()->first();

        $purchaseOrderLine2 = $purchaseOrder2->purchaseOrderLines()->first();

        (new BackorderManager())->withGlobalCoverage()->coverBackorderQueues([$purchaseOrderLine2->id]);

        $this->assertDatabaseHas((new BackorderQueue())->getTable(), [
            'released_quantity' => 3,
        ]);

        // Delete 1/2 purchase orders and assert that the released quantity cache goes from 3 to 1
        $purchaseOrder->delete();

        $this->assertDatabaseHas((new BackorderQueue())->getTable(), [
            'released_quantity' => 1,
        ]);

        $this->assertDatabaseCount((new BackorderQueueRelease())->getTable(), 1);
    }

    public function test_purchase_order_lines_scope_function_with_cost_values(): void
    {
        // Create currency
        $currency = Currency::factory()->create([
            'code' => 'MXN',
            'conversion' => '1.5',
        ]);

        // Create direct warehouses
        $warehouse = Warehouse::factory(1)->direct()->create();

        // Create suppliers
        $supplier = Supplier::factory(1)->create();

        // Create products, all associated with suppliers
        $products = Product::factory(1)
            ->withProductPricing()
            ->withSupplierProduct(
                (new FactoryDataRecycler([
                    $supplier,
                ]))
            )
            ->create();

        //Create purchase order
        PurchaseOrder::factory(1)
            ->withLines(
                1,
                (new FactoryDataRecycler())->addRecycledData($products, null, false),
                2
            )
            /*
             * This is Laravel's built-in recycle method.  We need to explore the differences in functionality between
             * my FactoryDataRecycler class and Laravel's recycle method.
             */
            //->recycle($suppliers)
            ->factoryDataRecycler(
                (new FactoryDataRecycler())
                    ->addRecycledData($warehouse, 'destination_warehouse_id')
                    ->addRecycledData($supplier)
            )
            ->create([
                'currency_id' => $currency->id,
                'currency_code' => $currency->code,
            ])
            ->first();

        $totalAmount = round((PurchaseOrderLine::query()->sum('amount') * PurchaseOrderLine::query()->sum('quantity')) * $currency->conversion, 2);
        $totalSumFromScopeFunction = round(PurchaseOrderLine::select('total_product_cost')->withCostValues(null, true)->get()->sum('total_product_cost'), 2);

        $this->assertEquals($totalSumFromScopeFunction, $totalAmount);
    }

    public function test_it_can_update_fifo_layer_and_inventory_movements_when_receipt_line_gets_removed(): void
    {
        Queue::fake();

        Sanctum::actingAs(User::factory()->create());

        $product1 = Product::factory()->create();
        $product2 = Product::factory()->create();
        /** @var Supplier $supplier */
        $supplier = Supplier::factory()->create();
        $warehouse = Warehouse::factory()->create();

        $supplier->products()->attach([$product1->id, $product2->id]);

        $response = $this->postJson('/api/purchase-orders', [
            'purchase_order_date' => now(),
            'destination_warehouse_id' => $warehouse->id,
            'supplier_id' => $supplier->id,
            'currency_code' => 'USD',
            'approval_status' => PurchaseOrderValidator::APPROVAL_STATUS_APPROVED,
            'purchase_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 3,
                    'amount' => 10,
                ],
                [
                    'product_id' => $product2->id,
                    'quantity' => 4,
                    'amount' => 10,
                ],
            ],
        ])->assertSuccessful();
        $this->assertDatabaseHas('purchase_order_lines', [
            'received_quantity' => 0,
        ]);

        // Partially receive PO
        $receiptResponse = $this->postJson('/api/purchase-order-shipments/receive', [
            'purchase_order_id' => $response->json('data.id'),
            'received_at' => now(),
            'receipt_lines' => [
                [
                    'purchase_order_line_id' => $response->json('data.items.0.id'),
                    'quantity' => 2,
                ],
                [
                    'purchase_order_line_id' => $response->json('data.items.1.id'),
                    'quantity' => 1,
                ],
            ],
        ])->assertSuccessful();

        $this->assertDatabaseHas('purchase_order_lines', [
            'product_id' => $product1->id,
            'received_quantity' => 2,
        ]);
        $this->assertDatabaseHas('purchase_order_lines', [
            'product_id' => $product2->id,
            'received_quantity' => 1,
        ]);
        $this->assertDatabaseCount('fifo_layers', 2);
        $this->assertDatabaseHas('fifo_layers', [
            'product_id' => $product1->id,
            'original_quantity' => 2,
            'fulfilled_quantity' => 0,
        ]);
        $this->assertDatabaseHas('fifo_layers', [
            'product_id' => $product2->id,
            'original_quantity' => 1,
            'fulfilled_quantity' => 0,
        ]);
        $this->assertDatabaseCount('inventory_movements', 2);

        // Remove second receipt line and reduce quantity of first receipt line
        $this->putJson('/api/purchase-order-shipments/receipts/'.$receiptResponse->json('data.id'), [
            'received_at' => now(),
            'receipt_lines' => [
                [
                    'purchase_order_line_id' => $response->json('data.items.0.id'),
                    'quantity' => 1,
                    'amount' => 10,
                ],
            ],
        ])->assertSuccessful();

        // Cache should be updated
        $this->assertDatabaseHas('purchase_order_lines', [
            'product_id' => $product1->id,
            'received_quantity' => 1,
        ]);
        $this->assertDatabaseHas('purchase_order_lines', [
            'product_id' => $product2->id,
            'received_quantity' => 0,
        ]);

        $this->assertDatabaseCount('fifo_layers', 1);
        $this->assertDatabaseHas('fifo_layers', [
            'product_id' => $product1->id,
            'original_quantity' => 1,
            'fulfilled_quantity' => 0,
        ]);

        $this->assertDatabaseCount('inventory_movements', 1);
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $product1->id,
            'quantity' => 1,
        ]);
    }

    public function test_it_can_apply_discount_to_all_purchase_order_lines(): void
    {
        // Apply 10% Discount to all purchase order lines.
        app(PurchaseOrderManager::class)
            ->applyDiscountLines(PurchaseOrder::factory()->hasPurchaseOrderLines(3)->create()->first(), 0.35);

        $this->assertDatabaseHas((new PurchaseOrderLine())->getTable(), [
            'discount_rate' => 0.35,
        ]);

        $this->assertDatabaseMissing((new PurchaseOrderLine())->getTable(), [
            'discount_rate' => null,
        ]);
    }

    public function test_it_can_apply_pricing_tier_to_all_purchase_order_lines(): void
    {
        $supplier = Supplier::factory()->create();
        $product1 = Product::factory()->create();
        $product2 = Product::factory()->create();
        $product3 = Product::factory()->create();

        $supplierPricingTier = SupplierPricingTier::factory()->create();

        SupplierProductPricing::factory()->setPricing($supplierPricingTier, $supplier, $product1, 5.00);
        SupplierProductPricing::factory()->setPricing($supplierPricingTier, $supplier, $product2, 6.00);
        SupplierProductPricing::factory()->setPricing($supplierPricingTier, $supplier, $product3, 7.00);

        $purchaseOrder = PurchaseOrder::factory()->has(
            PurchaseOrderLine::factory(3, new Sequence(
                ['product_id' => $product1->id, 'amount' => 1.00],
                ['product_id' => $product2->id, 'amount' => 2.00],
                ['product_id' => $product3->id, 'amount' => 3.00],
            ))
        )->create(['supplier_id' => $supplier->id]);

        // Apply Simple Pricing tier to all purchase order lines.
        app(PurchaseOrderManager::class)
            ->applyPricingTier($purchaseOrder, $supplierPricingTier);

        $this->assertDatabaseHas((new PurchaseOrderLine())->getTable(), [
            'product_id' => $product1->id,
            'amount' => 5.00,
        ]);

        $this->assertDatabaseHas((new PurchaseOrderLine())->getTable(), [
            'product_id' => $product2->id,
            'amount' => 6.00,
        ]);

        $this->assertDatabaseHas((new PurchaseOrderLine())->getTable(), [
            'product_id' => $product3->id,
            'amount' => 7.00,
        ]);
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_update_cogs_when_purchase_order_modified_after_receipt(): void
    {
        // Set up a received purchase order

        $supplier = Supplier::first();
        $product = Product::factory()->create();
        $warehouse = Warehouse::first();

        $purchaseOrder = PurchaseOrder::factory()
        ->hasPurchaseOrderLines(1, [
            'product_id' => $product->id,
            'amount' => 10,
            'quantity' => 5,
        ])
        ->create([
            'supplier_id' => $supplier->id,
            'destination_warehouse_id' => $warehouse->id,
        ]);

        (new ShipmentManager())->receiveShipment([
            'purchase_order_id' => $purchaseOrder->id,
            'received_at' => now(),
            'warehouse_id' => $warehouse->id,
            'receipt_lines' => [
                [
                    'purchase_order_line_id' => $purchaseOrder->purchaseOrderLines->first()->id,
                    'quantity' => 5,
                ],
            ],
        ]);

        $this->assertDatabaseHas((new FifoLayer())->getTable(), [
            'product_id' => $product->id,
            'original_quantity' => 5,
            'fulfilled_quantity' => 0,
            'total_cost' => 50,
        ]);

        /*
         * Create usages for each negative inventory movement possibility that has a COGS implication
         * - Sales Order Fulfillment Line
         * - Inventory Adjustment
         * - Stock Take Item
         * TODO: Add test for assembly once completed
         * - Assembly Line
         */


        // Sales Order Fulfillment Line

        app(SalesOrderManager::class)->createOrder([
            'sales_order_number' => 'TestSalesOrder',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'order_date' => now(),
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => 'TestProduct',
                    'warehouse_id' => $warehouse->id,
                    'quantity' => 1,
                    'amount' => 100,
                ],
            ],
        ]);

        app(FulfillmentManager::class)->fulfill(SalesOrder::first(), [
            'fulfilled_at' => now(),
            'warehouse_id' => $warehouse->id,
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => SalesOrderLine::first()->id,
                    'quantity' => 1,
                ],
            ],
        ]);

        $usageSalesOrderLine = SalesOrderLine::first();
        $usageSalesOrderFulfillmentLine = SalesOrderFulfillmentLine::first();

        // Inventory Adjustment
        // TODO: Currently we have no good way to create an adjustment outside of a request

        Sanctum::actingAs(User::first());

        $this->post(route('inventory-adjustments.store'), [
            'warehouse_id' => $warehouse->id,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_DECREASE,
            'adjustment_date' => now(),
            'product_id' => $product->id,
            'quantity' => 1,
            'amount' => 10,
        ])->assertOk();

        $this->assertDatabaseHas((new InventoryAdjustment())->getTable(), [
            'warehouse_id' => $warehouse->id,
            'product_id' => $product->id,
            'quantity' => -1,
        ]);

        $usageAdjustment = InventoryAdjustment::first();

        // Stock Take Item

        $stockTakeManager = app(StockTakeManager::class);

        $stockTake = $stockTakeManager->createStockTake(CreateStockTakeData::from([
            'warehouse_id' => $warehouse->id,
            'date_count' => now(),
            'items' => StockTakeItemData::collection([
                [
                    'product_id' => $product->id,
                ],
            ]),
        ]));

        $stockTakeManager->initiateCount($stockTake);

        $stockTake->refresh();

        $stockTakeManager->modifyStockTake($stockTake, UpdateStockTakeData::from([
            'items' => StockTakeItemData::collection([
                [
                    'product_id' => $product->id,
                    'qty_counted' => 2, // causes a -1 adjustment
                ],
            ]),
        ]));

        $stockTakeManager->finalizeStockTake($stockTake);
        $usageStockTakeItem = StockTakeItem::first();

        (new SalesOrderLineFinancialManager())->calculate();
        app(AccountingTransactionManager::class)->sync();

        $this->assertDatabaseHas((new FifoLayer())->getTable(), [
            'product_id' => $product->id,
            'original_quantity' => 5,
            'fulfilled_quantity' => 3,
            'total_cost' => 50,
        ]);

        $this->assertDatabaseHas((new SalesOrderLineFinancial())->getTable(), [
            'sales_order_line_id' => $usageSalesOrderLine->id,
            'cogs' => 10,
        ]);

        // Sales Order Fulfillment Line

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $usageSalesOrderFulfillmentLine->salesOrderFulfillment->id,
            'link_type' => SalesOrderFulfillment::class,
            'total' => 10,
        ]);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'link_id' => $usageSalesOrderFulfillmentLine->id,
            'link_type' => SalesOrderFulfillmentLine::class,
            'quantity' => 1,
            'amount' => 10,
        ]);

        // Adjustment

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $usageAdjustment->id,
            'link_type' => InventoryAdjustment::class,
            'total' => 10,
        ]);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'link_id' => $usageAdjustment->id,
            'link_type' => InventoryAdjustment::class,
            'quantity' => -1, // TODO: These should probably be positive
            'amount' => -10, // TODO: These should probably be positive
        ]);

        // Stock Take Item

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $usageStockTakeItem->stockTake->id,
            'link_type' => StockTake::class,
            'total' => 10,
        ]);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'link_id' => $usageStockTakeItem->id,
            'link_type' => StockTakeItem::class,
            'quantity' => 1,
            'amount' => 10,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Fifo value change
        |--------------------------------------------------------------------------
        */

        // Add other line item for customs duties for the purchase order
        $purchaseOrderLines = $purchaseOrder->purchaseOrderLines->toArray();
        $purchaseOrderLines[] = [
            'product_id' => null,
            'description' => 'Customs Duties',
            'quantity' => 1,
            'amount' => 10,
        ];

        $purchaseOrder->setPurchaseOrderLines($purchaseOrderLines);

        $fifoLayer = FifoLayer::first();

        app(FifoLayerRepository::class)->recalculateTotalCosts([$fifoLayer->id]);

        (new SalesOrderLineFinancialManager())->calculate();

        /*
         * Because tests run fast, there is no timestamp difference to trigger a COGS update in the accounting transactions
         * So for testing purposes, we are manually subtracting a minute so that the
         * accounting transaction gets updated.
         */
        AccountingTransaction::query()->update(['updated_at' => now()->subMinute()]);

        app(AccountingTransactionManager::class)->sync();

        $this->assertDatabaseHas((new FifoLayer())->getTable(), [
            'product_id' => $product->id,
            'original_quantity' => 5,
            'fulfilled_quantity' => 3,
            'total_cost' => 60,
        ]);

        $this->assertDatabaseHas((new SalesOrderLineFinancial())->getTable(), [
            'sales_order_line_id' => $usageSalesOrderLine->id,
            'cogs' => 12,
        ]);

        // Sales Order Fulfillment Line

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $usageSalesOrderFulfillmentLine->salesOrderFulfillment->id,
            'link_type' => SalesOrderFulfillment::class,
            'total' => 12,
        ]);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'link_id' => $usageSalesOrderFulfillmentLine->id,
            'link_type' => SalesOrderFulfillmentLine::class,
            'quantity' => 1,
            'amount' => 12,
        ]);

        // Adjustment

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $usageAdjustment->id,
            'link_type' => InventoryAdjustment::class,
            'total' => 12,
        ]);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'link_id' => $usageAdjustment->id,
            'link_type' => InventoryAdjustment::class,
            'quantity' => -1,
            'amount' => -12,
        ]);

        // Stock Take Item

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $usageStockTakeItem->stockTake->id,
            'link_type' => StockTake::class,
            'total' => 12,
        ]);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'link_id' => $usageStockTakeItem->id,
            'link_type' => StockTakeItem::class,
            'quantity' => 1,
            'amount' => 12,
        ]);
    }
}
