<?php

namespace Tests\Feature;

use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\User;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class PurchaseOrderLineTest extends TestCase
{
    use FastRefreshDatabase;

    public function test_discount_calculation_is_correct(): void
    {
        Queue::fake();
        Sanctum::actingAs(User::first());

        $purchaseOrder = PurchaseOrder::factory()->create();

        $purchaseOrderLine = PurchaseOrderLine::factory()->create([
            'purchase_order_id' => $purchaseOrder->id,
            'quantity' => 4,
            'amount' => 200.00,
            'discount_rate' => 0.1,
        ])->fresh();

        $endpoint = '/api/purchase-order-shipments/receive';

        // Define the request body as an array
        $requestBody = [
            'received_at' => '2023-09-24T14:01:46.588Z',
            'receipt_lines' => [
                [
                    'purchase_order_line_id' => $purchaseOrderLine->id,
                    'quantity' => 2,
                ],
            ],
            'purchase_order_id' => $purchaseOrder->id,
        ];

        $this->postJson($endpoint, $requestBody);

        // Calculate the expected discount amount
        $expectedDiscountAmount = ($purchaseOrderLine->amount * $purchaseOrderLine->discount_rate);

        // Assert that the 'discount_amount' column matches the expected value
        $this->assertEquals($expectedDiscountAmount, $purchaseOrderLine->discount_amount);
        $expectedTotalAmount = ($purchaseOrderLine->amount * $purchaseOrderLine->quantity - $purchaseOrderLine->discount_amount_extended);
        $this->assertEquals($expectedTotalAmount, $purchaseOrderLine->total_amount);

        // Assert that the FifoLayer table has the expected 'total_cost' value
        $this->assertDatabaseHas('fifo_layers', [
            'total_cost' => ($expectedTotalAmount / $purchaseOrderLine->quantity) * 2,
        ]);
    }
}
