<?php

namespace Tests\Feature;

use App;
use App\Http\Requests\StoreInventoryAdjustment;
use App\Models\Customer;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\ProductPricingTier;
use App\Models\PurchaseOrder;
use App\Models\SalesOrder;
use App\Models\Supplier;
use App\Models\User;
use App\Models\Warehouse;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\PurchaseOrderBuilderFactory;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\PurchaseOrderBuilderRepository;
use App\Services\PurchaseOrder\PurchaseOrderValidator;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Queue;
use Illuminate\Testing\Fluent\AssertableJson;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

/**
 * @group manual
 */
class PurchaseOrderBuilderTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    private function initialize(): array
    {
        $this->withoutExceptionHandling();

        Sanctum::actingAs(User::factory()->create());

        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::query()->first();
        /** @var Customer $customer */
        $customer = Customer::factory()->create();

        /** @var Supplier $supplier */
        $supplier = Supplier::query()->first();

        /** @var Product $product1 */
        $product1 = Product::factory()->create();
        /** @var Product $product2 */
        $product2 = Product::factory()->create();

        // Associate products with supplier
        $supplier->products()->attach($product1->id, ['is_default' => true]);
        $supplier->products()->attach($product2->id, ['is_default' => true]);

        return [
            $product1,
            $product2,
            $customer,
            $warehouse,
            $supplier,
        ];
    }

    public function test_it_can_fill_backorders_based_on_deliver_by_date_only_for_backordered_products(): void
    {
        [$product1, $product2, $customer, $warehouse, $supplier] = $this->initialize();

        $product1->setInitialInventory($warehouse->id, 3);

        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'order_date' => now()->subMinute(),
            'deliver_by_date' => now()->addDays(2),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 3,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
                [
                    'product_id' => $product2->id,
                    'quantity' => 5,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        $response = $this->postJson('api/purchase-orders/build', [
            'supplier_id' => $supplier->id,
            'destination_warehouse_id' => $warehouse->id,
            'forecast_type' => PurchaseOrderBuilderFactory::BUILDER_TYPE_FILL_BACKORDERS,
            'options' => [
                'sales_history_filters' => [
                    'filters' => [
                        'conjunction' => 'or',
                        'filterSet' => [
                            [
                                'column' => 'deliver_by_date',
                                'operator' => '<',
                                'value' => [
                                    'mode' => 'oneWeekFromNow',
                                ],
                            ],
                        ],
                    ],
                ],
                'product_filters' => [],
            ],
        ])->assertSuccessful();

        // We should get only one result as product1 is in stock.
        $response->assertJsonCount(1, 'data');
    }

    public function test_it_can_fill_backorders_based_on_deliver_by_date(): void
    {
        [$product1, $product2, $customer, $warehouse, $supplier] = $this->initialize();
        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'order_date' => now()->subMinute(),
            'deliver_by_date' => now()->addDays(2),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 3,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
                [
                    'product_id' => $product2->id,
                    'quantity' => 5,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        $response = $this->postJson('api/purchase-orders/build', [
            'supplier_id' => $supplier->id,
            'destination_warehouse_id' => $warehouse->id,
            'forecast_type' => PurchaseOrderBuilderFactory::BUILDER_TYPE_FILL_BACKORDERS,
            'options' => [
                'sales_history_filters' => [
                    'filters' => [
                        'conjunction' => 'and',
                        'filterSet' => [
                            [
                                'column' => 'deliver_by_date',
                                'operator' => '<',
                                'value' => [
                                    'mode' => 'oneWeekFromNow',
                                ],
                            ],
                        ],
                    ],
                ],
                'product_filters' => [],
            ],
        ])->assertSuccessful();

        // We should get the two products
        $response->assertJsonCount(2, 'data');

        $response = $this->postJson('api/purchase-orders/build', [
            'supplier_id' => $supplier->id,
            'destination_warehouse_id' => $warehouse->id,
            'forecast_type' => PurchaseOrderBuilderFactory::BUILDER_TYPE_FILL_BACKORDERS,
            'options' => [
                'sales_history_filters' => [
                    'filters' => [
                        'conjunction' => 'or',
                        'filterSet' => [
                            [
                                'column' => 'deliver_by_date',
                                'operator' => '=',
                                'value' => [
                                    'mode' => 'tomorrow',
                                ],
                            ],
                        ],
                    ],
                ],
                'product_filters' => [],
            ],
        ])->assertSuccessful();

        // We should get no response for receive by date of tomorrow
        $response->assertJsonCount(0, 'data');
    }

    public function test_it_can_fill_backorders_based_on_ship_by_date(): void
    {
        [$product1, $product2, $customer, $warehouse, $supplier] = $this->initialize();
        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'order_date' => now()->subMinute(),
            'ship_by_date' => now()->addDays(2),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 3,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
                [
                    'product_id' => $product2->id,
                    'quantity' => 5,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        $response = $this->postJson('api/purchase-orders/build', [
            'supplier_id' => $supplier->id,
            'destination_warehouse_id' => $warehouse->id,
            'forecast_type' => PurchaseOrderBuilderFactory::BUILDER_TYPE_FILL_BACKORDERS,
            'options' => [
                'sales_history_filters' => [
                    'filters' => [
                        'conjunction' => 'or',
                        'filterSet' => [
                            [
                                'column' => 'ship_by_date',
                                'operator' => '<',
                                'value' => [
                                    'mode' => 'oneWeekFromNow',
                                ],
                            ],
                        ],
                    ],
                ],
                'product_filters' => [],
            ],
        ])->assertSuccessful();

        // We should get the two products
        $response->assertJsonCount(2, 'data');

        $response = $this->postJson('api/purchase-orders/build', [
            'supplier_id' => $supplier->id,
            'destination_warehouse_id' => $warehouse->id,
            'forecast_type' => PurchaseOrderBuilderFactory::BUILDER_TYPE_FILL_BACKORDERS,
            'options' => [
                'sales_history_filters' => [
                    'filters' => [
                        'conjunction' => 'or',
                        'filterSet' => [
                            [
                                'column' => 'ship_by_date',
                                'operator' => '=',
                                'value' => [
                                    'mode' => 'tomorrow',
                                ],
                            ],
                        ],
                    ],
                ],
                'product_filters' => [],
            ],
        ])->assertSuccessful();

        // We should get no response for ship by date of tomorrow
        $response->assertJsonCount(0, 'data');
    }

    public function test_it_can_fill_backorders_based_on_ship_by_date_for_matching_orders(): void
    {
        [$product1, $_, $customer, $warehouse, $supplier] = $this->initialize();
        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'order_date' => now()->subMinute(),
            'ship_by_date' => now()->addDays(2),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 3,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'order_date' => now()->subMinute(),
            'ship_by_date' => now()->addMonths(2),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 5,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        $response = $this->postJson('api/purchase-orders/build', [
            'supplier_id' => $supplier->id,
            'destination_warehouse_id' => $warehouse->id,
            'forecast_type' => PurchaseOrderBuilderFactory::BUILDER_TYPE_FILL_BACKORDERS,
            'options' => [
                'sales_history_filters' => [
                    'filters' => [
                        'conjunction' => 'or',
                        'filterSet' => [
                            [
                                'column' => 'ship_by_date',
                                'operator' => '<',
                                'value' => [
                                    'mode' => 'oneWeekFromNow',
                                ],
                            ],
                        ],
                    ],
                ],
                'product_filters' => [],
            ],
        ])->assertSuccessful();

        // Only first sales order line's backorder should be accounted.
        $this->assertEquals(3, $response->json('data.0.quantity'));
    }

    public function test_it_can_fill_backorders(): void
    {
        [$product1, $product2, $customer, $warehouse, $supplier] = $this->initialize();

        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'order_date' => now()->subMinute(),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 3,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
                [
                    'product_id' => $product2->id,
                    'quantity' => 5,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        $response = $this->postJson('api/purchase-orders/build', [
            'supplier_id' => $supplier->id,
            'destination_warehouse_id' => $warehouse->id,
            'forecast_type' => PurchaseOrderBuilderFactory::BUILDER_TYPE_FILL_BACKORDERS,
            'options' => [
                'sales_history_filters' => [
                    'filters' => [
                        'conjunction' => 'or',
                        'filterSet' => [],
                    ],
                ],
                'product_filters' => [],
            ],
        ])->assertSuccessful();

        // Product 1 should be 3 units
        $this->assertEquals(3, $response->json('data.0.quantity'));
        // Product 2 should be 5 units
        $this->assertEquals(5, $response->json('data.1.quantity'));

        $response->assertJson(fn (AssertableJson $json) => $json->has('data.0.calculation', fn (AssertableJson $json) => $json->has('sample_days')
            ->has('num_sales')
            ->has('leadtime')
            ->has('num_backordered')
            ->has('num_ordered')
            ->has('num_stock')
            ->has('rounding_strategy')
            ->has('quantity_needed')
            ->has('quantity_calculated')
            ->etc()
        )->etc());
    }

    public function test_it_can_target_stock_level(): void
    {
        [$product1, $product2, $customer, $warehouse, $supplier] = $this->initialize();

        $product1->setInitialInventory($warehouse->id, 5);

        // Create sales order for the two products
        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'order_date' => now()->subMinute(),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 3,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
                [
                    'product_id' => $product2->id,
                    'quantity' => 5,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        // Product 2 has an incoming purchase order
        $response = $this->postJson('/api/purchase-orders', [
            'purchase_order_date' => now()->subMinute(),
            'destination_warehouse_id' => $warehouse->id,
            'approval_status' => PurchaseOrderValidator::APPROVAL_STATUS_APPROVED,
            'supplier_id' => $supplier->id,
            'currency_code' => 'USD',
            'purchase_order_lines' => [
                [
                    'product_id' => $product2->id,
                    'quantity' => 3,
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        $purchaseOrderId = $response->json('data.id');

        // Ship the PO
        $this->postJson('/api/purchase-order-shipments', [
            'purchase_order_id' => $purchaseOrderId,
            'shipment_date' => now()->format('Y-m-d H:i:s'),
        ])->assertSuccessful();

        $payload = [
            'supplier_id' => $supplier->id,
            'destination_warehouse_id' => $warehouse->id,
            'forecast_type' => PurchaseOrderBuilderFactory::BUILDER_TYPE_TARGET_STOCK_LEVEL,
            'options' => [
                'quantity' => 10,
                'product_filters' => [
                    'filters' => [
                        'conjunction' => 'or',
                        'filterSet' => [
                            [
                                'column' => 'id',
                                'operator' => '=',
                                'value' => $product1->id,
                            ],
                            [
                                'column' => 'id',
                                'operator' => '=',
                                'value' => $product2->id,
                            ],
                        ],
                    ],
                ],
            ],
        ];

        // Generate PO builder to target stock
        $response = $this->postJson('api/purchase-orders/build', $payload)->assertSuccessful();

        // Assert export works as well
        $this->postJson("api/purchase-orders/build-export", $payload)->assertSuccessful()->assertJson(fn (AssertableJson $json) => $json->has('data')->etc());

        // Product 1 should be 8 units
        // 5 initial, 3 reserved so 2 available
        $this->assertEquals(8, $response->json('data.0.quantity'));

        // Product 2 should be 12 units
        // 5 reserved, 3 incoming so -2 available
        $this->assertEquals(12, $response->json('data.1.quantity'));

        $response->assertJson(fn (AssertableJson $json) => $json->has('data.0.calculation', fn (AssertableJson $json) => $json->has('sample_days')
            ->has('num_sales')
            ->has('target_days_of_stock')
            ->has('leadtime')
            ->has('daily_average_consumption')
            ->has('num_backordered')
            ->has('num_ordered')
            ->has('num_stock')
            ->has('rounding_strategy')
            ->has('quantity_needed')
            ->has('quantity_calculated')
            ->etc()
        )->etc());
    }

    public function test_it_can_forecast_including_kit_sales_for_components(): void
    {
        // Build kit and components

        Queue::fake();

        /** @var Product $kit */
        $kit = Product::factory()->create([
            'sku' => 'KIT-1',
            'name' => 'KIT 1',
            'type' => Product::TYPE_KIT,
        ]);

        $defaultPricingTier = ProductPricingTier::default();

        $kit->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 100,
            ],
        ]);

        /** @var Product $component1 */
        $component1 = Product::factory()->create([
            'sku' => 'COMPONENT-1',
            'name' => 'Component 1',
            'type' => Product::TYPE_STANDARD,
        ]);

        $component1->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 30,
            ],
        ]);

        /** @var Product $component2 */
        $component2 = Product::factory()->create([
            'sku' => 'COMPONENT-2',
            'name' => 'Component 2',
            'type' => Product::TYPE_STANDARD,
        ]);

        $component2->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 20,
            ],
        ]);

        $kit->setBundleComponents([
            [
                'id' => $component1->id,
                'quantity' => 2,
            ],
            [
                'id' => $component2->id,
                'quantity' => 3,
            ],
        ]);

        /** @var User $user */
        $user = User::query()->first();

        Sanctum::actingAs($user);

        $response = $this->postJson('/api/sales-orders', [
            'order_status' => 'open',
            'currency_code' => 'USD',
            'order_date' => Carbon::now(),
            'customer_id' => Customer::factory()->create()->id,
            'sales_order_lines' => [
                [
                    'product_id' => $kit->id,
                    'description' => $kit->name,
                    'is_product' => true,
                    'quantity' => 5,
                    'amount' => 150, // Purposely set to different price than bundle product
                ],
            ],
        ])->assertOk();

        App::make(SalesOrderLineFinancialManager::class)->calculate();

        $purchaseOrderBuilderRepository = app(PurchaseOrderBuilderRepository::class);

        $this->assertEquals(5, $purchaseOrderBuilderRepository->getAverageDailySales($kit));
        $this->assertEquals(10, $purchaseOrderBuilderRepository->getAverageDailySales($component1));
        $this->assertEquals(15, $purchaseOrderBuilderRepository->getAverageDailySales($component2));
    }

    public function test_it_can_forecast_including_both_kit_and_component_sales(): void
    {
        // Build kit and components

        Queue::fake();

        /** @var Product $kit */
        $kit = Product::factory()->create([
            'sku' => 'KIT-A1',
            'name' => 'KIT 1',
            'type' => Product::TYPE_KIT,
        ]);

        $defaultPricingTier = ProductPricingTier::default();

        $kit->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 100,
            ],
        ]);

        /** @var Product $component1 */
        $component1 = Product::factory()->create([
            'sku' => 'COMPONENT-A1',
            'name' => 'Component 1',
            'type' => Product::TYPE_STANDARD,
        ]);

        $component1->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 30,
            ],
        ]);

        /** @var Product $component2 */
        $component2 = Product::factory()->create([
            'sku' => 'COMPONENT-A2',
            'name' => 'Component 2',
            'type' => Product::TYPE_STANDARD,
        ]);

        $component2->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 20,
            ],
        ]);

        $kit->setBundleComponents([
            [
                'id' => $component1->id,
                'quantity' => 2,
            ],
            [
                'id' => $component2->id,
                'quantity' => 3,
            ],
        ]);

        /** @var User $user */
        $user = User::query()->first();

        Sanctum::actingAs($user);

        $this->postJson('/api/sales-orders', [
            'order_status' => 'open',
            'currency_code' => 'USD',
            'order_date' => Carbon::now(),
            'customer_id' => Customer::factory()->create()->id,
            'sales_order_lines' => [
                [
                    'product_id' => $kit->id,
                    'description' => $kit->name,
                    'is_product' => true,
                    'quantity' => 5,
                    'amount' => 150, // Purposely set to different price than bundle product
                ],
                [
                    'product_id' => $component1->id,
                    'description' => $component1->name,
                    'is_product' => true,
                    'quantity' => 3,
                    'amount' => 30,
                ],
            ],
        ])->assertOk();

        App::make(SalesOrderLineFinancialManager::class)->calculate();

        $purchaseOrderBuilderRepository = app(PurchaseOrderBuilderRepository::class);

        $this->assertEquals(5, $purchaseOrderBuilderRepository->getAverageDailySales($kit));
        $this->assertEquals(13, $purchaseOrderBuilderRepository->getAverageDailySales($component1));
        $this->assertEquals(15, $purchaseOrderBuilderRepository->getAverageDailySales($component2));
    }

    public function test_it_only_returns_orders_with_deliver_by_date_set_when_the_filter_is_available(): void
    {
        [$product1, $product2, $customer, $warehouse, $supplier] = $this->initialize();
        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'order_date' => now()->subMinute(),
            'deliver_by_date' => now()->addDays(2),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 3,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'order_date' => now()->subMinute(),
            'sales_order_lines' => [
                [
                    'product_id' => $product2->id,
                    'quantity' => 5,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        $response = $this->postJson('api/purchase-orders/build', [
            'supplier_id' => $supplier->id,
            'destination_warehouse_id' => $warehouse->id,
            'forecast_type' => PurchaseOrderBuilderFactory::BUILDER_TYPE_FILL_BACKORDERS,
            'options' => [
                'sales_history_filters' => [
                    'filters' => [
                        'conjunction' => 'and',
                        'filterSet' => [
                            [
                                'column' => 'deliver_by_date',
                                'operator' => '>',
                                'value' => [
                                    'mode' => 'today',
                                ],
                            ],
                        ],
                    ],
                ],
                'product_filters' => [],
            ],
        ])->assertSuccessful();

        // We should get the two products
        $response->assertJsonCount(1, 'data');

        $response = $this->postJson('api/purchase-orders/build', [
            'supplier_id' => $supplier->id,
            'destination_warehouse_id' => $warehouse->id,
            'forecast_type' => PurchaseOrderBuilderFactory::BUILDER_TYPE_FILL_BACKORDERS,
            'options' => [
                'sales_history_filters' => [
                    'filters' => [
                        'conjunction' => 'or',
                        'filterSet' => [
                            [
                                'column' => 'deliver_by_date',
                                'operator' => '=',
                                'value' => [
                                    'mode' => 'tomorrow',
                                ],
                            ],
                        ],
                    ],
                ],
                'product_filters' => [],
            ],
        ])->assertSuccessful();

        // We should get no response for receive by date of tomorrow
        $response->assertJsonCount(0, 'data');
    }

    public function test_it_can_fill_backorders_accounting_for_parent_product_backordered_quantities(): void
    {
        [$product1, $product2, $customer, $warehouse, $supplier] = $this->initialize();

        /** @var Product $kit */
        $kit = Product::factory()->create([
            'sku' => 'KIT-B1',
            'name' => 'KIT 1',
            'type' => Product::TYPE_KIT,
        ]);

        $kit->setBundleComponents([
            [
                'id' => $product1->id,
                'quantity' => 3,
            ],
            [
                'id' => $product2->id,
                'quantity' => 5,
            ],
        ]);

        /** @var ProductPricingTier $defaultPricingTier */
        $defaultPricingTier = ProductPricingTier::default();
        $kit->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 100,
            ],
        ]);

        // Component sales
        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'order_date' => now()->subMinute(),
            'deliver_by_date' => now()->addDays(2),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 30,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
                [
                    'product_id' => $product2->id,
                    'quantity' => 10,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        // Kit Sales
        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'order_date' => now()->subMinute(),
            'deliver_by_date' => now()->addDays(2),
            'sales_order_lines' => [
                [
                    'product_id' => $kit->id,
                    'quantity' => 100,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        $response = $this->postJson('api/purchase-orders/build', [
            'supplier_id' => $supplier->id,
            'destination_warehouse_id' => $warehouse->id,
            'forecast_type' => PurchaseOrderBuilderFactory::BUILDER_TYPE_FILL_BACKORDERS,
            'options' => [
                'product_filters' => [],
            ],
        ])->assertSuccessful();

        // We should get the two products
        $response->assertJsonCount(2, 'data');

        // Component 1 should have 330 (30 + 100 * 3) units
        $this->assertEquals(330, $response->json('data.0.quantity'));

        // Component 2 should have 510 (10 + 100 * 5)
        $this->assertEquals(510, $response->json('data.1.quantity'));
    }

    public function test_it_can_filter_by_deliver_by_date_when_multiple_sales_orders_match_filter_for_same_product(): void
    {
        [$product1, $product2, $customer, $warehouse, $supplier] = $this->initialize();
        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'order_date' => now()->subMinute(),
            'deliver_by_date' => now()->addDays(2),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 3,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'order_date' => now()->subMinute(),
            'deliver_by_date' => now()->addDays(2),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 5,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        $response = $this->postJson('api/purchase-orders/build', [
            'supplier_id' => $supplier->id,
            'destination_warehouse_id' => $warehouse->id,
            'forecast_type' => PurchaseOrderBuilderFactory::BUILDER_TYPE_FILL_BACKORDERS,
            'options' => [
                'sales_history_filters' => [
                    'filters' => [
                        'conjunction' => 'and',
                        'filterSet' => [
                            [
                                'column' => 'deliver_by_date',
                                'operator' => '>',
                                'value' => [
                                    'mode' => 'today',
                                ],
                            ],
                        ],
                    ],
                ],
                'product_filters' => [],
            ],
        ])->assertSuccessful();

        // We should get the two products
        $response->assertJsonCount(2, 'data');

        $this->assertEquals(8, collect($response->json('data'))->sum('quantity'));
    }

    public function test_it_can_fill_backorders_based_on_ship_by_date_and_exclude_already_covered_backorders(): void
    {
        [$product1, $_, $customer, $warehouse, $supplier] = $this->initialize();

        // To be shipped later
        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'order_date' => now()->subMinute(),
            'ship_by_date' => now()->addWeeks(2),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 1,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        // Shipped within filter date
        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'order_date' => now()->subMinute(),
            'ship_by_date' => now()->addDays(2),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 3,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        $this->assertDatabaseEmpty('backorder_queue_coverages');

        // PO to cover backorder
        $this->postJson('/api/purchase-orders', [
            'purchase_order_date' => now()->subMinute(),
            'destination_warehouse_id' => $warehouse->id,
            'approval_status' => PurchaseOrderValidator::APPROVAL_STATUS_APPROVED,
            'supplier_id' => $supplier->id,
            'currency_code' => 'USD',
            'purchase_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 4,
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        (new App\Jobs\SyncBackorderQueueCoveragesJob())->withGlobalCoverage()->handle();

        // Backorder queue should be covered
        $this->assertDatabaseCount('backorder_queue_coverages', 2);

        // Another sales order to be shipped within filter date
        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'order_date' => now()->subMinute(),
            'ship_by_date' => now()->addDays(2),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 2,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        $response = $this->postJson('api/purchase-orders/build', [
            'supplier_id' => $supplier->id,
            'destination_warehouse_id' => $warehouse->id,
            'forecast_type' => PurchaseOrderBuilderFactory::BUILDER_TYPE_FILL_BACKORDERS,
            'options' => [
                'sales_history_filters' => [
                    'filters' => [
                        'conjunction' => 'or',
                        'filterSet' => [
                            [
                                'column' => 'ship_by_date',
                                'operator' => '<',
                                'value' => [
                                    'mode' => 'oneWeekFromNow',
                                ],
                            ],
                        ],
                    ],
                ],
                'product_filters' => [],
            ],
        ])->assertSuccessful();

        // Only uncovered backorder quantities should be returned.
        $this->assertEquals(2, $response->json('data.0.quantity'));
    }

    public function test_it_can_add_sales_order_lines_to_response(): void
    {
        [$product1, $_, $customer, $warehouse, $supplier] = $this->initialize();

        SalesOrder::query()->each(function (SalesOrder $salesOrder) {
            $salesOrder->delete();
        });

        // To be shipped later
        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'order_date' => now()->subMinute(),
            'ship_by_date' => now()->addDays(2),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 1,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        // Shipped within filter date
        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'order_date' => now()->subMinute(),
            'ship_by_date' => now()->addDays(2),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 3,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        $response = $this->postJson('api/purchase-orders/build', [
            'supplier_id' => $supplier->id,
            'destination_warehouse_id' => $warehouse->id,
            'forecast_type' => PurchaseOrderBuilderFactory::BUILDER_TYPE_FILL_BACKORDERS,
            'options' => [
                'sales_history_filters' => [
                    'filters' => [
                        'conjunction' => 'or',
                        'filterSet' => [
                            [
                                'column' => 'ship_by_date',
                                'operator' => '<',
                                'value' => [
                                    'mode' => 'oneWeekFromNow',
                                ],
                            ],
                        ],
                    ],
                ],
                'product_filters' => [],
            ],
        ])->assertSuccessful();

        $this->assertEquals(1, $response->json('data.0.sales_order_line.quantity'));
        $this->assertEquals(3, $response->json('data.1.sales_order_line.quantity'));
    }

    public function test_it_can_cover_attached_backorder_queue_for_purchase_order_line(): void
    {
        [$product1, $_, $customer, $warehouse, $supplier] = $this->initialize();

        PurchaseOrder::query()->each(function (PurchaseOrder $purchaseOrder) {
            $purchaseOrder->delete();
        });

        // Create a sales order
        $response = $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'order_date' => now()->subMinute(),
            'ship_by_date' => now()->addDays(2),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 3,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        $backorderQueueId = $response->json('data.item_info.0.backorder_queue_id');

        // Create a PO tied to sales order
        $response = $this->postJson('/api/purchase-orders', [
            'purchase_order_date' => now()->subMinute(),
            'destination_warehouse_id' => $warehouse->id,
            'supplier_id' => $supplier->id,
            'currency_code' => 'USD',
            'purchase_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 2,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'linked_backorders' => [
                        [
                            'id' => $backorderQueueId,
                            'quantity' => 2,
                        ],
                    ],
                ],
            ],
        ])->assertSuccessful();

        // Backorder info should be stored in purchase order lines table
        $this->assertDatabaseMissing('purchase_order_lines', [
            'linked_backorders' => null,
        ]);

        // Approving the PO should cover the linked backorders
        // with an indication that the coverage must release the PO.
        $this->putJson('/api/purchase-orders/'.$response->json('data.id'), [
            'approval_status' => PurchaseOrderValidator::APPROVAL_STATUS_APPROVED,
        ])->assertSuccessful();

        $this->assertDatabaseCount('backorder_queue_coverages', 1);
        $this->assertDatabaseHas('backorder_queue_coverages', [
            'backorder_queue_id' => $backorderQueueId,
            'covered_quantity' => 2,
            'is_tight' => true,
        ]);

        // Receive PO and ensure that it releases tight coverages first.
        $this->postJson('/api/purchase-order-shipments/receive', [
            'purchase_order_id' => $response->json('data.id'),
            'received_at' => now(),
            'receipt_lines' => [
                [
                    'purchase_order_line_id' => $response->json('data.items.0.id'),
                    'quantity' => 2,
                ],
            ],
        ])->assertSuccessful();

        // Sync backorder coverages & releases
        (new App\Jobs\SyncBackorderQueueCoveragesJob())->withGlobalCoverage()->handle();

        // Coverage and releases should be updated
        $this->assertDatabaseHas('backorder_queues', [
            'id' => $backorderQueueId,
            'backordered_quantity' => 3,
            'released_quantity' => 2,
        ]);
        $this->assertDatabaseHas('backorder_queue_coverages', [
            'covered_quantity' => 2,
            'released_quantity' => 2,
            'backorder_queue_id' => $backorderQueueId,
        ]);

        $this->assertDatabaseHas('backorder_queue_releases', [
            'link_type' => App\Models\PurchaseOrderShipmentReceiptLine::class,
            'released_quantity' => 2,
            'backorder_queue_id' => $backorderQueueId,
        ]);
    }

    public function test_that_other_positive_events_dont_release_tight_coverages(): void
    {
        [$product1, $_, $customer, $warehouse, $supplier] = $this->initialize();

        PurchaseOrder::query()->each(function (PurchaseOrder $purchaseOrder) {
            $purchaseOrder->delete();
        });

        // Create a sales order
        $response = $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'order_date' => now()->subMinute(),
            'ship_by_date' => now()->addDays(2),
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 3,
                    'warehouse_id' => $warehouse->id,
                    'description' => $this->faker->sentence(),
                    'amount' => $this->faker->numberBetween(5, 10),
                ],
            ],
        ])->assertSuccessful();

        $backorderQueueId = $response->json('data.item_info.0.backorder_queue_id');

        // Create a PO tied to sales order
        $response = $this->postJson('/api/purchase-orders', [
            'purchase_order_date' => now()->subMinute(),
            'destination_warehouse_id' => $warehouse->id,
            'supplier_id' => $supplier->id,
            'currency_code' => 'USD',
            'purchase_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'quantity' => 2,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'linked_backorders' => [
                        [
                            'id' => $backorderQueueId,
                            'quantity' => 2,
                        ],
                    ],
                ],
            ],
        ])->assertSuccessful();

        // Backorder info should be stored in purchase order lines table
        $this->assertDatabaseMissing('purchase_order_lines', [
            'linked_backorders' => null,
        ]);

        // Approving the PO should cover the linked backorders
        // with an indication that the coverage must release the PO.
        $this->putJson('/api/purchase-orders/'.$response->json('data.id'), [
            'approval_status' => PurchaseOrderValidator::APPROVAL_STATUS_APPROVED,
        ])->assertSuccessful();

        $this->assertDatabaseCount('backorder_queue_coverages', 1);
        $this->assertDatabaseHas('backorder_queue_coverages', [
            'backorder_queue_id' => $backorderQueueId,
            'covered_quantity' => 2,
            'is_tight' => true,
        ]);

        // Add positive adjustment
        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => now(),
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'product_id' => $product1->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 3,
            'unit_cost' => 10,
        ])->assertSuccessful();

        // Sync backorder queues and run release script.
        (new App\Jobs\SyncBackorderQueueCoveragesJob())->withGlobalCoverage()->handle();

        // Coverage and releases should not be updated
        $this->assertDatabaseHas('backorder_queues', [
            'id' => $backorderQueueId,
            'backordered_quantity' => 3,
            'released_quantity' => 0,
        ]);
        $this->assertDatabaseHas('backorder_queue_coverages', [
            'covered_quantity' => 2,
            'released_quantity' => 0,
            'backorder_queue_id' => $backorderQueueId,
        ]);

        $this->assertDatabaseEmpty('backorder_queue_releases');
    }
}
