<?php

namespace Tests\Feature;

use App\Managers\ProductListingManager;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Shopify\ShopifyOrder;
use App\Models\Shopify\ShopifyProduct;
use App\Models\User;
use Illuminate\Foundation\Testing\WithFaker;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class ProductListingManagerTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    private ProductListingManager $productListingManager;

    protected function setUp(): void
    {
        parent::setUp();

        $this->productListingManager = new ProductListingManager();
    }

    public function test_shopify_map_sales_order_lines()
    {
        //Get integration
        $integration = Integration::whereName(Integration::NAME_SHOPIFY)->first();

        //Create shopify integration instance
        $shopifyIntegrationInstance = IntegrationInstance::factory()->hasSalesChannel()->for($integration)->create();
        $shopifyIntegrationInstance->load('salesChannel');

        //Login as user
        Sanctum::actingAs(User::factory()->create());

        //Create shopify products
        $shopifyProductData = '{"id":44742653641013,"title":"Baby Bumble Onesie","body_html":"<meta charset=\"utf-8\">\n<p data-pm-slice=\"1 3 []\"><strong>Soft and comfortable onesie for newborns<\/strong><\/p>\n<ul>\n<li>\n<p>Fits Newborn<\/p>\n<\/li>\n<\/ul>","vendor":"Mamimoo Baby Clothing","product_type":"","created_at":"2023-03-26T22:07:43-04:00","handle":"baby-bumble-onesie","updated_at":"2023-05-09T08:00:14-04:00","published_at":"2023-03-26T22:07:43-04:00","template_suffix":"","status":"active","published_scope":"web","tags":"","admin_graphql_api_id":"gid:\/\/shopify\/ProductVariant\/44742653641013","options":[{"id":10406532120885,"product_id":8201861366069,"name":"Title","position":1,"values":["Default Title"]}],"images":[{"id":40951766483253,"product_id":8201861366069,"position":1,"created_at":"2023-03-26T22:07:44-04:00","updated_at":"2023-03-26T22:07:44-04:00","alt":null,"width":512,"height":512,"src":"https:\/\/cdn.shopify.com\/s\/files\/1\/0741\/7420\/2165\/products\/AHjexCj.jpg?v=1679882864","variant_ids":[],"admin_graphql_api_id":"gid:\/\/shopify\/ProductImage\/40951766483253"}],"image":"https:\/\/cdn.shopify.com\/s\/files\/1\/0741\/7420\/2165\/products\/AHjexCj.jpg?v=1679882864","product_id":8201861366069,"price":15.99,"sku":"BB001","position":1,"inventory_policy":"continue","compare_at_price":null,"fulfillment_service":"manual","inventory_management":"shopify","option1":"Default Title","option2":null,"option3":null,"taxable":false,"barcode":"123456789012","grams":454,"image_id":null,"weight":1,"weight_unit":"lb","inventory_item_id":46791574946101,"inventory_quantity":-1,"old_inventory_quantity":-1,"requires_shipping":true,"integration_instance_id":3,"image_count":1,"variant_title":"Baby Bumble Onesie","removed_from_shopify":0,"product_title":"Baby Bumble Onesie"}';
        $shopifyProduct1 = ShopifyProduct::factory()->create([
            'variant_id' => $this->faker->ean13(),
            'unit_cost' => 5,
            'integration_instance_id' => $shopifyIntegrationInstance->id,
            'json_object' => json_decode($shopifyProductData, true),
        ]);
        $shopifyProduct1->refresh();
        $shopifyProduct2 = ShopifyProduct::factory()->create([
            'variant_id' => $this->faker->ean13(),
            'unit_cost' => 2,
            'integration_instance_id' => $shopifyIntegrationInstance->id,
            'json_object' => json_decode($shopifyProductData, true),
        ]);
        $shopifyProduct2->refresh();

        //Create shopify orders
        $shopifyOrder1 = new ShopifyOrder([
            'integration_instance_id' => $shopifyIntegrationInstance->id,
            'order_number' => 'ORDER - 1',
            'json_object' => [
                'name' => 'ORDER - 1',
                'fulfillment_status' => 'partial',
                'financial_status' => 'paid',
                'line_items' => [
                    [
                        'id' => 1,
                        'sku' => $shopifyProduct1->sku,
                        'product_exists' => true,
                        'quantity' => 5,
                        'name' => $this->faker->sentence(),
                        'variant_id' => $shopifyProduct1->variant_id,
                        'is_product' => true,
                    ],
                ],
                'refunds' => [],
                'fulfillments' => [
                    [
                        'id' => 1,
                        'status' => 'success',
                        'tracking_company' => 'USPS',
                        'tracking_numbers' => ['1Z2345'],
                        'line_items' => [
                            [
                                'id' => 1,
                                'quantity' => 2,
                                'sku' => 'SKU-1',
                                'variant_id' => 1,
                                'product_exists' => true,
                            ],
                        ],
                        'created_at' => now()->subDay(),
                        'updated_at' => now()->subDay(),
                    ],
                ],
            ],
        ]);
        $shopifyOrder1->save();
        $salesOrder1 = SalesOrder::factory()->for($shopifyIntegrationInstance->salesChannel)->create([
            'sales_channel_order_type' => ShopifyOrder::class,
            'sales_channel_order_id' => $shopifyOrder1->id,
        ]);
        SalesOrderLine::factory()
            ->for($salesOrder1)
            ->create([
                'sales_channel_line_id' => '1',
                'product_id' => null,
            ]);

        $shopifyOrder2 = new ShopifyOrder([
            'integration_instance_id' => $shopifyIntegrationInstance->id,
            'order_number' => 'ORDER - 2',
            'json_object' => [
                'name' => 'ORDER - 2',
                'fulfillment_status' => 'partial',
                'financial_status' => 'paid',
                'line_items' => [
                    [
                        'id' => 2,
                        'sku' => $shopifyProduct2->sku,
                        'product_exists' => true,
                        'quantity' => 5,
                        'name' => $this->faker->sentence(),
                        'variant_id' => $shopifyProduct2->variant_id,
                        'is_product' => true,
                    ],
                ],
                'refunds' => [],
                'fulfillments' => [
                    [
                        'id' => 1,
                        'status' => 'success',
                        'tracking_company' => 'USPS',
                        'tracking_numbers' => ['1Z2345'],
                        'line_items' => [
                            [
                                'id' => 1,
                                'quantity' => 2,
                                'sku' => 'SKU-1',
                                'variant_id' => 1,
                                'product_exists' => true,
                            ],
                        ],
                        'created_at' => now()->subDay(),
                        'updated_at' => now()->subDay(),
                    ],
                ],
            ],
        ]);
        $shopifyOrder2->save();
        $salesOrder2 = SalesOrder::factory()->for($shopifyIntegrationInstance->salesChannel)->create([
            'sales_channel_order_type' => ShopifyOrder::class,
            'sales_channel_order_id' => $shopifyOrder2->id,
        ]);
        SalesOrderLine::factory()
            ->for($salesOrder2)
            ->create([
                'sales_channel_line_id' => '2',
                'product_id' => null,
            ]);

        //Create sku products
        $product1 = Product::factory()->create();
        $product2 = Product::factory()->create();
        $product3 = Product::factory()->create();

        //Create request body
        $mappings = [
            [
                'sales_channel_listing_id' => $shopifyProduct1->variant_id,
                'mapped_sku' => $product1->sku,
            ],
            [
                'sales_channel_listing_id' => $shopifyProduct2->variant_id,
                'mapped_sku' => $product2->sku,
            ],
        ];

        //Main testing start here
        $this->putJson("/api/shopify/$shopifyIntegrationInstance->id/products/map", [
            'mapping' => $mappings,
        ]);

        $mappings = [
            [
                'sales_channel_listing_id' => $shopifyProduct1->variant_id,
                'mapped_sku' => $product3->sku,
            ],
        ];

        $this->putJson("/api/shopify/$shopifyIntegrationInstance->id/products/map", [
            'mapping' => $mappings,
        ]);

        $this->assertFalse(SalesOrderLine::where('product_id', $product3->id)->exists());
        $this->assertTrue(SalesOrderLine::where('product_id', $product1->id)->exists());
        $this->assertTrue(SalesOrderLine::where('product_id', $product2->id)->exists());
    }
}
