<?php

namespace Tests\Feature;

use App\DTO\DataImporterDto;
use App\Exceptions\DataImportNoRecordsImportedException;
use App\Exceptions\DataImportTooLargeException;
use App\Exceptions\DataImportValidationErrorsException;
use App\Exceptions\ImportFailedException;
use App\Managers\DataImportManager;
use App\Models\Product;
use App\Models\ProductComponent;
use App\Models\Tag;
use App\Models\User;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Queue;
use Illuminate\Support\Facades\Storage;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;


class ProductImportTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    /**
     * @throws DataImportValidationErrorsException
     * @throws DataImportTooLargeException
     * @throws DataImportNoRecordsImportedException
     * @throws ImportFailedException
     */
    public function test_it_can_import_products(): void
    {
        $data = [
            ['sku', 'brand', 'mpn', 'default_supplier_sku'],
            ['ABC', 'brand', 'mpn', 'ABC123'],
            ['DEF', 'brand', 'mpn', 'ABC123'],
        ];

        $fp = fopen('php://temp', 'w+');
        foreach ($data as $fields) {
            fputcsv($fp, $fields);
        }
        rewind($fp);

        $fileContents = stream_get_contents($fp);

        $filename = 'products.csv';
        Storage::disk('model-imports')->put($filename, $fileContents);

        $importManager = (new DataImportManager((new Product()), DataImporterDto::from([
            'stored_name' => $filename,
            'task_id' => 1,
        ])));

        $importManager->import();

        $this->assertDatabaseCount((new Product())->getTable(), 2);
    }

    public function test_import_syntax_error_handles_exception_through_controller(): void
    {
        $data = [
            ['sku', 'brand', 'mpn', 'default_supplier_sku', '', ''], // Duplicate column names
            ['ABC', 'brand', 'mpn', 'ABC123', '', ''],
            ['DEF', 'brand', 'mpn', 'ABC123', '', ''],
        ];

        $fp = fopen('php://temp', 'w+');
        foreach ($data as $fields) {
            fputcsv($fp, $fields);
        }
        rewind($fp);

        $fileContents = stream_get_contents($fp);

        $filename = 'products.csv';
        Storage::disk('model-imports')->put($filename, $fileContents);

        Sanctum::actingAs(User::first());

        $response = $this->post('/api/products/import', [
            'original_name' => $filename,
            'stored_name' => $filename,
            'task_id' => 1,
            'csv_enclosure' => '\n',
            'csv_delimiter' => ',',
            'override' => 1,
        ])->assertBadRequest();
    }

    public function test_it_can_import_product_tags(): void
    {
        $tag1 = 'tag 1';
        $tag2 = 'tag 2';

        $data = [
            ['sku', 'tags', 'brand', 'mpn', 'default_supplier_sku'],
            ['ABC', "$tag1|$tag2", 'brand', 'mpn', 'ABC123'],
        ];

        $fp = fopen('php://temp', 'w+');
        foreach ($data as $fields) {
            fputcsv($fp, $fields);
        }
        rewind($fp);

        $fileContents = stream_get_contents($fp);

        $filename = 'products.csv';
        Storage::disk('model-imports')->put($filename, $fileContents);

        Sanctum::actingAs(User::first());

        $this->post('/api/products/import', [
            'original_name' => $filename,
            'stored_name' => $filename,
            'task_id' => 1,
            'csv_enclosure' => '\n',
            'csv_delimiter' => ',',
            'override' => 1,
        ])->assertOk();

        /** @var Product $product */
        $product = Product::query()->first();

        $this->assertEquals($tag1, $product->tags->offsetGet(0)->name);
        $this->assertEquals($tag2, $product->tags->offsetGet(1)->name);

        /*
        |--------------------------------------------------------------------------
        | Make sure it won't duplicate tags
        |--------------------------------------------------------------------------
        */

        $this->post('/api/products/import', [
            'original_name' => $filename,
            'stored_name' => $filename,
            'task_id' => 1,
            'csv_enclosure' => '\n',
            'csv_delimiter' => ',',
            'override' => 1,
        ])->assertOk();

        $product->refresh();

        $this->assertDatabaseCount((new Tag())->getTable(), 2);
        $this->assertDatabaseCount('taggables', 2);
        $this->assertEquals($tag1, $product->tags->offsetGet(0)->name);
        $this->assertEquals($tag2, $product->tags->offsetGet(1)->name);

        Storage::disk('model-imports')->delete($filename);
    }

    public function test_importing_products_with_component_relationships(): void
    {
        // Create a bundle product
        $bundle = Product::factory()->create([
            'sku' => '807-159-101_RAMAIR',
            'name' => '807-159-101_RAMAIR',
            'type' => Product::TYPE_BUNDLE,
        ]);

        // Create a kit product
        $kit = Product::factory()->create([
            'sku' => 'D',
            'name' => 'D',
            'type' => Product::TYPE_KIT,
        ]);
        // test Import to create relationships through parents and children

        // Prepare test data in CSV format
        $data = [
            ['sku', 'name', 'type', 'component_skus', 'component_of_sku'],
            ['A', 'A', 'standard', null, 'C->1,807-159-101_RAMAIR->1'],
            ['B', 'B', 'standard', null, 'C->3,D->2'],
            ['C', 'C', 'bundle', 'A->1', null],
            ['D', 'D', 'kit', 'A->1,B->2', null],
        ];

        // Write test data to a temporary file
        $fp = fopen('php://temp', 'w+');
        foreach ($data as $fields) {
            fputcsv($fp, $fields);
        }
        rewind($fp);

        $fileContents = stream_get_contents($fp);
        $filename = 'products.csv';

        // Store the test file on the disk
        Storage::disk('model-imports')->put($filename, $fileContents);

        // Simulate an authenticated user
        Sanctum::actingAs(User::factory()->create());

        // Send a POST request to import products
        $this->post('/api/products/import', [
            'original_name' => $filename,
            'stored_name' => $filename,
            'task_id' => 1,
            'csv_enclosure' => '\n',
            'csv_delimiter' => ',',
            'override' => 1,
        ])->assertOk();

        // Assertions for the imported products and components

        // Assert standard products
        $this->assertDatabaseHas('products', [
            'sku' => 'A',
            'type' => 'standard',
        ]);

        $this->assertDatabaseHas('products', [
            'sku' => 'B',
            'type' => 'standard',
        ]);

        // Assert bundle product
        $this->assertDatabaseHas('products', [
            'sku' => 'C',
            'type' => 'bundle',
        ]);

        // Assert kit product
        $this->assertDatabaseHas('products', [
            'sku' => 'D',
            'type' => 'kit',
        ]);

        // Assertions for the product components

        // Assert component of SKU 'C' and quantity 1
        $this->assertDatabaseHas('product_components', [
            'parent_product_id' => Product::where('sku', 'C')->first()->id,
            'component_product_id' => Product::where('sku', 'A')->first()->id,
            'quantity' => 1,
        ]);

        // Assert components of SKU 'D' with quantities 1 and 2
        $this->assertDatabaseHas('product_components', [
            'parent_product_id' => Product::where('sku', 'D')->first()->id,
            'component_product_id' => Product::where('sku', 'A')->first()->id,
            'quantity' => 1,
        ]);

        $this->assertDatabaseHas('product_components', [
            'parent_product_id' => Product::where('sku', 'D')->first()->id,
            'component_product_id' => Product::where('sku', 'B')->first()->id,
            'quantity' => 2,
        ]);

        // Test create relationships through children
        // Assert component of bundle product with quantity 1
        $this->assertDatabaseHas('product_components', [
            'parent_product_id' => $bundle->id,
            'component_product_id' => Product::where('sku', 'A')->first()->id,
            'quantity' => 1,
        ]);

        // Additional assertions for component counts

        // Assert the number of components for the bundle product
        $this->assertEquals(1, ProductComponent::where('parent_product_id', $bundle->id)->count());

        // Assert the number of components for SKU 'C' product
        $this->assertEquals(1, ProductComponent::where('parent_product_id', Product::where('sku', 'C')->first()->id)->count());

        // Assert the number of components for SKU 'D' product
        $this->assertEquals(2, ProductComponent::where('parent_product_id', Product::where('sku', 'D')->first()->id)->count());

        // Import again to remove relationships

        // Update test data to remove component relationships
        $data = [
            ['sku', 'name', 'type', 'component_skus', 'component_of_sku'],
            ['A', 'A', 'standard', null, 'C->1,807-159-101_RAMAIR->1'],
            ['B', 'B', 'standard', null, 'C->3,D->2'],
            ['C', 'C', 'bundle', null, null],
            ['D', 'D', 'kit', null, null],
        ];

        // Write updated test data to a temporary file
        $fp = fopen('php://temp', 'w+');
        foreach ($data as $fields) {
            fputcsv($fp, $fields);
        }
        rewind($fp);

        $fileContents = stream_get_contents($fp);
        $filename = 'products.csv';

        // Store the updated test file on the disk
        Storage::disk('model-imports')->put($filename, $fileContents);

        // Send a POST request to import products again
        $this->post('/api/products/import', [
            'original_name' => $filename,
            'stored_name' => $filename,
            'task_id' => 1,
            'csv_enclosure' => '\n',
            'csv_delimiter' => ',',
            'override' => 1,
        ])->assertOk();

        // Assertions for the imported products and components after removing relationships

        // Assert bundle product without components
        $this->assertDatabaseHas('products', [
            'sku' => 'C',
            'type' => 'bundle',
        ]);

        // Assert kit product without components
        $this->assertDatabaseHas('products', [
            'sku' => 'D',
            'type' => 'kit',
        ]);

        // Additional assertions for component counts after removing relationships

        // Assert the number of components for the bundle product is 0
        $this->assertEquals(0, ProductComponent::where('parent_product_id', Product::where('sku', 'D')->first()->id)->count());

        // Assert the number of components for SKU 'C' product is 0
        $this->assertEquals(0, ProductComponent::where('parent_product_id', Product::where('sku', 'C')->first()->id)->count());

        // Assert the number of components for SKU 'D' product is 0
        $this->assertEquals(0, ProductComponent::where('parent_product_id', Product::where('sku', 'D')->first()->id)->count());

        /*
         * Test Summary:
         * 1. Import to create relationships through parents
         * 2. Import to create relationships through children
         * 3. Import to remove relationships
         * With appropriate assertions throughout
         */
    }
}
