<?php

namespace Tests\Feature;

use App\Enums\PaymentType as EnumPaymentType;
use App\Models\PaymentType;
use App\Models\SalesOrder;
use App\Models\User;
use Illuminate\Foundation\Testing\WithFaker;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class PaymentsTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    public function test_can_create_payment_more_than_balance_on_sales_order(): void
    {
        Sanctum::actingAs(User::factory()->create());
        $salesOrder = SalesOrder::factory()->hasSalesOrderLines()->create();
        $balance = (float) (($salesOrder->orderLines->sum('subtotal') + ($salesOrder->exact_discount_total ?? 0)) + ($salesOrder->tax_total && ! $salesOrder->is_tax_included ? $salesOrder->tax_total : 0) - $salesOrder->total_paid);
        $paymentAmount = round($balance, 2) + 100;
        self::assertGreaterThan($balance, $paymentAmount);
        $this->postJson('/api/sales-orders/'.$salesOrder->id.'/payments', [
            'amount' => $paymentAmount,
            'payment_type_id' => $this->createPaymentType()->id,
        ])->assertSuccessful();
    }

    private function createPaymentType(): PaymentType
    {
        return PaymentType::factory()->create();
    }

    public function test_create_sales_credits_for_overpayment(): void
    {
        Sanctum::actingAs(User::factory()->create());
        $salesOrder = SalesOrder::factory()->hasSalesOrderLines()->create();
        $balance = (float) (($salesOrder->orderLines->sum('subtotal') + ($salesOrder->exact_discount_total ?? 0)) + ($salesOrder->tax_total && ! $salesOrder->is_tax_included ? $salesOrder->tax_total : 0) - $salesOrder->total_paid);
        $paymentAmount = round($balance, 2) + 100;
        self::assertGreaterThan($balance, $paymentAmount);

        $this->postJson('/api/sales-orders/'.$salesOrder->id.'/payments', [
            'amount' => $paymentAmount,
            'payment_type_id' => $this->createPaymentType()->id,
        ])->assertSuccessful();

        $this->assertEquals(100, $salesOrder->salesCredits->first()->total_credit);
        $this->assertEquals(1, $salesOrder->salesCredits->first()->is_for_overpayment);
    }

    public function test_check_create_payment_and_separate_payment_record_for_overpayment(): void
    {
        Sanctum::actingAs(User::factory()->create());
        $paymentType = $this->createPaymentType();
        $salesOrder = SalesOrder::factory()->hasSalesOrderLines()->create();
        $balance = (float) (($salesOrder->orderLines->sum('subtotal') + ($salesOrder->exact_discount_total ?? 0)) + ($salesOrder->tax_total && ! $salesOrder->is_tax_included ? $salesOrder->tax_total : 0) - $salesOrder->total_paid);
        $paymentAmount = round($balance, 2) + 100;
        self::assertGreaterThan($balance, $paymentAmount);
        $this->postJson('/api/sales-orders/'.$salesOrder->id.'/payments', [
            'amount' => $paymentAmount,
            'payment_type_id' => $paymentType->id,
        ])->assertSuccessful();

        $this->assertNotNull($salesOrder->salesCredits->first()->payments()->first());
        $this->assertEquals(round($paymentAmount, 2), $salesOrder->payments()->first()->amount);
        $this->assertEquals(100, $salesOrder->salesCredits->first()->payments()->first()->amount);
        $this->assertEquals(EnumPaymentType::TYPE_OVERPAYMENT,
            $salesOrder->salesCredits->first()->payments()->first()->type);
    }

    public function test_create_sales_credits_for_refund(): void
    {
        $paymentAmount = -100;
        Sanctum::actingAs(User::factory()->create());
        $salesOrder = SalesOrder::factory()->hasSalesOrderLines()->create();
        $paymentType = $this->createPaymentType();
        $salesOrder->payments()->create([
            'amount' => 1000,
            'payment_type_id' => $paymentType->id,
        ]);

        $this->postJson('/api/sales-orders/'.$salesOrder->id.'/payments', [
            'amount' => $paymentAmount,
            'payment_type_id' => $paymentType->id,
        ])->assertSuccessful();

        $this->assertNotNull($salesOrder->salesCredits->first());
        $this->assertEquals($salesOrder->salesCredits->first()->total_credit, abs($paymentAmount));
        $this->assertEquals(0, $salesOrder->salesCredits->first()->is_for_overpayment);
    }

    public function test_create_payment_associated_to_sales_credits_for_refund(): void
    {
        $paymentAmount = -100;
        $paymentType = $this->createPaymentType();
        Sanctum::actingAs(User::factory()->create());
        $salesOrder = SalesOrder::factory()->hasSalesOrderLines()->create();
        $salesOrder->payments()->create([
            'amount' => 1000,
            'payment_type_id' => $paymentType->id,
        ]);
        $this->postJson('/api/sales-orders/'.$salesOrder->id.'/payments', [
            'amount' => $paymentAmount,
            'payment_type_id' => $paymentType->id,
        ])->assertSuccessful();

        $this->assertNotNull($salesOrder->salesCredits->first()->payments()->first());
        $this->assertEquals($salesOrder->salesCredits->first()->payments()->first()->amount, abs($paymentAmount));
        $this->assertNotEquals(EnumPaymentType::TYPE_OVERPAYMENT,
            $salesOrder->salesCredits->first()->payments()->first()->type
        );
    }
}
