<?php

namespace Tests\Feature;

use App\Actions\TakeInventorySnapshot;
use App\Http\Requests\StoreInventoryAdjustment;
use App\Models\Customer;
use App\Models\InventorySnapshot;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\SalesOrder;
use App\Models\Setting;
use App\Models\User;
use App\Models\Warehouse;
use App\Services\FinancialManagement\DailyFinancialManager;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\DatabaseTransactions;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Tests\TestCaseWithoutTransactions;

/**
 * TODO: Disabling Daily Financials for now due to performance issues
 *
 * @group manual
 */
class InventorySnapshotTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    public function test_it_can_take_an_inventory_snapshot_for_backorder(): void
    {
        $customer = Customer::factory()->create();
        $product = Product::factory()->create();

        Sanctum::actingAs(User::factory()->create());

        Setting::query()->updateOrCreate(['key' => Setting::KEY_INVENTORY_START_DATE], [
            'value' => $this->faker->dateTimeThisYear(),
        ]);

        $timeOfOrder = $this->faker->dateTimeThisMonth()->format('Y-m-d H:i:s');

        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => $timeOfOrder,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => $product->name,
                    'amount' => 5.00,
                    'quantity' => 10,
                ],
            ],
        ])->assertSuccessful();

        (new SalesOrderLineFinancialManager())->calculate();
        (new DailyFinancialManager())->calculate();
        (new TakeInventorySnapshot())->handle();

        $this->assertEquals([
            'inventory_available' => round(-10, 1),
            'inventory_reserved' => round(10, 1),
            'inventory_in_transit' => round(0, 1),
            'inventory_stock_value' => round(0, 1),
            'inventory_in_stock' => false,
        ],
            InventorySnapshot::query()
                ->select(
                    'inventory_available',
                    'inventory_reserved',
                    'inventory_in_transit',
                    'inventory_stock_value',
                    'inventory_in_stock'
                )
                ->latest()->first()->toArray()
        );

        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => $timeOfOrder,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => $product->name,
                    'amount' => 5.00,
                    'quantity' => 20,
                ],
            ],
        ])->assertSuccessful();

        (new SalesOrderLineFinancialManager())->calculate();
        (new DailyFinancialManager())->calculate();
        (new TakeInventorySnapshot())->handle();

        $this->assertEquals([
            'inventory_available' => round(-30, 1),
            'inventory_reserved' => round(30, 1),
            'inventory_in_transit' => round(0, 1),
            'inventory_stock_value' => round(0, 1),
            'inventory_in_stock' => false,
        ],
            InventorySnapshot::query()
                ->select(
                    'inventory_available',
                    'inventory_reserved',
                    'inventory_in_transit',
                    'inventory_stock_value',
                    'inventory_in_stock'
                )
                ->latest()->first()
                ->toArray()
        );
    }

    public function test_it_can_take_an_inventory_snapshot_for_in_stock(): void
    {
        $customer = Customer::factory()->create();
        $product = Product::factory()->create();
        $warehouse = Warehouse::factory()->create();

        Sanctum::actingAs(User::factory()->create());

        $timeOfOrder = $this->faker->dateTimeThisMonth()->format('Y-m-d H:i:s');

        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => $timeOfOrder,
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 5.00,
            'quantity' => 50,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ])->assertSuccessful();

        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => $timeOfOrder,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => $product->name,
                    'warehouse_id' => $warehouse->id,
                    'amount' => 5.00,
                    'quantity' => 10,
                ],
            ],
        ])->assertSuccessful();

        (new SalesOrderLineFinancialManager())->calculate();
        (new DailyFinancialManager())->calculate();
        (new TakeInventorySnapshot())->handle();

        $this->assertEquals([
            'inventory_available' => round(40, 1),
            'inventory_reserved' => round(10, 1),
            'inventory_in_transit' => round(0, 1),
            'inventory_stock_value' => round(50 * 5, 1),
            'inventory_in_stock' => true,
        ],
            InventorySnapshot::query()
                ->select(
                    'inventory_available',
                    'inventory_reserved',
                    'inventory_in_transit',
                    'inventory_stock_value',
                    'inventory_in_stock'
                )
                ->first()
                ->toArray()
        );

        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => $timeOfOrder,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => $product->name,
                    'amount' => 5.00,
                    'quantity' => 20,
                ],
            ],
        ])->assertSuccessful();

        (new SalesOrderLineFinancialManager())->calculate();
        (new DailyFinancialManager())->calculate();
        (new TakeInventorySnapshot())->handle();

        $this->assertEquals([
            'inventory_available' => round(20, 1),
            'inventory_reserved' => round(30, 1),
            'inventory_in_transit' => round(0, 1),
            'inventory_stock_value' => round(50 * 5, 1),
            'inventory_in_stock' => true,
        ],
            InventorySnapshot::query()
                ->select(
                    'inventory_available',
                    'inventory_reserved',
                    'inventory_in_transit',
                    'inventory_stock_value',
                    'inventory_in_stock'
                )
                ->first()
                ->toArray()
        );
    }

    public function test_it_can_take_delete_an_inventory_snapshot(): void
    {
        $customer = Customer::factory()->create();
        $product = Product::factory()->create();
        $warehouse = Warehouse::factory()->create();

        Sanctum::actingAs(User::factory()->create());

        $timeOfOrder = $this->faker->dateTimeThisMonth()->format('Y-m-d H:i:s');

        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => $timeOfOrder,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => $product->name,
                    'warehouse_id' => $warehouse->id,
                    'amount' => 5.00,
                    'quantity' => 10,
                ],
            ],
        ])->assertSuccessful();

        (new SalesOrderLineFinancialManager())->calculate();
        (new DailyFinancialManager())->calculate();
        (new TakeInventorySnapshot())->handle();

        $this->assertDatabaseCount((new InventorySnapshot())->getTable(), 1);
        SalesOrder::query()->each(function (SalesOrder $salesOrder) {
            $salesOrder->delete();
        });

        (new SalesOrderLineFinancialManager())->calculate();
        (new DailyFinancialManager())->calculate();
        (new TakeInventorySnapshot())->handle();

        $this->assertDatabaseCount((new InventorySnapshot())->getTable(), 0);
    }

    public function test_inventory_snapshots_get_invalidated_after_bulk_receipt(): void
    {
        Queue::fake();
        Sanctum::actingAs(User::factory()->create());

        $purchaseOrder = PurchaseOrder::factory()->fullyInvoiced(5)->create();

        $purchaseOrderLines = $purchaseOrder->purchaseOrderLines;

        $receipt_lines = [];

        foreach ($purchaseOrderLines as $line) {
            $receipt_lines[] = ['purchase_order_line_id' => $line->id, 'quantity' => $line->quantity];
        }

        $this->postJson('/api/purchase-order-shipments/receive',
            [
                'received_at' => Carbon::now()->toString(),
                'receipt_lines' => $receipt_lines,
            ]
        )->assertSuccessful();

        (new SalesOrderLineFinancialManager())->calculate();
        (new DailyFinancialManager())->calculate();

        $this->assertDatabaseCount((new InventorySnapshot())->getTable(), 0);

        (new TakeInventorySnapshot())->handle();

        $this->assertEquals(5,
            InventorySnapshot::query()->where('is_cache_valid', 1)->count()
        );

        $this->postJson('/api/purchase-order-shipments/receive',
            [
                'received_at' => Carbon::now()->toString(),
                'receipt_lines' => $receipt_lines,
            ]
        )->assertSuccessful();

        $this->assertEquals(5,
            InventorySnapshot::query()->where('is_cache_valid', 0)->count()
        );
    }
}
