<?php

namespace Tests\Feature;

use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrderLine;
use App\Models\User;
use App\Models\Warehouse;
use DateTimeImmutable;
use DateTimeInterface;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class InventoryMovementTest extends TestCase
{
    use FastRefreshDatabase;

    protected Warehouse $warehouse;

    protected function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->warehouse = Warehouse::factory()->create()->withDefaultLocation();
    }

    public function test_get_tallies(): void
    {
        $product = Product::factory()->create();

        // NOTE: intentionally creating records with non-sequential dates
        $im2 = $this->createInventoryMovement(
            $product,
            -8,
            new DateTimeImmutable('2020-06-15')
        );

        $im1 = $this->createInventoryMovement(
            $product,
            12,
            new DateTimeImmutable('2020-06-01')
        );

        $im3 = $this->createInventoryMovement(
            $product,
            30,
            new DateTimeImmutable('2020-06-20')
        );

        $queryParams = [
            'warehouse_id' => $this->warehouse->id,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'product_id' => $product->id,
        ];

        $response = $this->actingAs($this->user)
            ->get('/api/inventory-movements/tallies?'.http_build_query($queryParams));

        $response->assertOk();

        $tallies = $response->json()['data'];

        $this->assertEquals(
            [
                $im1->id => 12.0,
                $im2->id => 4.0,
                $im3->id => 34.0,
            ],
            $tallies
        );
    }

    protected function createInventoryMovement(
        Product $product,
        int $quantity,
        DateTimeInterface $inventoryMovementDate,
    ): InventoryMovement {
        return InventoryMovement::factory()->create(
            [
                'warehouse_id' => $this->warehouse->id,
                'product_id' => $product->id,
                'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
                'quantity' => $quantity,
                'inventory_movement_date' => $inventoryMovementDate,
                'link_type' => SalesOrderLine::class,
            ]
        );
    }
}
