<?php

namespace Tests\Feature;

use App\Models\InventoryForecast;
use App\Models\InventoryForecastItem;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\Supplier;
use App\Models\SupplierPricingTier;
use App\Models\User;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class InventoryForecastControllerTest extends TestCase
{
    use FastRefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        Sanctum::actingAs(User::factory()->create());
    }

    public function test_it_can_create_multiple_purchase_orders()
    {

        /** @var SupplierPricingTier $defaultSupplierPricingTier */
        $defaultSupplierPricingTier = SupplierPricingTier::query()->firstOrFail();

        /** @var Supplier $supplier1 */
        $supplier1 = Supplier::factory()->create();
        $supplier1->pricingTiers()->attach($defaultSupplierPricingTier->id);

        /** @var Supplier $supplier2 */
        $supplier2 = Supplier::factory()->create();
        $supplier2->pricingTiers()->attach($defaultSupplierPricingTier->id);

        /** @var Product $product */
        $product = Product::factory()->create();

        /** @var InventoryForecast $forecast1 */
        $forecast1 = InventoryForecast::factory()
            ->has(InventoryForecastItem::factory()->state([
                'product_id' => $product->id,
                'suggested_purchase_quantity' => 10,
            ]))
            ->create([
            'supplier_id' => $supplier1->id
        ]);

        /** @var InventoryForecast $forecast2 */
        $forecast2 = InventoryForecast::factory()
            ->has(InventoryForecastItem::factory()->state([
                'product_id' => $product->id,
                'suggested_purchase_quantity' => 5,
            ]))
            ->create([
                'supplier_id' => $supplier2->id
            ]);

        $response = $this->postJson('/api/inventory/forecast/purchase-orders', [
            'purchase_orders' => [
                [
                    'supplier_id' => $supplier1->id,
                    'purchase_order_lines' => [
                        [
                            'product_id' => $product->id,
                            'quantity' => 10,
                            'amount' => 5,
                            'description' => $product->name,
                            'forecast_item_id' => $forecast1->inventoryForecastItems->first()->id,
                        ],
                    ],
                ],
                [
                    'supplier_id' => $supplier2->id,
                    'purchase_order_lines' => [
                        [
                            'product_id' => $product->id,
                            'quantity' => 5,
                            'amount' => 12,
                            'description' => $product->name,
                            'forecast_item_id' => $forecast2->inventoryForecastItems->first()->id,
                        ],
                    ],
                ],
            ],
        ])->assertSuccessful();

        $response->assertJsonStructure([
            'data' => [
                '*' => [
                    'id',
                    'supplier_id',
                    'order_status',
                    'destination_warehouse_id',
                    'po_number',
                    'items' => [
                        '*' => [
                            'id',
                            'sku' => ['id', 'sku'],
                            'quantity',
                            'price',
                        ],
                    ],
                ],
            ],
        ]);

        // Assert that the purchase orders were created
        $this->assertDatabaseCount('purchase_orders', 2);
        $this->assertDatabaseCount('purchase_order_lines', 2);

        $this->assertDatabaseHas('purchase_orders', [
            'supplier_id' => $supplier1->id,
            'order_status' => PurchaseOrder::STATUS_OPEN,
        ]);

        $this->assertDatabaseHas('purchase_orders', [
            'supplier_id' => $supplier2->id,
            'order_status' => PurchaseOrder::STATUS_OPEN,
        ]);

        // Purchase order lines should be linked to forecast items.
        $this->assertDatabaseCount('forecast_item_po_line_links', 2);
        $this->assertDatabaseHas('forecast_item_po_line_links', [
            'forecast_item_id' => $forecast1->inventoryForecastItems->first()->id,
            'purchase_order_line_id' => $response->json('data.0.items.0.id'),
        ]);
        $this->assertDatabaseHas('forecast_item_po_line_links', [
            'forecast_item_id' => $forecast2->inventoryForecastItems->first()->id,
            'purchase_order_line_id' => $response->json('data.1.items.0.id'),
        ]);
    }
}
