<?php

namespace Tests\Feature;

use App\Models\Integration;
use App\Models\IntegrationInstance;
use Illuminate\Console\Scheduling\Event;
use Illuminate\Console\Scheduling\Schedule;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class IntegrationInstanceTest extends TestCase
{
    use FastRefreshDatabase;

    public function test_it_can_schedule_integration_instance_sync_jobs(): void
    {
        /** @var Schedule $schedule */
        $schedule = app(Schedule::class);

        $events = collect($schedule->events())->filter(function (Event $event) {
            return
                stripos($event->command, 'xero') ||
                stripos($event->command, 'amazon') ||
                stripos($event->command, 'magento') ||
                stripos($event->command, 'shopify') ||
                stripos($event->command, 'shipstation') ||
                stripos($event->command, 'starshipit');
        });

        $this->assertEquals(0, $events->count(), 'There should be no scheduled jobs for integration instances that do not exist');
    }

    public function test_it_does_not_schedule_integration_instance_sync_jobs_when_disabled(): void
    {
        IntegrationInstance::factory()->create([
            'is_automatic_sync_enabled' => false,
            'integration_id' => Integration::query()->where('name', Integration::NAME_XERO)->firstOrFail()->id,
        ]);

        IntegrationInstance::factory()->create([
            'is_automatic_sync_enabled' => false,
            'integration_id' => Integration::query()->where('name', Integration::NAME_AMAZON_US)->firstOrFail()->id,
        ]);

        IntegrationInstance::factory()->create([
            'is_automatic_sync_enabled' => false,
            'integration_id' => Integration::query()->where('name', Integration::NAME_MAGENTO)->firstOrFail()->id,
        ]);

        IntegrationInstance::factory()->create([
            'is_automatic_sync_enabled' => false,
            'integration_id' => Integration::query()->where('name', Integration::NAME_SHOPIFY)->firstOrFail()->id,
        ]);

        IntegrationInstance::factory()->create([
            'is_automatic_sync_enabled' => false,
            'integration_id' => Integration::query()->where('name', Integration::NAME_SHIPSTATION)->firstOrFail()->id,
        ]);

        IntegrationInstance::factory()->create([
            'is_automatic_sync_enabled' => false,
            'integration_id' => Integration::query()->where('name', Integration::NAME_STARSHIPIT)->firstOrFail()->id,
        ]);

        /** @var Schedule $schedule */
        $schedule = app(Schedule::class);

        $events = collect($schedule->events())->filter(function (Event $event) {
            return
                stripos($event->command, 'xero') ||
                stripos($event->command, 'amazon') ||
                stripos($event->command, 'magento') ||
                stripos($event->command, 'shopify') ||
                stripos($event->command, 'shipstation') ||
                stripos($event->command, 'starshipit');
        });

        $this->assertEquals(0, $events->count(), 'There should be no scheduled jobs for integration instances that have auto sync disabled');
    }

    public function test_it_does_schedule_integration_instance_sync_jobs_when_enabled(): void
    {
        IntegrationInstance::factory()->create([
            'is_automatic_sync_enabled' => true,
            'integration_id' => Integration::query()->where('name', Integration::NAME_XERO)->firstOrFail()->id,
        ]);

        IntegrationInstance::factory()->create([
            'is_automatic_sync_enabled' => true,
            'integration_settings' => [
                'is_fba_enabled' => false,
            ],
            'integration_id' => Integration::query()->where('name', Integration::NAME_AMAZON_US)->firstOrFail()->id,
        ]);

        IntegrationInstance::factory()->create([
            'is_automatic_sync_enabled' => true,
            'integration_id' => Integration::query()->where('name', Integration::NAME_MAGENTO)->firstOrFail()->id,
        ]);

        IntegrationInstance::factory()->create([
            'is_automatic_sync_enabled' => true,
            'integration_id' => Integration::query()->where('name', Integration::NAME_SHOPIFY)->firstOrFail()->id,
        ]);

        IntegrationInstance::factory()->create([
            'is_automatic_sync_enabled' => true,
            'integration_id' => Integration::query()->where('name', Integration::NAME_SHIPSTATION)->firstOrFail()->id,
        ]);

        IntegrationInstance::factory()->create([
            'is_automatic_sync_enabled' => true,
            'integration_id' => Integration::query()->where('name', Integration::NAME_STARSHIPIT)->firstOrFail()->id,
        ]);

        /** @var Schedule $schedule */
        $schedule = app(Schedule::class);

        $xeroEvents = collect($schedule->events())->filter(function (Event $event) {
            return stripos($event->command, 'xero');
        });

        $this->assertTrue($xeroEvents->count() > 0, 'A scheduled job for Xero integration instance should exist');

        $amazonEvents = collect($schedule->events())->filter(function (Event $event) {
            return stripos($event->command, 'amazon');
        });

        $this->assertTrue($amazonEvents->count() > 0, 'A scheduled job for Amazon integration instance should exist');

        $magentoEvents = collect($schedule->events())->filter(function (Event $event) {
            return stripos($event->command, 'magento');
        });

        $this->assertTrue($magentoEvents->count() > 0, 'A scheduled job for Magento integration instance should exist');

        $shopifyEvents = collect($schedule->events())->filter(function (Event $event) {
            return stripos($event->command, 'shopify');
        });

        $this->assertTrue($shopifyEvents->count() > 0, 'A scheduled job for Shopify integration instance should exist');

        $amazonEvents = collect($schedule->events())->filter(function (Event $event) {
            return stripos($event->command, 'amazon');
        });

        $this->assertTrue($amazonEvents->count() > 0, 'A scheduled job for Amazon integration instance should exist');

        $shipstationEvents = collect($schedule->events())->filter(function (Event $event) {
            return stripos($event->command, 'shipstation');
        });

        $this->assertTrue($shipstationEvents->count() > 0, 'A scheduled job for ShipStation integration instance should exist');

        $starshipitEvents = collect($schedule->events())->filter(function (Event $event) {
            return stripos($event->command, 'starshipit');
        });

        $this->assertTrue($starshipitEvents->count() > 0, 'A scheduled job for StarShipIt integration instance should exist');
    }
}
