<?php

namespace Tests\Feature;

use App\Data\CreateDropshipData;
use App\Data\CreateDropshipShipmentData;
use App\Models\Address;
use App\Models\Customer;
use App\Models\PurchaseOrder;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Models\ShippingMethod;
use App\Models\Supplier;
use App\Models\User;
use App\Models\Warehouse;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class DropshipControllerTest extends TestCase
{
    use FastRefreshDatabase;

    public function setUp(): void
    {
        parent::setUp();

        Sanctum::actingAs(User::first());
    }

    public function test_dropship_controller()
    {
        $this->markTestSkipped('WIP');
        $address = Address::factory()->create();

        $customer = Customer::factory([
            'default_shipping_address_id' => $address->id,
        ])
            ->create();

        $supplier = Supplier::factory()
            ->hasSupplierProducts()
            ->create();

        $warehouse = Warehouse::factory([
            'type' => Warehouse::TYPE_SUPPLIER,
            'supplier_id' => $supplier->id,
            'dropship_enabled' => true,
        ])->create();
        $supplier->default_warehouse_id = $warehouse->id;
        $supplier->update();
        $supplier->refresh();

        $product = $supplier->supplierProducts->first()->product;

        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines(1, [
                'product_id' => $product->id,
                'warehouse_id' => $warehouse->id,
            ])
            ->create();

        $shippingMethod = ShippingMethod::first();

        /*
        |--------------------------------------------------------------------------
        | Create Dropship
        |--------------------------------------------------------------------------
        */

        $response = $this->postJson(route('sales-orders.dropship', $salesOrder->id), CreateDropshipData::from([
            'sales_order_id' => $salesOrder->id,
            'warehouse_id' => $warehouse->id,
            'requested_shipping_method_id' => $shippingMethod->id,
            'sales_order_lines' => [
                [
                    'id' => $salesOrder->salesOrderLines->first()->id,
                ]
            ],
        ])->toArray());

        $response->assertOk();

        $purchaseOrderId = $response->json('data.id');

        $this->assertDatabaseHas(PurchaseOrder::class, [
            'id' => $purchaseOrderId,
            'supplier_id' => $supplier->id,
            'supplier_warehouse_id' => $warehouse->id,
            'destination_address_id' => $salesOrder->shipping_address_id,
            'currency_id' => $salesOrder->currency_id,
            'store_id' => $salesOrder->store_id,
            'purchase_order_date' => $salesOrder->order_date,
            'sales_order_id' => $salesOrder->id,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Ship Dropship
        |--------------------------------------------------------------------------
        */

        $response = $this->postJson(route('purchase-orders.dropship', $purchaseOrderId), CreateDropshipShipmentData::from([
            'tracking_number' => '123456',
        ])->toArray())->assertOk();


        $salesOrderFulfillmentId = $response->json('data.id');

        $response->assertJson(([
            'data'=> [
                'tracking_number' => '123456',
                'fulfilled_shipping_method_id' => $shippingMethod->id,
            ]
        ]));

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'id' => $salesOrder->id,
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_FULFILLED,
            'order_status' => SalesOrder::STATUS_CLOSED,
        ]);

        $this->assertDatabaseHas(PurchaseOrder::class, [
            'id' => $purchaseOrderId,
            'order_status' => PurchaseOrder::STATUS_CLOSED,
            'receipt_status' => PurchaseOrder::RECEIPT_STATUS_RECEIVED,
            'shipment_status' => PurchaseOrder::SHIPMENT_STATUS_SHIPPED_CUSTOMER,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Delete Sales Order Fulfillment
        |--------------------------------------------------------------------------
        */

        $response = $this->deleteJson(route('sales-order-fulfillments.destroy', $salesOrderFulfillmentId));

        $response->assertOk();

        $this->assertDatabaseEmpty(SalesOrderFulfillment::class);
        $this->assertDatabaseHas(SalesOrder::class, [
            'id' => $salesOrder->id,
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_AWAITING_TRACKING,
            'order_status' => SalesOrder::STATUS_OPEN,
        ]);
        $this->assertDatabaseHas(PurchaseOrder::class, [
            'id' => $purchaseOrderId,
            'order_status' => PurchaseOrder::STATUS_OPEN,
            'receipt_status' => PurchaseOrder::RECEIPT_STATUS_DROPSHIP,
            'shipment_status' => PurchaseOrder::SHIPMENT_STATUS_UNSHIPPED,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Delete Unshipped Dropship Purchase Order
        |--------------------------------------------------------------------------
        */

        $response = $this->deleteJson(route('purchase-orders.destroy', $purchaseOrderId));

        $response->assertOk();

        $this->assertDatabaseEmpty(PurchaseOrder::class);

        $this->assertDatabaseHas(SalesOrder::class, [
            'id' => $salesOrder->id,
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_UNFULFILLED,
            'order_status' => SalesOrder::STATUS_OPEN,
        ]);

        $response = $this->postJson(route('sales-orders.dropship', $salesOrder->id), CreateDropshipData::from([
            'sales_order_id' => $salesOrder->id,
            'warehouse_id' => $warehouse->id,
            'requested_shipping_method_id' => $shippingMethod->id,
            'sales_order_lines' => [
                [
                    'id' => $salesOrder->salesOrderLines->first()->id,
                ]
            ],
        ])->toArray())->assertOk();

        $purchaseOrderId = $response->json('data.id');

        $this->postJson(route('purchase-orders.dropship', $purchaseOrderId), CreateDropshipShipmentData::from([
            'tracking_number' => '123456',
        ])->toArray())->assertOk();

        /*
        |--------------------------------------------------------------------------
        | Delete Shipped Dropship Purchase Order
        |--------------------------------------------------------------------------
        */

        $response = $this->deleteJson(route('purchase-orders.destroy', $purchaseOrderId));

        $response->assertOk();

        $this->assertDatabaseEmpty(PurchaseOrder::class);

        $this->assertDatabaseEmpty(SalesOrderFulfillment::class);

        $this->assertDatabaseHas(SalesOrder::class, [
            'id' => $salesOrder->id,
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_UNFULFILLED,
            'order_status' => SalesOrder::STATUS_OPEN,
        ]);
    }
}
