<?php

namespace Tests\Feature;

use App\Jobs\GenerateCacheDailyAverageConsumptionForProductsJob;
use App\Jobs\CalculateDailyFinancialsJob;
use App\Models\Product;
use App\Models\ReportingDailyFinancial;
use App\Models\SalesOrder;
use App\Models\Setting;
use App\Repositories\SettingRepository;
use App\Services\FinancialManagement\DailyFinancialManager;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use Carbon\Carbon;
use Illuminate\Support\Facades\Redis;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Queue;
use Tests\TestCase;

class DailyFinancialManagerTest extends TestCase
{
    use FastRefreshDatabase;

    public function test_it_can_calculate_daily_financials(): void
    {
        Queue::fake();

        // Set up orders
        SalesOrder::factory()->hasSalesOrderLines()->create(['order_date' => Carbon::now()->subDay()]);
        SalesOrder::factory()->hasSalesOrderLines()->create(['order_date' => Carbon::now()->subDays(2)]);
        SalesOrder::factory()->hasSalesOrderLines()->create(['order_date' => Carbon::now()->subDays(3)]);
        SalesOrder::factory()->hasSalesOrderLines()->create(['order_date' => Carbon::now()->subDays(4)]);

        $this->assertEquals(4, Redis::scard(ReportingDailyFinancial::INVALID_FINANCIALS_KEY));
        $this->assertEquals(0, Redis::scard(Product::INVALID_DAILY_AVERAGE_CONSUMPTION_KEY));

        (new SalesOrderLineFinancialManager())->calculate();
        app(DailyFinancialManager::class)->calculate();

        $this->assertEquals(0, Redis::scard(ReportingDailyFinancial::INVALID_FINANCIALS_KEY));

        $this->assertDatabaseCount((new ReportingDailyFinancial())->getTable(), 4);

        $this->assertEquals(4, Redis::scard(Product::INVALID_DAILY_AVERAGE_CONSUMPTION_KEY));

        Queue::assertPushed(GenerateCacheDailyAverageConsumptionForProductsJob::class);
    }

    public function test_it_can_recalculate_daily_financials(): void
    {
        Queue::fake();

        app(SettingRepository::class)->set(Setting::KEY_DEFAULT_TIMEZONE, 'UTC');
        $date1 = Carbon::parse('2020-01-01');
        $date2 = Carbon::parse('2020-01-02');
        $date3 = Carbon::parse('2020-01-03');
        $date4 = Carbon::parse('2020-01-04');

        $newDate = Carbon::parse('2021-01-01');

        // Set up orders
        SalesOrder::factory()->hasSalesOrderLines()->create(['order_date' => $date1]);
        SalesOrder::factory()->hasSalesOrderLines()->create(['order_date' => $date2]);
        SalesOrder::factory()->hasSalesOrderLines()->create(['order_date' => $date3]);
        SalesOrder::factory()->hasSalesOrderLines()->create(['order_date' => $date4]);

        (new SalesOrderLineFinancialManager())->calculate();
        app(DailyFinancialManager::class)->calculate();

        $dailyFinancials = ReportingDailyFinancial::all();

        SalesOrder::each(function (SalesOrder $salesOrder) use ($newDate) {
            $salesOrder->order_date = $newDate;
            $salesOrder->save();
        });
        (new SalesOrderLineFinancialManager())->calculate();

        // Recalculate
        app(DailyFinancialManager::class)->recalculateDailyFinancials($dailyFinancials);
        Queue::assertPushed(CalculateDailyFinancialsJob::class);
        app(DailyFinancialManager::class)->calculate();

        $this->assertCount(0, ReportingDailyFinancial::whereTotalRevenue(0)->get());
        $this->assertDatabaseCount(ReportingDailyFinancial::class, 4);
        $this->assertCount(4, ReportingDailyFinancial::whereDate('date', $newDate)->get());
    }
}