<?php

namespace Tests\Feature;

use App\Actions\CacheDailyAverageConsumptionForProducts;
use App\Actions\TakeInventorySnapshot;
use App\Helpers;
use App\Http\Requests\StoreInventoryAdjustment;
use App\Jobs\GenerateCacheDailyAverageConsumptionForProductsJob;
use App\Models\Customer;
use App\Models\Product;
use App\Models\ReportingDailyFinancial;
use App\Models\SalesOrder;
use App\Models\Setting;
use App\Models\User;
use App\Models\Warehouse;
use App\Services\FinancialManagement\DailyFinancialManager;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class DailyAverageConsumptionTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    /**
     * A basic feature test example.
     */
    public function test_it_can_update_daily_average_consumption_for_new_sales_order(): void
    {

        Queue::fake();

        $customer = Customer::factory()->create();
        $product = Product::factory()->create();

        Sanctum::actingAs(User::factory()->create());

        $orderDate = $this->faker->dateTimeThisMonth()->format('Y-m-d H:i:s');

        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => $orderDate,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => $product->name,
                    'amount' => 5.00,
                    'quantity' => 10,
                ],
            ],
        ])->assertSuccessful();

        (new SalesOrderLineFinancialManager())->calculate();
        (new DailyFinancialManager())->calculate();
        Queue::assertPushed(GenerateCacheDailyAverageConsumptionForProductsJob::class);
        (new TakeInventorySnapshot())->handle();
        (new CacheDailyAverageConsumptionForProducts())->handle();

        $this->assertEquals(
            round(10 / Helpers::setting(Setting::KEY_DAYS_SALES_HISTORY), 4),
            Product::findOrFail($product->id)->daily_average_consumption
        );
    }

    public function test_changing_days_of_sales_history_updates_daily_average_consumption(): void
    {
        Queue::fake();

        $customer = Customer::factory()->create();
        $product = Product::factory()->create();

        Sanctum::actingAs(User::factory()->create());

        $this->putJson('/api/settings/inventory-forecasting', [
            'settings' => [
                [
                    'key' => Setting::KEY_DAYS_SALES_HISTORY,
                    'value' => '10',
                ],
            ],
        ])->assertSuccessful();

        for ($i = 0; $i < 2; $i++) {
            $this->postJson('/api/sales-orders', [
                'order_status' => SalesOrder::STATUS_OPEN,
                'order_date' => Helpers::utcStartOfLocalDate($this->faker->dateTimeBetween(Carbon::now()->addDay()->subDays(10), Carbon::now()->subDays(5)->subMinute())->format('Y-m-d H:i:s')),
                'customer_id' => $customer->id,
                'currency_code' => 'USD',
                'sales_order_lines' => [
                    [
                        'product_id' => $product->id,
                        'description' => $product->name,
                        'amount' => 5.00,
                        'quantity' => 2,
                    ],
                ],
            ])->assertSuccessful();
        }
        for ($i = 0; $i < 2; $i++) {
            $this->postJson('/api/sales-orders', [
                'order_status' => SalesOrder::STATUS_OPEN,
                'order_date' => Helpers::utcStartOfLocalDate($this->faker->dateTimeBetween(Carbon::now()->addDay()->subDays(5), Carbon::now())->format('Y-m-d H:i:s')),
                'customer_id' => $customer->id,
                'currency_code' => 'USD',
                'sales_order_lines' => [
                    [
                        'product_id' => $product->id,
                        'description' => $product->name,
                        'amount' => 5.00,
                        'quantity' => 1,
                    ],
                ],
            ])->assertSuccessful();
        }

        (new SalesOrderLineFinancialManager())->calculate();
        (new DailyFinancialManager())->calculate();
        Queue::assertPushed(GenerateCacheDailyAverageConsumptionForProductsJob::class);
        (new TakeInventorySnapshot())->handle();
        (new CacheDailyAverageConsumptionForProducts())->handle();

        $this->assertEquals(
            0.6,
            Product::findOrFail($product->id)->daily_average_consumption
        );

        $this->putJson('/api/settings/inventory-forecasting', [
            'settings' => [
                [
                    'key' => Setting::KEY_DAYS_SALES_HISTORY,
                    'value' => '5',
                ],
            ],
        ])->assertSuccessful();

        (new SalesOrderLineFinancialManager())->calculate();
        Queue::assertPushed(GenerateCacheDailyAverageConsumptionForProductsJob::class);
        (new DailyFinancialManager())->calculate();
        (new TakeInventorySnapshot())->handle();
        (new CacheDailyAverageConsumptionForProducts())->handle();

        $this->assertEquals(
            0.4,
            Product::findOrFail($product->id)->daily_average_consumption
        );
    }

    public function test_it_can_update_daily_average_consumption_for_new_sales_order_with_out_of_stock_adjustment(): void
    {
        $this->markTestSkipped('Disabling test until we bring the functionality back.');
        Queue::fake();

        $customer = Customer::factory()->create();
        $product = Product::factory()->create();
        $warehouse = Warehouse::factory()->create();

        Sanctum::actingAs(User::factory()->create());

        $this->putJson('/api/settings/inventory-forecasting', [
            'settings' => [
                [
                    'key' => Setting::KEY_DAYS_SALES_HISTORY,
                    'value' => '10',
                ],
            ],
        ])->assertSuccessful();

        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => Carbon::now()->subDays(10)->format('Y-m-d H:i:s'),
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 5.00,
            'quantity' => 4,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ])->assertSuccessful();

        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => Carbon::now()->subDays(9)->format('Y-m-d H:i:s'),
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => $product->name,
                    'amount' => 5.00,
                    'quantity' => 2,
                ],
            ],
        ])->assertSuccessful();

        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => Carbon::now()->subDays(8)->format('Y-m-d H:i:s'),
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => $product->name,
                    'amount' => 5.00,
                    'quantity' => 2,
                ],
            ],
        ])->assertSuccessful();

        $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => Carbon::now()->subDays(7)->format('Y-m-d H:i:s'),
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => $product->name,
                    'amount' => 5.00,
                    'quantity' => 2,
                ],
            ],
        ])->assertSuccessful();

        (new SalesOrderLineFinancialManager())->calculate();
        (new DailyFinancialManager())->calculate();
        Queue::assertPushed(GenerateCacheDailyAverageConsumptionForProductsJob::class);
        (new TakeInventorySnapshot())->handle();
        (new CacheDailyAverageConsumptionForProducts())->handle();

        $this->assertEquals(
            0.6,
            Product::query()->first()->daily_average_consumption
        );

        $this->putJson('/api/settings/inventory-forecasting', [
            'settings' => [
                [
                    'key' => Setting::KEY_ADJUST_FORECAST_OUT_OF_STOCK,
                    'value' => true,
                ],
            ],
        ])->assertSuccessful();

        (new SalesOrderLineFinancialManager())->calculate();
        Queue::assertPushed(GenerateCacheDailyAverageConsumptionForProductsJob::class);
        (new DailyFinancialManager())->calculate();
        (new TakeInventorySnapshot())->handle();
        (new CacheDailyAverageConsumptionForProducts())->handle();

        $this->assertEquals(
            3,
            Product::query()->first()->daily_average_consumption
        );
    }

    public function test_it_can_update_daily_average_consumption_for_sales_order_deletion(): void
    {
        Queue::fake();

        /** @var Customer $customer */
        $customer = Customer::factory()->create();

        /** @var Product $product */
        $product = Product::factory()->create();

        Sanctum::actingAs(User::factory()->create());

        $responseCreateSalesOrder = $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => $this->faker->dateTimeThisMonth()->format('Y-m-d H:i:s'),
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => $product->name,
                    'amount' => 5.00,
                    'quantity' => 10,
                ],
            ],
        ]);

        $responseCreateSalesOrder->assertSuccessful();

        $salesOrderId = $responseCreateSalesOrder->json('data.id');

        /** @var SalesOrder $salesOrder */
        $salesOrder = SalesOrder::query()->find($salesOrderId);

        (new SalesOrderLineFinancialManager())->calculate();
        (new DailyFinancialManager())->calculate();
        (new TakeInventorySnapshot())->handle();
        (new CacheDailyAverageConsumptionForProducts())->handle();
        $product->refresh();

        $this->assertEquals(
            round(10 / Helpers::setting(Setting::KEY_DAYS_SALES_HISTORY), 4),
            $product->daily_average_consumption
        );

        $salesOrder->delete();

        (new SalesOrderLineFinancialManager())->calculate();
        (new DailyFinancialManager())->calculate();
        (new TakeInventorySnapshot())->handle();
        (new CacheDailyAverageConsumptionForProducts())->handle();
        $product->refresh();

        $this->assertDatabaseCount((new ReportingDailyFinancial)->getTable(), 0);

        $this->assertEquals(
            0,
            $product->daily_average_consumption
        );
    }
}
