<?php

namespace Tests\Feature;

use App\Models\Currency;
use App\Models\Customer;
use App\Models\Payment;
use App\Models\PaymentType;
use App\Models\Product;
use App\Models\ProductPricingTier;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\SalesCredit;
use App\Models\SalesCreditLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Models\User;
use App\Services\FinancialManagement\DailyFinancialManager;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use Illuminate\Foundation\Testing\DatabaseTransactions;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Tests\TestCaseWithoutTransactions;

/**
 * TODO: Bring back with InventorySnapshot test
 *
 * @group manual
 */
class CurrencyTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    public function test_it_can_cache_and_change_currency_rate_for_sales_orders(): void
    {
        $customer = Customer::factory()->create();

        $product = Product::factory()
            ->hasProductPricing(1, [
                'price' => 5.00,
            ])->create()->first();

        $productPrice = $product->getDefaultPricing(ProductPricingTier::default()->id)->price;

        $currency = Currency::factory()
            ->create([
                'code' => 'CAD',
                'conversion' => 0.66,
            ]);

        Sanctum::actingAs(User::factory()->create());

        $response = $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => $this->faker->dateTimeThisMonth()->format('Y-m-d H:i:s'),
            'customer_id' => $customer->id,
            'currency_code' => $currency->code,
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => $product->name,
                    'amount' => $productPrice,
                    'quantity' => 1,
                ],
            ],
        ])->assertSuccessful();

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'currency_id' => $currency->id,
            'currency_rate' => $currency->conversion,
            'currency_id_tenant_snapshot' => Currency::default()->id,
        ]);

        $this->assertEquals(
            $productPrice * $currency->conversion,
            SalesOrderLine::first()->amount_in_tenant_currency
        );

        $salesOrder = SalesOrder::find($response->offsetGet('data')['id']);

        $salesOrder->update([
            'currency_code' => 'USD',
        ]);

        $this->assertEquals(
            $productPrice,
            SalesOrderLine::first()->amount_in_tenant_currency
        );
    }

    public function test_it_can_cache_and_change_currency_rate_for_purchase_orders(): void
    {
        $supplier = Supplier::factory()->withWarehouse()->create();

        $unitCost = 3.00;

        $product = Product::factory()->has(
            SupplierProduct::factory(1, [
                'supplier_id' => $supplier->id,
            ])
        )->create([
            'unit_cost' => $unitCost,
        ])->first();

        $currency = Currency::factory()
            ->create([
                'code' => 'CAD',
                'conversion' => 0.66,
            ]);

        Sanctum::actingAs(User::factory()->create());

        $response = $this->postJson('/api/purchase-orders', [
            'order_status' => PurchaseOrder::STATUS_DRAFT,
            'purchase_order_date' => $this->faker->dateTimeThisMonth()->format('Y-m-d H:i:s'),
            'supplier_id' => $supplier->id,
            'currency_code' => $currency->code,
            'purchase_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => $product->name,
                    'amount' => $product->unit_cost,
                    'quantity' => 1,
                ],
            ],
        ])->assertSuccessful();

        $this->assertDatabaseHas((new PurchaseOrder())->getTable(), [
            'currency_id' => $currency->id,
            'currency_rate' => $currency->conversion,
            'currency_id_tenant_snapshot' => Currency::default()->id,
        ]);

        $this->assertEquals(
            $unitCost * $currency->conversion,
            PurchaseOrderLine::first()->amount_in_tenant_currency
        );

        $purchaseOrder = PurchaseOrder::find($response->offsetGet('data')['id']);

        $purchaseOrder->update([
            'currency_code' => 'USD',
        ]);

        $this->assertEquals(
            $unitCost,
            PurchaseOrderLine::first()->amount_in_tenant_currency
        );
    }

    public function test_it_can_cache_and_change_currency_rate_for_sales_credits(): void
    {
        $creditAmount = 5.00;

        $currency = Currency::factory()
            ->create([
                'code' => 'CAD',
                'conversion' => 0.66,
            ]);

        Sanctum::actingAs(User::factory()->create());

        $response = $this->postJson('/api/sales-credits', [
            'order_status' => SalesCredit::CREDIT_STATUS_OPEN,
            'credit_date' => $this->faker->dateTimeThisMonth()->format('Y-m-d H:i:s'),
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => $currency->code,
            'sales_credit_lines' => [
                [
                    'description' => $this->faker->sentence(3),
                    'amount' => $creditAmount,
                    'quantity' => 1,
                ],
            ],
        ])->assertSuccessful();

        $this->assertDatabaseHas((new SalesCredit())->getTable(), [
            'currency_id' => $currency->id,
            'currency_rate' => $currency->conversion,
            'currency_id_tenant_snapshot' => Currency::default()->id,
        ]);

        $this->assertEquals(
            $creditAmount * $currency->conversion,
            SalesCreditLine::first()->amount_in_tenant_currency
        );

        $salesCredit = SalesCredit::find($response->offsetGet('data')['id']);

        $salesCredit->update([
            'currency_code' => 'USD',
        ]);

        $this->assertEquals(
            $creditAmount,
            SalesCreditLine::first()->amount_in_tenant_currency
        );
    }

    public function test_it_can_cache_and_change_currency_rate_for_payments(): void
    {
        $paymentAmount = 5.00;

        $currency = Currency::factory()
            ->create([
                'code' => 'CAD',
                'conversion' => 0.66,
            ]);

        $product = Product::factory()
            ->hasProductPricing(1, [
                'price' => 5.00,
            ])->create()->first();

        $productPrice = $product->getDefaultPricing(ProductPricingTier::default()->id)->price;

        Sanctum::actingAs(User::factory()->create());

        $salesOrderResponse = $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => $this->faker->dateTimeThisMonth()->format('Y-m-d H:i:s'),
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => $currency->code,
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => $product->name,
                    'amount' => $productPrice,
                    'quantity' => 1,
                ],
            ],
        ])->assertSuccessful();

        $paymentResponse = $this->postJson('/api/sales-orders/'.$salesOrderResponse->offsetGet('data')['id'].'/payments', [
            'amount' => $paymentAmount,
            'payment_type_id' => PaymentType::factory()->create()->id,
            'currency_id' => $currency->id,
        ])->assertSuccessful();

        $this->assertDatabaseHas((new Payment())->getTable(), [
            'currency_id' => $currency->id,
            'currency_rate' => $currency->conversion,
            'currency_id_tenant_snapshot' => Currency::default()->id,
        ]);

        $this->assertEquals(
            $paymentAmount * $currency->conversion,
            Payment::first()->amount_in_tenant_currency
        );

        $payment = Payment::find($paymentResponse->offsetGet('data')['id']);

        $payment->update([
            'currency_id' => Currency::default()->id,
        ]);

        $this->assertEquals(
            $paymentAmount,
            Payment::first()->amount_in_tenant_currency
        );
    }

    public function test_it_can_update_profit_report_when_currency_changed_for_sales_order(): void
    {
        Queue::fake();

        $customer = Customer::factory()->create();

        $product = Product::factory()
            ->hasProductPricing(1, [
                'price' => 5.00,
            ])->create()->first();

        $productPrice = $product->getDefaultPricing(ProductPricingTier::default()->id)->price;

        $currency = Currency::factory()
            ->create([
                'code' => 'CAD',
                'conversion' => 0.66,
            ]);

        Sanctum::actingAs(User::factory()->create([
            'is_power_user' => true,
        ]));

        $salesOrderResponse = $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => $this->faker->dateTimeThisMonth()->format('Y-m-d H:i:s'),
            'customer_id' => $customer->id,
            'currency_code' => Currency::default()->code,
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => $product->name,
                    'amount' => $productPrice,
                    'quantity' => 1,
                ],
            ],
        ])->assertSuccessful();

        $salesOrder = SalesOrder::find($salesOrderResponse->offsetGet('data')['id']);

        (new SalesOrderLineFinancialManager(Arr::wrap($salesOrder->id)))->calculate();
        (new DailyFinancialManager())->calculate();

        $profitReportResponse = $this->getJson('/api/sales-orders/'.$salesOrder->id.'/financials');
        $this->assertEquals(
            5,
            collect($profitReportResponse->offsetGet('data')['sales_order_lines'])->sum('total_revenue')
        );

        $dailyProfitReportResponse = $this->getJson('/api/financials/summary-by-period?period=day');

        $this->assertEquals(
            5,
            collect($dailyProfitReportResponse->offsetGet('data'))->sum('total_revenue')
        );

        $dailyProductProfitReportResponse = $this->getJson('/api/financials/products/'.$product->id.'/summary-by-period?period=day');

        $this->assertEquals(
            5,
            collect($dailyProductProfitReportResponse->offsetGet('data'))->sum('total_revenue')
        );

        $salesOrder->update([
            'currency_code' => 'CAD',
        ]);

        $profitReportResponse = $this->getJson('/api/sales-orders/'.$salesOrder->id.'/financials');

        $this->assertEquals(
            round(5 * $currency->conversion, 4),
            round(collect($profitReportResponse->offsetGet('data')['sales_order_lines'])->sum('total_revenue'), 4)
        );

        $dailyProfitReportResponse = $this->getJson('/api/financials/summary-by-period?period=day');

        $this->assertEquals(
            round(5 * $currency->conversion, 4),
            round(collect($dailyProfitReportResponse->offsetGet('data'))->sum('total_revenue'), 4)
        );

        (new DailyFinancialManager())->calculate();

        $dailyProductProfitReportResponse = $this->getJson('/api/financials/products/'.$product->id.'/summary-by-period?period=day');

        $this->assertEquals(
            round(5 * $currency->conversion, 4),
            round(collect($dailyProductProfitReportResponse->offsetGet('data'))->sum('total_revenue'), 4)
        );
    }

    /*
     * TODO: In progress
     */
    /*public function test_it_can_update_profit_report_when_currency_changed_for_purchase_order()
    {
        $this->withoutJobs();
        Event::fake();

        $unitCost = 3.00;

        $warehouse = Warehouse::factory()->create([
            'type' => Warehouse::TYPE_SUPPLIER,
        ]);
        $supplier = Supplier::factory()->create([
            'default_warehouse_id' => $warehouse->id
        ]);
        $customer = Customer::factory()->create();

        $product = Product::factory()->has(
                SupplierProduct::factory(1, [
                    'supplier_id' => $supplier->id
                ])
            )
            ->hasProductPricing(1, [
                'price' => 5.00,
            ])->create([
                'unit_cost' => $unitCost
            ])->first();

        $productPrice = $product->getDefaultPricing(ProductPricingTier::default()->id)->price;

        $currency = Currency::factory()
            ->create([
                'code' => 'CAD',
                'conversion' => 0.66,
            ]);

        Sanctum::actingAs(User::factory()->create([
            'is_power_user' => true
        ]));

        $salesOrderResponse = $this->postJson('/api/sales-orders', [
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => $this->faker->dateTimeThisMonth()->format("Y-m-d H:i:s"),
            'customer_id' => $customer->id,
            'currency_code' => Currency::default()->code,
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => $product->name,
                    'amount' => $productPrice,
                    'quantity' => 1
                ]
            ]
        ])->assertSuccessful();

        $salesOrder = SalesOrder::find($salesOrderResponse->offsetGet('data')['id']);

        $response = $this->postJson('/api/purchase-orders', [
            'order_status' => PurchaseOrder::STATUS_DRAFT,
            'purchase_order_date' => $this->faker->dateTimeThisMonth()->format("Y-m-d H:i:s"),
            'supplier_id' => $supplier->id,
            'currency_code' => Currency::default()->code,
            'purchase_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => $product->name,
                    'amount' => $product->unit_cost,
                    'quantity' => 1
                ]
            ]
        ])->assertSuccessful();

        $purchaseOrder = PurchaseOrder::find($response->offsetGet('data')['id']);

        $response = $this->putJson('/api/purchase-orders/' . $purchaseOrder->id, [
            'approval_status' => 'approved',
        ]);

        $response = $this->postJson('/api/purchase-order-shipments/receive', [
            'received_at' => Carbon::now()->format("Y-m-d"),
            'purchase_order_id' => $purchaseOrder->id,
            'receipt_lines' => [
                [
                    'purchase_order_line_id' => $purchaseOrder->purchaseOrderLines->first()->id,
                    'quantity' => 1
                ]
            ]
        ]);
dd($response->decodeResponseJson());
        $profitReportResponse = $this->getJson('/api/sales-orders/' . $salesOrder->id . '/financials');
dd($profitReportResponse);
        $this->assertEquals(
            5,
            collect($profitReportResponse->offsetGet('data')['sales_order_lines'])->sum('proforma_total_revenue')
        );

        $dailyProfitReportResponse = $this->getJson('/api/financials/summary-by-period?period=day');

        $this->assertEquals(
            5,
            collect($dailyProfitReportResponse->offsetGet('data'))->sum('total_revenue')
        );

        $dailyProductProfitReportResponse = $this->getJson('/api/financials/products/' . $product->id . '/summary-by-period?period=day');

        $this->assertEquals(
            5,
            collect($dailyProductProfitReportResponse->offsetGet('data'))->sum('total_revenue')
        );

        $salesOrder->update([
            'currency_code' => 'CAD'
        ]);

        $profitReportResponse = $this->getJson('/api/sales-orders/' . $salesOrder->id . '/financials');

        $this->assertEquals(
            round(5 * $currency->conversion, 4),
            round(collect($profitReportResponse->offsetGet('data')['sales_order_lines'])->sum('proforma_total_revenue'), 4)
        );

        $dailyProfitReportResponse = $this->getJson('/api/financials/summary-by-period?period=day');

        $this->assertEquals(
            round(5 * $currency->conversion, 4),
            round(collect($dailyProfitReportResponse->offsetGet('data'))->sum('total_revenue'), 4)
        );

        (new FinancialReportingProductManager())->calculate();

        $dailyProductProfitReportResponse = $this->getJson('/api/financials/products/' . $product->id . '/summary-by-period?period=day');

        $this->assertEquals(
            round(5 * $currency->conversion, 4),
            round(collect($dailyProductProfitReportResponse->offsetGet('data'))->sum('total_revenue'), 4)
        );
    }*/
}
