<?php

namespace Tests\Feature;

use App\Http\Requests\StoreInventoryAdjustment;
use App\Integrations\ShipStation;
use App\Models\Address;
use App\Models\Customer;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\Product;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\User;
use App\Models\Warehouse;
use App\Repositories\SettingRepository;
use App\SDKs\ShipStation\Model\ShipStationWebhook;
use App\SDKs\ShipStation\Responses\CreateUpdateOrdersResponse;
use App\SDKs\ShipStation\Responses\ListShipmentsResponse;
use App\Services\ShippingProvider\ShipStationManager;
use Illuminate\Foundation\Testing\WithFaker;
use Laravel\Sanctum\Sanctum;
use Mockery;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class CreateShipstationWebhooksTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    public function test_it_can_create_webhooks_event_logs(): void
    {
        \Queue::fake();
        // Mock the dependencies
        $integration = Integration::query()->where('name', Integration::NAME_SHIPSTATION)->firstOrFail();
        $integrationInstance = IntegrationInstance::factory()->create([
            'integration_id' => $integration->id,
            'connection_settings' => [
                'apiKey' => ' ',
                'apiSecret' => ' ',
            ],
        ]);
        $shipStationMock = Mockery::mock(ShipStation::class);
        $this->app->instance(ShipStation::class, $shipStationMock);
        $routeName = 'webhooks.shipstation';
        $targetUrl = route($routeName);
        $event = ShipstationWebhook::TYPE_SHIP_NOTIFY;
        $shipstationWebhook = new ShipstationWebhook();
        $shipstationWebhook->event = $event;
        $shipstationWebhook->target_url = $targetUrl;

        // Mock the ShipStation service
        $shipStationMock->shouldReceive('createWebhook')
            ->once()
            ->with(Mockery::type(ShipstationWebhook::class))
            ->andReturn($shipstationWebhook);
        $this->app->bind(ShipStation::class, function ($app) use ($shipStationMock) {
            return $shipStationMock;
        });

        // Call the method under test
        $shipStationManager = new ShipStationManager();
        $shipStationManager->createWebhook($integrationInstance, $event, $routeName);
        // Assert the webhook is saved to the database
        $this->assertDatabaseCount('shipstation_webhooks', 1);
    }

    public function test_it_can_shopify_order_fulfilled_by_webhook(): void
    {
        $integration = Integration::query()->where('name', Integration::NAME_SHIPSTATION)->firstOrFail();

        $integrationInstance = IntegrationInstance::factory()->create([
            'integration_id' => $integration->id,
            'name' => 'Starshipit',
            'connection_settings' => [
                'apiKey' => ' ',
                'apiSecret' => ' ',
            ],
        ]);

        /** @var Customer $customer */
        $customer = Customer::factory()->create();
        /** @var Product $product */
        $product = Product::factory()->create([
            'dimension_unit' => 'in',
            'weight_unit' => 'lb',
            'weight' => 3,
            'length' => 5,
            'width' => 5,
            'height' => 5,
        ]);
        /** @var Address $address */
        $address = Address::factory()->create();
        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::factory()->create([
            'address_id' => $address->id,
        ])->withDefaultLocation();

        Sanctum::actingAs(User::factory()->create());

        $timeOfOrder = SettingRepository::getInventoryStartDate()->addDay()->format('Y-m-d H:i:s');

        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => $timeOfOrder,
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 5.00,
            'quantity' => 50,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ])->assertSuccessful();

        $abc = 'SSI_test'.$this->faker->randomNumber(5);
        $salesOrderResponse = $this->postJson('/api/sales-orders', [
            'sales_order_number' => $abc,
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => $timeOfOrder,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => $product->name,
                    'warehouse_id' => $warehouse->id,
                    'amount' => 5.00,
                    'quantity' => 10,
                ],
            ],
        ]);

        $salesOrderResponse->assertSuccessful();
        $response = new CreateUpdateOrdersResponse(
            json_decode('{"orderId":1039899986,"orderNumber":"SO-0005.1","orderKey":"SKU.io_41","orderDate":"2022-11-02T06:01:23.0000000","createDate":"2022-11-02T00:14:56.7770000","modifyDate":"2023-05-23T03:43:21.2800000","paymentDate":null,"shipByDate":"2022-11-02T06:01:23.0000000","orderStatus":"awaiting_shipment","customerId":483391681,"customerUsername":"Brett Seidman","customerEmail":"s@test.com","billTo":{"name":"kalven","company":null,"street1":"TEST 123","street2":"","street3":null,"city":"SANTA MONICA","state":"California","postalCode":"90401-0000","country":"US","phone":"17348903043","residential":null,"addressVerified":null},"shipTo":{"name":"kalven","company":null,"street1":"TEST 123","street2":null,"street3":null,"city":"SANTA MONICA","state":"CA","postalCode":"90401-0000","country":"US","phone":"17348903043","residential":false,"addressVerified":"Address validation warning"},"items":[{"orderItemId":1663800849,"lineItemKey":"5","sku":"BB001","name":"Baby Bumble Onesie","imageUrl":"https:\/\/i.imgur.com\/AHjexCj.jpg","weight":{"value":0,"units":"ounces","WeightUnits":1},"quantity":1,"unitPrice":15.99,"taxAmount":0,"shippingAmount":0,"warehouseLocation":null,"options":[],"productId":62894490,"fulfillmentSku":null,"adjustment":false,"upc":"123456789012","createDate":"2023-05-23T03:43:21.28","modifyDate":"2023-05-23T03:43:21.28"}],"orderTotal":15.99,"amountPaid":0,"taxAmount":0,"shippingAmount":0,"customerNotes":null,"internalNotes":null,"gift":false,"giftMessage":null,"paymentMethod":null,"requestedShippingService":"UPS Ground","carrierCode":null,"serviceCode":null,"packageCode":null,"confirmation":"none","shipDate":null,"holdUntilDate":null,"weight":{"value":0,"units":"ounces","WeightUnits":1},"dimensions":null,"insuranceOptions":{"provider":null,"insureShipment":false,"insuredValue":0},"internationalOptions":{"contents":"merchandise","customsItems":null,"nonDelivery":"return_to_sender"},"advancedOptions":{"warehouseId":151573,"nonMachinable":false,"saturdayDelivery":false,"containsAlcohol":false,"mergedOrSplit":false,"mergedIds":[],"parentId":null,"storeId":258511,"customField1":null,"customField2":null,"customField3":null,"source":null,"billToParty":null,"billToAccount":null,"billToPostalCode":null,"billToCountryCode":null,"billToMyOtherAccount":null},"tagIds":null,"userId":null,"externallyFulfilled":false,"externallyFulfilledBy":null,"externallyFulfilledById":null,"externallyFulfilledByName":null,"labelMessages":null}',
                1)
        );

        $salesOrderId = $salesOrderResponse->json('data.id');
        $shipStationMock = Mockery::mock(ShipStation::class);
        $shipStationMock->shouldReceive('submitOrder')
            ->once()
            ->andReturn($response);
        $this->app->bind(ShipStation::class, function ($app) use ($shipStationMock) {
            return $shipStationMock;
        });
        $this->app->instance(ShipStation::class, $shipStationMock);

        $responseJson = $this->postJson("/api/sales-orders/$salesOrderId/fulfill", [
            'fulfilled_at' => $timeOfOrder,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrderResponse->json('data.item_info.0.sales_order_line_id'),
                    'quantity' => 2,
                ],
            ],
            'fulfillment_type' => SalesOrderFulfillment::TYPE_SHIPSTATION,
            'metadata' => ['signature_required' => false],
            'requested_shipping_method' => 'UPS Ground',
            'requested_shipping_method_id' => '',
            'warehouse' => $warehouse->id,
        ]);

        $this->assertEquals(SalesOrderFulfillment::STATUS_SUBMITTED, SalesOrderFulfillment::first()->status);

        //getWebhooksResult
        $data = $this->getShipmentsData();

        $shipStationMock->shouldReceive('getWebhooksResult')
            ->once()
            ->andReturn($data);
        $this->app->bind(ShipStation::class, function ($app) use ($shipStationMock) {
            return $shipStationMock;
        });
        $this->app->instance(ShipStation::class, $shipStationMock);

        $webhook = new ShipStationWebhook();
        $webhook->resource_url = 'https://ssapi6.shipstation.com/shipments?storeID=258511&batchId=276958424&includeShipmentItems=False';
        $webhook->resource_type = 'SHIP_NOTIFY';
        dispatch_sync(new \App\Jobs\ShipStation\ShipStationHandleWebhookJob($integrationInstance, $webhook));

        $this->assertEquals(SalesOrderFulfillment::STATUS_FULFILLED, SalesOrderFulfillment::first()->status);
    }

    public function getShipmentsData()
    {
        return $data = [
            new ListShipmentsResponse([
                'shipments' => [
                    [
                        'shipmentId' => 682665520,
                        'orderId' => 1039899986,
                        'orderKey' => 'SKU.io_40',
                        'userId' => '4e97c843-15fd-4c75-b8c2-2d69307e7ddf',
                        'customerEmail' => 's@test.com',
                        'orderNumber' => 'SO-0005.1',
                        'createDate' => '2023-05-23T03:40:25.8670000',
                        'shipDate' => '2023-05-23',
                        'shipmentCost' => 3.66,
                        'insuranceCost' => 0.0,
                        'trackingNumber' => '9400111206214033491191',
                        'isReturnLabel' => false,
                        'batchNumber' => null,
                        'carrierCode' => 'stamps_com',
                        'serviceCode' => 'usps_first_class_mail',
                        'packageCode' => 'package',
                        'confirmation' => 'delivery',
                        'warehouseId' => 151573,
                        'voided' => true,
                        'voidDate' => '2023-05-23T03:46:41.0100000',
                        'marketplaceNotified' => false,
                        'notifyErrorMessage' => null,
                        'shipTo' => [
                            'name' => 'kalven',
                            'company' => null,
                            'street1' => 'TEST 123',
                            'street2' => null,
                            'street3' => null,
                            'city' => 'SANTA MONICA',
                            'state' => 'CA',
                            'postalCode' => '90401-0000',
                            'country' => 'US',
                            'phone' => '17348903043',
                            'residential' => null,
                            'addressVerified' => null,
                        ],
                        'weight' => [
                            'value' => 0.0,
                            'units' => 'ounces',
                            'WeightUnits' => 1,
                        ],
                        'dimensions' => null,
                        'insuranceOptions' => [
                            'provider' => null,
                            'insureShipment' => false,
                            'insuredValue' => 0.0,
                        ],
                        'advancedOptions' => [
                            'billToParty' => '4',
                            'billToAccount' => null,
                            'billToPostalCode' => null,
                            'billToCountryCode' => null,
                            'storeId' => 258511,
                        ],
                        'shipmentItems' => null,
                        'labelData' => null,
                        'formData' => null,
                    ],
                ],
                'total' => 1,
                'page' => 1,
                'pages' => 1,
            ]),
        ];
    }

    public function test_shipstations_can_get_shipments_and_update_sales_order_fulfillment_status(): void
    {
        $integration = Integration::query()->where('name', Integration::NAME_SHIPSTATION)->firstOrFail();

        $integrationInstance = IntegrationInstance::factory()->create([
            'integration_id' => $integration->id,
            'name' => 'Starshipit',
            'connection_settings' => [
                'apiKey' => ' ',
                'apiSecret' => ' ',
            ],
        ]);

        /** @var Customer $customer */
        $customer = Customer::factory()->create();
        /** @var Product $product */
        $product = Product::factory()->create([
            'dimension_unit' => 'in',
            'weight_unit' => 'lb',
            'weight' => 3,
            'length' => 5,
            'width' => 5,
            'height' => 5,
        ]);
        /** @var Address $address */
        $address = Address::factory()->create();
        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::factory()->create([
            'address_id' => $address->id,
        ])->withDefaultLocation();

        $shopifyIntergration = IntegrationInstance::factory()->create([
            'integration_id' => Integration::where('name', 'Shopify')->first()->id,
            'name' => 'Shopify',
            'connection_settings' => [
                'apiKey' => ' ',
                'apiSecret' => ' ',
            ],
            'is_automatic_sync_enabled' => true,
        ]);
        Sanctum::actingAs(User::factory()->create());

        $timeOfOrder = SettingRepository::getInventoryStartDate()->addDay()->format('Y-m-d H:i:s');

        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => $timeOfOrder,
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 5.00,
            'quantity' => 50,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ])->assertSuccessful();

        $salesOrderResponse = $this->postJson('/api/sales-orders', [
            'sales_order_number' => 'SSI_test'.$this->faker->randomNumber(5),
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => $timeOfOrder,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'sales_channel_id' => SalesChannel::factory()->create(['integration_instance_id' => $shopifyIntergration->id])->id,
            'integration_instance_id' => $integrationInstance->id,
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => $product->name,
                    'warehouse_id' => $warehouse->id,
                    'amount' => 5.00,
                    'quantity' => 10,
                ],
            ],
        ]);

        $salesOrderResponse->assertSuccessful();
        $response = new CreateUpdateOrdersResponse(
            json_decode('{"orderId":1039899986,"orderNumber":"SO-0005.1","orderKey":"SKU.io_41","orderDate":"2022-11-02T06:01:23.0000000","createDate":"2022-11-02T00:14:56.7770000","modifyDate":"2023-05-23T03:43:21.2800000","paymentDate":null,"shipByDate":"2022-11-02T06:01:23.0000000","orderStatus":"awaiting_shipment","customerId":483391681,"customerUsername":"Brett Seidman","customerEmail":"s@test.com","billTo":{"name":"kalven","company":null,"street1":"TEST 123","street2":"","street3":null,"city":"SANTA MONICA","state":"California","postalCode":"90401-0000","country":"US","phone":"17348903043","residential":null,"addressVerified":null},"shipTo":{"name":"kalven","company":null,"street1":"TEST 123","street2":null,"street3":null,"city":"SANTA MONICA","state":"CA","postalCode":"90401-0000","country":"US","phone":"17348903043","residential":false,"addressVerified":"Address validation warning"},"items":[{"orderItemId":1663800849,"lineItemKey":"5","sku":"BB001","name":"Baby Bumble Onesie","imageUrl":"https:\/\/i.imgur.com\/AHjexCj.jpg","weight":{"value":0,"units":"ounces","WeightUnits":1},"quantity":1,"unitPrice":15.99,"taxAmount":0,"shippingAmount":0,"warehouseLocation":null,"options":[],"productId":62894490,"fulfillmentSku":null,"adjustment":false,"upc":"123456789012","createDate":"2023-05-23T03:43:21.28","modifyDate":"2023-05-23T03:43:21.28"}],"orderTotal":15.99,"amountPaid":0,"taxAmount":0,"shippingAmount":0,"customerNotes":null,"internalNotes":null,"gift":false,"giftMessage":null,"paymentMethod":null,"requestedShippingService":"UPS Ground","carrierCode":null,"serviceCode":null,"packageCode":null,"confirmation":"none","shipDate":null,"holdUntilDate":null,"weight":{"value":0,"units":"ounces","WeightUnits":1},"dimensions":null,"insuranceOptions":{"provider":null,"insureShipment":false,"insuredValue":0},"internationalOptions":{"contents":"merchandise","customsItems":null,"nonDelivery":"return_to_sender"},"advancedOptions":{"warehouseId":151573,"nonMachinable":false,"saturdayDelivery":false,"containsAlcohol":false,"mergedOrSplit":false,"mergedIds":[],"parentId":null,"storeId":258511,"customField1":null,"customField2":null,"customField3":null,"source":null,"billToParty":null,"billToAccount":null,"billToPostalCode":null,"billToCountryCode":null,"billToMyOtherAccount":null},"tagIds":null,"userId":null,"externallyFulfilled":false,"externallyFulfilledBy":null,"externallyFulfilledById":null,"externallyFulfilledByName":null,"labelMessages":null}',
                1)
        );

        $salesOrderId = $salesOrderResponse->json('data.id');

        $salesOrder = SalesOrder::find($salesOrderId);

        $salesOrderLine = $salesOrder->salesOrderLines()->first();

        $shipStationMock = Mockery::mock(ShipStation::class);
        $shipStationMock->shouldReceive('submitOrder')
            ->once()
            ->andReturn($response);
        $this->app->bind(ShipStation::class, function ($app) use ($shipStationMock) {
            return $shipStationMock;
        });
        $this->app->instance(ShipStation::class, $shipStationMock);

        $response = $this->postJson("/api/sales-orders/$salesOrderId/fulfill", [
            'fulfilled_at' => $timeOfOrder,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrderLine->id,
                    'quantity' => 2,
                ],
            ],
            'fulfillment_type' => SalesOrderFulfillment::TYPE_SHIPSTATION,
            'metadata' => ['signature_required' => false],
            'requested_shipping_method' => 'UPS Ground',
            'requested_shipping_method_id' => '',
            'warehouse' => $warehouse->id,
        ]);

        $this->assertEquals(SalesOrderFulfillment::STATUS_SUBMITTED, SalesOrderFulfillment::first()->status);

        //getWebhooksResult
        $data = $this->getShipmentsData();

        $shipStationMock->shouldReceive('getShipments')
            ->once()
            ->andReturn($data);
        $this->app->bind(ShipStation::class, function ($app) use ($shipStationMock) {
            return $shipStationMock;
        });
        $this->app->instance(ShipStation::class, $shipStationMock);
        dispatch_sync(new \App\Jobs\ShipStation\GetShipments($integrationInstance));
        $this->assertEquals(SalesOrderFulfillment::STATUS_FULFILLED, SalesOrderFulfillment::first()->status);
        $response->assertSuccessful();
    }
}
