<?php

namespace Tests\Feature\Controllers;

use App\Data\UpdateWarehouseTransferLinesData;
use App\Data\WarehouseTransferReceiptData;
use App\Data\WarehouseTransferReceiptProductData;
use App\Exceptions\WarehouseTransfers\NotOpenWarehouseTransferException;
use App\Helpers;
use App\Managers\WarehouseTransferManager;
use App\Models\AccountingTransaction;
use App\Models\AccountingTransactionLine;
use App\Models\NominalCode;
use App\Models\Product;
use App\Models\Setting;
use App\Models\User;
use App\Models\Warehouse;
use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferLine;
use App\Models\WarehouseTransferShipmentReceipt;
use App\Repositories\SettingRepository;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class WarehouseTransferControllerTest extends TestCase
{
    use FastRefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        Sanctum::actingAs(User::first());
    }

    /**
     * @throws Throwable
     * @throws NotOpenWarehouseTransferException
     */
    public function test_it_can_create_receiving_discrepancy_loss_from_warehouse_transfer(): void
    {
        $nc = NominalCode::create([
            'name' => 'Lost In Transit Expense',
            'code' => '601',
            'type' => NominalCode::TYPE_EXPENSE,
        ]);

        app(SettingRepository::class)->set(Setting::KEY_NC_MAPPING_RECEIVING_DISCREPANCIES, $nc->id);

        $product     = Product::factory()->create();
        $warehouse   = Warehouse::first();
        $toWarehouse = Warehouse::factory()->create([
            'type' => Warehouse::TYPE_DIRECT
        ]);

        $product->setInitialInventory($warehouse->id, 5, 100);

        $warehouseTransfer = WarehouseTransfer::factory()
            ->hasWarehouseTransferLines(1, [
                'quantity' => 5,
                'product_id' => $product->id,
            ])
            ->create([
                'transfer_status' => WarehouseTransfer::TRANSFER_STATUS_DRAFT,
                'transfer_date' => now(),
                'from_warehouse_id' => $warehouse->id,
                'to_warehouse_id' => $toWarehouse->id,
            ]);

        $warehouseTransferLine = $warehouseTransfer->warehouseTransferLines()->first();

        app(WarehouseTransferManager::class)->openWarehouseTransfer($warehouseTransfer, []);

        $warehouseTransfer->refresh();
        $warehouseTransferShipment = $warehouseTransfer->shipments->first();

        app(WarehouseTransferManager::class)->receiveShipment($warehouseTransfer, WarehouseTransferReceiptData::from([
            'shipment_id' => $warehouseTransferShipment->id,
            'receipt_date' => now(),
            'products' => WarehouseTransferReceiptProductData::collection([
                WarehouseTransferReceiptProductData::from([
                    'id' => $warehouseTransferLine->product->id,
                    'quantity' => 3,
                ]),
            ]),
        ]));

        $this->assertDatabaseHas(WarehouseTransferShipmentReceipt::class, [
            'warehouse_transfer_shipment_id' => $warehouseTransferShipment->id,
        ]);

        $response = $this->postJson(route('warehouses.transfers.create-receiving-discrepancy', $warehouseTransfer->id));

        $response->assertOk();

        $this->assertDatabaseHas(AccountingTransaction::class, [
            'type' => AccountingTransaction::TYPE_RECEIVING_DISCREPANCY,
            'name' => $warehouseTransfer->toWarehouse->name,
            'reference' => $warehouseTransfer->fromWarehouse->name.'->'.$warehouseTransfer->toWarehouse->name.': Receiving discrepancy for '.$warehouseTransfer->warehouse_transfer_number,
            'link_id' => $warehouseTransfer->id,
            'link_type' => WarehouseTransfer::class,
        ]);

        $this->assertDatabaseHas(AccountingTransactionLine::class, [
            'type' => AccountingTransactionLine::TYPE_DEBIT,
            'nominal_code_id' => Helpers::setting(Setting::KEY_NC_MAPPING_RECEIVING_DISCREPANCIES),
            'description' => 'Receiving discrepancy for '.$warehouseTransferLine->product->sku,
            'quantity' => 2,
            'amount' => 100,
            'link_id' => $warehouseTransferLine->id,
            'link_type' => WarehouseTransferLine::class,
        ]);

        $this->assertDatabaseHas(AccountingTransactionLine::class, [
            'type' => AccountingTransactionLine::TYPE_CREDIT,
            'nominal_code_id' => Helpers::setting(Setting::KEY_NC_MAPPING_INVENTORY_IN_TRANSIT),
            'description' => 'Receiving discrepancy for '.$warehouseTransferLine->product->sku,
            'quantity' => 2,
            'amount' => 100,
            'link_id' => $warehouseTransferLine->id,
            'link_type' => WarehouseTransferLine::class,
        ]);

        // Assert bad request due to already created discrepancy
        $this->postJson(route('warehouses.transfers.create-receiving-discrepancy', $warehouseTransfer->id))
            ->assertBadRequest()
            ->assertJsonPath('message',
                'Receiving discrepancy already exists for warehouse transfer '.$warehouseTransfer->warehouse_transfer_number);
    }

    /**
     * @throws Throwable
     * @throws NotOpenWarehouseTransferException
     */
    public function test_it_can_create_receiving_discrepancy_gain_from_warehouse_transfer(): void
    {
        $nc = NominalCode::create([
            'name' => 'Lost In Transit Expense',
            'code' => '601',
            'type' => NominalCode::TYPE_EXPENSE,
        ]);

        app(SettingRepository::class)->set(Setting::KEY_NC_MAPPING_RECEIVING_DISCREPANCIES, $nc->id);

        $product = Product::factory()->create();
        $warehouse = Warehouse::first();
        $toWarehouse = Warehouse::factory()->create([
            'type' => Warehouse::TYPE_DIRECT
        ]);

        $product->setInitialInventory($warehouse->id, 100, 100);

        $warehouseTransfer = WarehouseTransfer::factory()
            ->hasWarehouseTransferLines(1, [
                'quantity' => 5,
                'product_id' => $product->id,
            ])
            ->create([
                'transfer_status' => WarehouseTransfer::TRANSFER_STATUS_DRAFT,
                'transfer_date' => now(),
                'from_warehouse_id' => $warehouse->id,
                'to_warehouse_id' => $toWarehouse->id,
            ]);

        $warehouseTransferLine = $warehouseTransfer->warehouseTransferLines()->first();

        app(WarehouseTransferManager::class)->openWarehouseTransfer($warehouseTransfer, []);

        $warehouseTransfer->refresh();
        $warehouseTransferShipment = $warehouseTransfer->shipments->first();

        app(WarehouseTransferManager::class)->receiveShipment($warehouseTransfer, WarehouseTransferReceiptData::from([
            'shipment_id' => $warehouseTransferShipment->id,
            'receipt_date' => now(),
            'products' => WarehouseTransferReceiptProductData::collection([
                WarehouseTransferReceiptProductData::from([
                    'id' => $warehouseTransferLine->product->id,
                    'quantity' => 7,
                ]),
            ]),
        ]));

        $this->assertDatabaseHas(WarehouseTransferShipmentReceipt::class, [
            'warehouse_transfer_shipment_id' => $warehouseTransferShipment->id,
        ]);

        $response = $this->postJson(route('warehouses.transfers.create-receiving-discrepancy', $warehouseTransfer->id));

        $response->assertOk();

        $this->assertDatabaseHas(AccountingTransaction::class, [
            'type' => AccountingTransaction::TYPE_RECEIVING_DISCREPANCY,
            'name' => $warehouseTransfer->toWarehouse->name,
            'reference' => $warehouseTransfer->fromWarehouse->name . '->' . $warehouseTransfer->toWarehouse->name . ': Receiving discrepancy for ' . $warehouseTransfer->warehouse_transfer_number,
            'link_id' => $warehouseTransfer->id,
            'link_type' => WarehouseTransfer::class,
        ]);

        $this->assertDatabaseHas(AccountingTransactionLine::class, [
            'type' => AccountingTransactionLine::TYPE_DEBIT,
            'nominal_code_id' => Helpers::setting(Setting::KEY_NC_MAPPING_INVENTORY_IN_TRANSIT),
            'description' => 'Receiving discrepancy for ' . $warehouseTransferLine->product->sku,
            'quantity' => 2,
            'amount' => 100,
            'link_id' => $warehouseTransferLine->id,
            'link_type' => WarehouseTransferLine::class,
        ]);

        $this->assertDatabaseHas(AccountingTransactionLine::class, [
            'type' => AccountingTransactionLine::TYPE_CREDIT,
            'nominal_code_id' => Helpers::setting(Setting::KEY_NC_MAPPING_RECEIVING_DISCREPANCIES),
            'description' => 'Receiving discrepancy for ' . $warehouseTransferLine->product->sku,
            'quantity' => 2,
            'amount' => 100,
            'link_id' => $warehouseTransferLine->id,
            'link_type' => WarehouseTransferLine::class,
        ]);

        // Assert bad request due to already created discrepancy
        $this->postJson(route('warehouses.transfers.create-receiving-discrepancy', $warehouseTransfer->id))
            ->assertBadRequest()
            ->assertJsonPath('message', 'Receiving discrepancy already exists for warehouse transfer ' . $warehouseTransfer->warehouse_transfer_number);
    }

    /**
     * @throws Throwable
     */
    public function test_user_can_set_quantities_shipped_before_start_date(): void
    {
        $product     = Product::factory()->create();
        $warehouse   = Warehouse::first();
        $toWarehouse = Warehouse::factory()->create([
            'type' => Warehouse::TYPE_DIRECT
        ]);

        $product->setInitialInventory($warehouse->id, 5, 100);

        $warehouseTransfer = WarehouseTransfer::factory()
            ->hasWarehouseTransferLines(1, [
                'quantity' => 5,
                'product_id' => $product->id,
            ])
            ->create([
                'transfer_status' => WarehouseTransfer::TRANSFER_STATUS_DRAFT,
                'transfer_date' => now(),
                'from_warehouse_id' => $warehouse->id,
                'to_warehouse_id' => $toWarehouse->id,
            ]);

        app(WarehouseTransferManager::class)->openWarehouseTransfer($warehouseTransfer, []);
        $warehouseTransferShipment = $warehouseTransfer->shipments->first();
        $warehouseTransferLine = $warehouseTransfer->warehouseTransferLines->first();

        app(WarehouseTransferManager::class)->receiveShipment($warehouseTransfer, WarehouseTransferReceiptData::from([
            'shipment_id' => $warehouseTransferShipment->id,
            'receipt_date' => now(),
            'products' => WarehouseTransferReceiptProductData::collection([
                WarehouseTransferReceiptProductData::from([
                    'id' => $warehouseTransferLine->product->id,
                    'quantity' => 2,
                ]),
            ]),
        ]));
        $warehouseTransfer->refresh();

        $this->assertEquals(0, $warehouseTransferLine->quantity_shipped_before_start_date);

        $this->assertEquals(WarehouseTransfer::TRANSFER_STATUS_OPEN, $warehouseTransfer->transfer_status);
        $this->assertEquals(WarehouseTransfer::TRANSFER_SHIPMENT_STATUS_SHIPPED, $warehouseTransfer->shipment_status);
        $this->assertEquals(WarehouseTransfer::TRANSFER_RECEIPT_STATUS_PARTIALLY_RECEIVED, $warehouseTransfer->receipt_status);

        $data = UpdateWarehouseTransferLinesData::from([
            'ids' => $warehouseTransfer->warehouseTransferLines->pluck('id')->toArray(),
            'quantity_shipped_before_start_date' => 3
        ]);
        $this->patchJson(route('warehouses.transfers.lines', $warehouseTransfer->id), $data->toArray())->assertOk();

        $warehouseTransfer->refresh();
        $warehouseTransferLine = $warehouseTransfer->warehouseTransferLines->first();

        $this->assertEquals(3, $warehouseTransferLine->quantity_shipped_before_start_date);

        $this->assertEquals(WarehouseTransfer::TRANSFER_STATUS_CLOSED, $warehouseTransfer->transfer_status);
        $this->assertEquals(WarehouseTransfer::TRANSFER_SHIPMENT_STATUS_SHIPPED, $warehouseTransfer->shipment_status);
        $this->assertEquals(WarehouseTransfer::TRANSFER_RECEIPT_STATUS_RECEIVED, $warehouseTransfer->receipt_status);
    }
}


