<?php

namespace Tests\Feature\Controllers;

use App\Http\Requests\StoreInventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\User;
use App\Models\Warehouse;
use Illuminate\Testing\TestResponse;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class SalesOrderFulfillmentControllerTest extends TestCase
{
    use FastRefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        Sanctum::actingAs(User::first());
    }

    public function test_it_can_delete_fulfillments()
    {
        $salesOrderFulfillment = SalesOrderFulfillment::factory()->create();

        $response = $this->get(route('sales-order-fulfillments.destroy', $salesOrderFulfillment))->assertOk();

        $response->assertStatus(200);
    }

    public function test_it_can_filter_by_fulfillment_number(): void
    {
        Sanctum::actingAs(User::first());
        $filters = [
            "filterSet" => [
                [
                    "column" => "fulfillment_number",
                    "value" => "104838"
                ]
            ]
        ];
        $this->get(route('sales-order-fulfillments.index') . '?' . http_build_query(['filters' => json_encode($filters)]))->assertOk();
    }

    /**
     * @throws Throwable
     */
    public function test_sales_order_fulfillment_via_manual_sequence()
    {
        $warehouse = Warehouse::first();

        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines(1, [
                'warehouse_id' => $warehouse->id,
            ])
            ->create();
        $salesOrderLine = $salesOrder->salesOrderLines->first();

        $product = $salesOrderLine->product;

        $product->setInitialInventory($warehouse->id, 1);

        $response = $this->postJson(route('sales-orders.fulfill', $salesOrder->id),[
            'fulfilled_at' => now()->toDateString(),
            'warehouse_id' => $warehouse->id,
            'sales_order_id' => $salesOrder->id,
            'fulfillment_sequence' => 2,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrderLine->id,
                    'quantity' => 1,
                ]
            ],
            'fulfillment_type' => 'manual'
        ]);

        $response->assertOk();

        $this->assertDatabaseHas(SalesOrderFulfillment::class, [
            'sales_order_id' => $salesOrder->id,
            'fulfillment_sequence' => 2,
        ]);
    }
}
