<?php

namespace Tests\Feature\Controllers;

use App\Helpers;
use App\Models\SalesCreditLine;
use App\Models\SalesCreditReturnLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\User;
use App\Models\Warehouse;
use App\Repositories\SalesOrder\SalesOrderRepository;
use App\Services\InventoryManagement\BulkInventoryManager;
use Illuminate\Contracts\Container\BindingResolutionException;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class SalesCreditControllerTest extends TestCase
{
    use FastRefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        Sanctum::actingAs(User::first());
    }

    /**
     * A basic unit test example.
     */
    public function testFilterProductDatesOfReturnsIfDateExist(): void
    {
        $date = '2022-12-08';
        SalesOrder::factory()->createWithFullSalesCreditFullyReturned(3, $date);
        $count = SalesCreditReturnLine::whereHas('salesCreditReturn', function ($q) use ($date) {
            $q->whereDate('received_at', $date);
        })->count();
        $dateLocal = Helpers::dateUtcToLocal($date);
        $response = $this->getJson('/api/products?limit=100&filters={"conjunction":"and","filterSet":[{"id":28875211,"column":"date_of_return","operator":"=","dateOption":"exactDate","headerValue":"date_of_return","value":"'.$dateLocal.'"}]}&sortObjs=[]&page=1&table_specifications=0&total=0&archived=0&visible_only=0&included=["sku"]')->decodeResponseJson()['data'];
        $this->assertCount($count, $response);
    }

    public function testFilterProductDatesOfReturnsIfDateGreaterThan(): void
    {
        $date = '2022-12-08';
        SalesOrder::factory()->createWithFullSalesCreditFullyReturned(3, $date);
        $response = $this->getJson('/api/products?limit=100&filters={"conjunction":"and","filterSet":[{"id":28875211,"column":"date_of_return","operator":">","dateOption":"exactDate","headerValue":"date_of_return","value":"2022-12-06"}]}&sortObjs=[]&page=1&table_specifications=0&total=0&archived=0&visible_only=0&included=["sku","image","name","brand","price","average_cost","default_supplier","supplier_pricing","inventory_stock_value"]')->decodeResponseJson()['data'];
        $count = SalesCreditReturnLine::whereHas('salesCreditReturn', function ($q) {
            $q->whereDate('received_at', '>', '2022-12-06');
        })->count();
        $this->assertCount($count, $response);
        $response = $this->getJson('/api/products?limit=100&filters={"conjunction":"and","filterSet":[{"id":28875211,"column":"date_of_return","operator":">","dateOption":"exactDate","headerValue":"date_of_return","value":"2022-12-08"}]}&sortObjs=[]&page=1&table_specifications=0&total=0&archived=0&visible_only=0&included=["sku","image","name","brand","price","average_cost","default_supplier","supplier_pricing","inventory_stock_value"]')->decodeResponseJson()['data'];
        $this->assertCount(0, $response);
    }

    public function testFilterProductDatesOfReturnsIfDateInRange(): void
    {
        $date = '2022-12-08';
        SalesOrder::factory()->createWithFullSalesCreditFullyReturned(3, $date);
        $response = $this->getJson('/api/products?limit=100&filters={"conjunction":"and","filterSet":[{"id":28875211,"column":"date_of_return","operator":"isWithin","dateOption":"range","headerValue":"date_of_return","value":{"from":"2022-12-07","to":"2022-12-31"}}]}&sortObjs=[]&page=1&table_specifications=0&total=0&archived=0&visible_only=0&included=["sku","image","name","brand","price","average_cost","default_supplier","supplier_pricing","inventory_stock_value"]')->decodeResponseJson()['data'];
        $count = SalesCreditReturnLine::whereHas('salesCreditReturn', function ($q) use ($date) {
            $q->whereDate('received_at', $date);
        })->count();
        $this->assertCount($count, $response);
        $response = $this->getJson('/api/products?limit=100&filters={"conjunction":"and","filterSet":[{"id":28875211,"column":"date_of_return","operator":"isWithin","dateOption":"range","headerValue":"date_of_return","value":{"from":"2022-12-10","to":"2022-12-31"}}]}&sortObjs=[]&page=1&table_specifications=0&total=0&archived=0&visible_only=0&included=["sku","image","name","brand","price","average_cost","default_supplier","supplier_pricing","inventory_stock_value"]')->decodeResponseJson()['data'];
        $this->assertCount(0, $response);
    }

    public function testFilterProductDateOfSalesCreditIfDateExist(): void
    {
        $date = '2022-12-08';
        SalesOrder::factory()->createWithFullSalesCreditFullyReturned(3, $date);
        $dateLocal = Helpers::dateUtcToLocal($date);
        $response = $this->getJson('/api/products?limit=100&filters={"conjunction":"and","filterSet":[{"id":2887211,"column":"date_of_sales_credit","operator":"=","dateOption":"exactDate","headerValue":"date_of_sales_credit","value":"'.$dateLocal.'"}]}&sortObjs=[]&page=1&table_specifications=0&total=0&archived=0&visible_only=0&included=["sku"]')->decodeResponseJson()['data'];
        $this->assertCount(3, $response);
    }

    public function testFilterProductDateOfSalesCreditIfDateGreaterThan(): void
    {
        $date = '2022-12-08';
        SalesOrder::factory()->createWithFullSalesCreditFullyReturned(3, $date);
        $response = $this->getJson('/api/products?limit=100&filters={"conjunction":"and","filterSet":[{"id":28875211,"column":"date_of_sales_credit","operator":">","dateOption":"exactDate","headerValue":"date_of_sales_credit","value":"2022-12-06"}]}&sortObjs=[]&page=1&table_specifications=0&total=0&archived=0&visible_only=0&included=["sku","image","name","brand","price","average_cost","default_supplier","supplier_pricing","inventory_stock_value"]')->decodeResponseJson()['data'];
        $count = SalesCreditLine::where('is_product', 1)->whereHas('salesCredit', function ($q) {
            $q->whereDate('credit_date', '>', '2022-12-06');
        })->count();
        $this->assertCount($count, $response);
        $response = $this->getJson('/api/products?limit=100&filters={"conjunction":"and","filterSet":[{"id":28875211,"column":"date_of_sales_credit","operator":">","dateOption":"exactDate","headerValue":"date_of_sales_credit","value":"2022-12-08"}]}&sortObjs=[]&page=1&table_specifications=0&total=0&archived=0&visible_only=0&included=["sku","image","name","brand","price","average_cost","default_supplier","supplier_pricing","inventory_stock_value"]')->decodeResponseJson()['data'];
        $this->assertCount(0, $response);
    }

    public function testFilterDateOfSalesCreditIfDateInRange(): void
    {
        $date = '2022-12-08';
        SalesOrder::factory()->createWithFullSalesCreditFullyReturned(3, $date);
        $response = $this->getJson('/api/products?limit=100&filters={"conjunction":"and","filterSet":[{"id":28875211,"column":"date_of_sales_credit","operator":"isWithin","dateOption":"range","headerValue":"date_of_sales_credit","value":{"from":"2022-12-07","to":"2022-12-31"}}]}&sortObjs=[]&page=1&table_specifications=0&total=0&archived=0&visible_only=0&included=["sku","image","name","brand","price","average_cost","default_supplier","supplier_pricing","inventory_stock_value"]')->decodeResponseJson()['data'];
        $count = SalesCreditLine::where('is_product', 1)->whereHas('salesCredit', function ($q) use ($date) {
            $q->whereDate('credit_date', $date);
        })->count();
        $this->assertCount($count, $response);
        $response = $this->getJson('/api/products?limit=100&filters={"conjunction":"and","filterSet":[{"id":28875211,"column":"date_of_sales_credit","operator":"isWithin","dateOption":"range","headerValue":"date_of_sales_credit","value":{"from":"2022-12-10","to":"2022-12-31"}}]}&sortObjs=[]&page=1&table_specifications=0&total=0&archived=0&visible_only=0&included=["sku","image","name","brand","price","average_cost","default_supplier","supplier_pricing","inventory_stock_value"]')->decodeResponseJson()['data'];
        $this->assertCount(0, $response);
    }

    /**
     * @throws BindingResolutionException
     */
    public function test_it_can_create_sales_credit_standard(): void
    {
        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines()
            ->create();
        $salesOrderLine = $salesOrder->salesOrderLines()->first();

        (new BulkInventoryManager())->bulkAllocateNegativeInventoryEvents(SalesOrderLine::all());

        $warehouse = Warehouse::first();

        $response = $this->postJson(route('sales-credits.store'), [
            'sales_order_id' => $salesOrder->id,
            'to_warehouse_id' => $warehouse->id,
            'sales_credit_lines' => [
                [
                    'sales_order_line_id' => $salesOrderLine->id,
                    'quantity' => 1,
                ]
            ]
        ]);
        $response->assertOk();

        $responesJson = $response->json()['data'];

        // Update sales credit
        $salesCreditId = $responesJson['id'];
        $updateData = [
            'sales_credit_number' => 'SC-99999',
            'to_warehouse_id' => $warehouse->id,
            'sales_credit_note' => 'Updated sales credit note',
            'credit_date' => now()->format('Y-m-d'),
            'store_id' => $salesOrder->store_id,
            'customer_id' => $salesOrder?->customer_id,
            'currency_id' => $salesOrder?->currency_id,
            'currency_code' => $salesOrder->currency?->code,
            'sales_credit_lines' => [
                [
                    'sales_order_line_id' => $salesOrderLine->id,
                    'quantity' => 2,
                ]
            ],
            'is_tax_included' => false,
            'is_product' => true,
        ];

        $updateResponse = $this->putJson(route('sales-credits.update', $salesCreditId), $updateData);
        $updateResponse->assertOk();

        $this->assertDatabaseHas('sales_credits', [
            'id' => $salesCreditId,
            'sales_credit_number' => 'SC-99999',
            'sales_credit_note' => 'Updated sales credit note',
        ]);
        $this->assertDatabaseHas('sales_credit_lines', [
            'sales_credit_id' => $salesCreditId,
            'quantity' => 2,
        ]);

    }

    public function test_it_can_create_sales_credit_when_sales_order_has_no_reservations(): void
    {
        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines()
            ->create();
        $salesOrderLine = $salesOrder->salesOrderLines()->first();

        $warehouse = Warehouse::first();

        $response = $this->postJson(route('sales-credits.store'), [
            'sales_order_id' => $salesOrder->id,
            'to_warehouse_id' => $warehouse->id,
            'sales_credit_lines' => [
                [
                    'sales_order_line_id' => $salesOrderLine->id,
                    'quantity' => 1,
                ]
            ]
        ]);
        $response->assertOk();
    }

}
