<?php

namespace Tests\Feature\Controllers;

use App\Models\Product;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Models\User;
use App\Models\Warehouse;
use Laravel\Sanctum\Sanctum;
use Modules\Ebay\Entities\EbayIntegrationInstance;
use Modules\Ebay\Entities\EbayLegacyProduct;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class ReportingControllerTest extends TestCase
{
    use FastRefreshDatabase;

    function setUp(): void
    {
        parent::setUp();

        Sanctum::actingAs(User::first());
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_get_sales_channel_coverage(): void
    {
        $defaultSupplier = Supplier::first();

        $ebayIntegration = EbayIntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->create([
                'name' => 'Ebay: abc'
            ]);

        $warehouse = Warehouse::first();

        $product = Product::factory()->create([
            'sku' => 'ABC',
            'name' => 'Product Name',
            'type' => Product::TYPE_STANDARD,
        ]);
        
        $product->setSupplierProducts([
            [
                'supplier_id' => $defaultSupplier->id,
                'is_default' => true,
            ],
        ]);

        $product->setInitialInventory($warehouse->id, 10, 5.00);

        $ebayProduct = EbayLegacyProduct::factory()->create([
            'integration_instance_id' => $ebayIntegration->id,
        ]);

        $endTime = now()->subDay()->toDateTimeString();

        $jsonObject = $ebayProduct->json_object;
        $jsonObject['ItemID'] = '123';
        $jsonObject['ListingDuration'] = 'Days_7';
        $jsonObject['ListingType'] = 'Chinese';
        $jsonObject['EndTimeUtc'] = $endTime;
        $jsonObject['SellingStatus']['ListingStatus'] = 'Active';
        $jsonObject['ListingDetails']['EndingReason'] = 'NotSold';
        $ebayProduct->json_object = $jsonObject;
        $ebayProduct->save();
        $ebayProduct->refresh();

        ProductListing::factory()->create([
            'sales_channel_id' => $ebayIntegration->salesChannel->id,
            'product_id' => $product->id,
            'document_type' => EbayLegacyProduct::class,
            'document_id' => $ebayProduct->id,
        ]);

        $response = $this->postJson(route('reporting.sales-channel-coverage'), [
            'byIntegration' => false,
            'showNumberOfCoverages' => true,
            'ids' => [$ebayIntegration->id],
        ])->assertOk();

        $response->assertExactJson([
            'data' => [
                [
                    'id' => $product->id,
                    'sku' => 'ABC',
                    'type' => Product::TYPE_STANDARD,
                    'stock' => 10,
                    'valuation' => 50.00,
                    'supplier' => [
                        'id' => $defaultSupplier->id,
                        'name' => $defaultSupplier->name,
                    ],
                    'coverage' => [
                        'Ebay: abc' => 1,
                        'Total' => 1,
                    ]
                ],
            ],
            'status' => 'success'
        ]);
    }
}
