<?php

namespace Tests\Feature\Controllers;

use App\Data\CreateDropshipShipmentData;
use App\Exceptions\PurchaseOrder\NotOpenPurchaseOrderException;
use App\Exceptions\PurchaseOrder\ReceivePurchaseOrderLineException;
use App\Helpers;
use App\Managers\DropshipManager;
use App\Models\AccountingTransaction;
use App\Models\AccountingTransactionLine;
use App\Models\Currency;
use App\Models\NominalCode;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\PurchaseOrderShipment;
use App\Models\PurchaseOrderShipmentLine;
use App\Models\PurchaseOrderShipmentReceipt;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\Setting;
use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Models\TaxRate;
use App\Models\User;
use App\Repositories\SettingRepository;
use App\Services\PurchaseOrder\ShipmentManager;
use Illuminate\Foundation\Testing\WithFaker;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class PurchaseOrderControllerTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    protected function setUp(): void
    {
        parent::setUp();
        Sanctum::actingAs(User::first());
    }

    public function test_it_can_filter_purchase_orders_by_sku(): void
    {
        Sanctum::actingAs(User::first());

        $product = Product::factory()->create(['sku' => '123456'])->first();
        PurchaseOrder::factory()->has(
            PurchaseOrderLine::factory()->count(1)->state([
                'product_id' => $product->id,
            ]),
        )->create();

        $response = $this->getJson(route('purchase-orders.index').'?'.Helpers::simpleFilter('sku',
            '123456'))->assertOk();

        $this->assertGreaterThanOrEqual(1, $response->json()['data']);
    }

    public function test_it_shows_a_purchase_order_with_total_units_only_for_products(): void
    {
        Sanctum::actingAs(User::first());

        $product = Product::factory()->create(['sku' => '123456'])->first();

        /** @var PurchaseOrder $purchaseOrder */
        $purchaseOrder = PurchaseOrder::factory()
            ->has(
                PurchaseOrderLine::factory()
                    ->state(['product_id' => $product->id]), 'purchaseOrderLines')
            ->has(
                PurchaseOrderLine::factory()
                    ->state(['product_id' => null]), 'purchaseOrderLines')
            ->create();

        $response = $this->getJson(
            route('purchase-orders.show', $purchaseOrder->id)
        );

        $response->assertOk();

        $response->assertJson([
            'data' => [
                'id' => $purchaseOrder->id,
                'summary_info' => [
                    'total_units' => $purchaseOrder->productLines()->sum('quantity'),
                ],
            ],
        ]);
    }

    public function test_it_should_return_3_items_from_purchase_order_view(): void
    {
        $total_items = 3;

        Sanctum::actingAs(User::first());

        $product = Product::factory()->create(['sku' => '123456'])->first();
        $purchase_order = PurchaseOrder::factory()->has(
            PurchaseOrderLine::factory()->count($total_items)->state([
                'product_id' => $product->id,
            ]),
        )->create();

        $response = $this->getJson(route('purchase-orders.show', $purchase_order->id));

        $response->assertOk();

        $this->assertCount($total_items, $response->json()['data']['items']);
    }

    public function test_it_can_calculate_purchase_order_totals(): void
    {
        Sanctum::actingAs(User::first());

        $product = Product::factory()->create(['sku' => '123456'])->first();

        $taxRate = TaxRate::factory()->create(['rate' => 10]);

        $supplier = Supplier::factory()->create();

        SupplierProduct::factory()->create([
            'product_id' => $product->id,
            'supplier_id' => $supplier->id,
        ]);

        $response = $this->postJson(route('purchase-orders.store'), [
            'purchase_order_date' => $this->faker()->date(),
            'supplier_id' => $supplier->id,
            'currency_id' => Currency::first()->id,
            'purchase_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 1,
                    'amount' => 100,
                ],
            ],
            'tax_rate_id' => $taxRate->id,
        ])->assertOk();

        $this->assertEquals(110, $response->json()['data']['total']);
    }

    public function test_user_cannot_delete_purchase_order_line_with_receipt(): void
    {
        Sanctum::actingAs(User::first());

        $purchaseOrder = PurchaseOrder::factory()->create();
        $purchaseOrderLine = PurchaseOrderLine::factory()->create([
            'purchase_order_id' => $purchaseOrder->id,
        ]);
        $purchaseOrderShipment = PurchaseOrderShipment::factory()->create([
            'purchase_order_id' => $purchaseOrder->id,
        ]);
        $purchaseOrderShipmentLine = PurchaseOrderShipmentLine::factory()->create([
            'purchase_order_shipment_id' => $purchaseOrderShipment->id,
            'purchase_order_line_id' => $purchaseOrderLine->id,
        ]);
        $purchaseOrderShipmentReceipt = PurchaseOrderShipmentReceipt::factory()->create([
            'purchase_order_shipment_id' => $purchaseOrderShipment->id,
        ]);
        PurchaseOrderShipmentReceiptLine::factory()->create([
            'purchase_order_shipment_receipt_id' => $purchaseOrderShipmentReceipt->id,
            'purchase_order_shipment_line_id' => $purchaseOrderShipmentLine->id,
        ]);

        $response = $this->patchJson(route('purchase-orders.update', $purchaseOrder->id), [
            'purchase_order_lines' => [
                [
                    'id' => $purchaseOrderLine->id,
                    'is_deleted' => true,
                    'quantity' => 1,
                    'amount' => 0,
                ],
            ],
        ]);

        $response->assertStatus(422);
    }

    public function test_it_can_duplicate_purchase_order(): void
    {
        $purchaseOrder = PurchaseOrder::factory()
            ->has(PurchaseOrderLine::factory()->count(5))
            ->create();
        $response = $this->postJson(route('purchase-orders.duplicate', $purchaseOrder->id));
        $data = $response->json()['data'];

        $response->assertOk();
        $this->assertDatabaseHas((new PurchaseOrder())->getTable(), ['id' => $data['id']]);
        $this->assertEquals(PurchaseOrder::STATUS_DRAFT, $data['order_status']);

        $getLineCounts = PurchaseOrder::with(PurchaseOrder::getLinesRelationName())->find($data['id']);
        $this->assertEquals(5, $getLineCounts->purchaseOrderLines()->count());
    }

    /**
     * @throws Throwable
     * @throws NotOpenPurchaseOrderException
     * @throws ReceivePurchaseOrderLineException
     */
    public function test_it_can_create_receiving_discrepancy_loss_from_purchase_order(): void
    {
        $nc = NominalCode::create([
            'name' => 'Lost In Transit Expense',
            'code' => '601',
            'type' => NominalCode::TYPE_EXPENSE,
        ]);

        app(SettingRepository::class)->set(Setting::KEY_NC_MAPPING_RECEIVING_DISCREPANCIES, $nc->id);

        $purchaseOrder = PurchaseOrder::factory()
            ->hasPurchaseOrderLines(1, [
                'quantity' => 5,
                'amount' => 100,
            ])
            ->create([
                'purchase_order_date' => now(),
            ]);

        $purchaseOrderLine = $purchaseOrder->purchaseOrderLines->first();
        (new ShipmentManager())->receiveShipment([
            'purchase_order_id' => $purchaseOrder->id,
            'received_at' => now(),
            'receipt_lines' => [
                [
                    'purchase_order_line_id' => $purchaseOrderLine->id,
                    'quantity' => 3,
                ],
            ],
        ]);

        $purchaseOrderShipment = $purchaseOrder->purchaseOrderShipments->first();

        $this->assertDatabaseHas(PurchaseOrderShipmentReceipt::class, [
            'purchase_order_shipment_id' => $purchaseOrderShipment->id,
        ]);

        $this->postJson(route('purchase-orders.create-receiving-discrepancy', $purchaseOrder->id))->assertOk();

        $this->assertDatabaseHas(AccountingTransaction::class, [
            'type' => AccountingTransaction::TYPE_RECEIVING_DISCREPANCY,
            'name' => $purchaseOrder->supplier->name,
            'reference' => $purchaseOrder->supplier->name . ': Receiving discrepancy for ' . $purchaseOrder->purchase_order_number,
            'link_id' => $purchaseOrder->id,
            'link_type' => PurchaseOrder::class,
        ]);

        $this->assertDatabaseHas(AccountingTransactionLine::class, [
            'type' => AccountingTransactionLine::TYPE_DEBIT,
            'nominal_code_id' => Helpers::setting(Setting::KEY_NC_MAPPING_RECEIVING_DISCREPANCIES),
            'description' => 'Receiving discrepancy for ' . $purchaseOrderLine->product->sku,
            'quantity' => 2,
            'amount' => 100,
            'link_id' => $purchaseOrderLine->id,
            'link_type' => PurchaseOrderLine::class,
        ]);

        $this->assertDatabaseHas(AccountingTransactionLine::class, [
            'type' => AccountingTransactionLine::TYPE_CREDIT,
            'nominal_code_id' => Helpers::setting(Setting::KEY_NC_MAPPING_ACCRUED_PURCHASES),
            'description' => 'Receiving discrepancy for ' . $purchaseOrderLine->product->sku,
            'quantity' => 2,
            'amount' => 100,
            'link_id' => $purchaseOrderLine->id,
            'link_type' => PurchaseOrderLine::class,
        ]);

        // Assert bad request due to already created discrepancy
        $this->postJson(route('purchase-orders.create-receiving-discrepancy', $purchaseOrder->id))
            ->assertBadRequest()
            ->assertJsonPath('message', 'Receiving discrepancy already exists for purchase order ' . $purchaseOrder->purchase_order_number);
    }

    /**
     * @throws Throwable
     * @throws NotOpenPurchaseOrderException
     * @throws ReceivePurchaseOrderLineException
     */
    public function test_it_can_create_receiving_discrepancy_gain_from_purchase_order(): void
    {
        $nc = NominalCode::create([
            'name' => 'Lost In Transit Expense',
            'code' => '601',
            'type' => NominalCode::TYPE_EXPENSE,
        ]);

        app(SettingRepository::class)->set(Setting::KEY_NC_MAPPING_RECEIVING_DISCREPANCIES, $nc->id);

        $purchaseOrder = PurchaseOrder::factory()
            ->hasPurchaseOrderLines(1, [
                'quantity' => 5,
                'amount' => 100,
            ])
            ->create([
                'purchase_order_date' => now(),
            ]);

        $purchaseOrderLine = $purchaseOrder->purchaseOrderLines->first();
        (new ShipmentManager())->receiveShipment([
            'purchase_order_id' => $purchaseOrder->id,
            'received_at' => now(),
            'receipt_lines' => [
                [
                    'purchase_order_line_id' => $purchaseOrderLine->id,
                    'quantity' => 7,
                ],
            ],
        ]);

        $purchaseOrderShipment = $purchaseOrder->purchaseOrderShipments->first();

        $this->assertDatabaseHas(PurchaseOrderShipmentReceipt::class, [
            'purchase_order_shipment_id' => $purchaseOrderShipment->id,
        ]);

        $this->postJson(route('purchase-orders.create-receiving-discrepancy', $purchaseOrder->id))->assertOk();

        $this->assertDatabaseHas(AccountingTransaction::class, [
            'type' => AccountingTransaction::TYPE_RECEIVING_DISCREPANCY,
            'name' => $purchaseOrder->supplier->name,
            'reference' => $purchaseOrder->supplier->name . ': Receiving discrepancy for ' . $purchaseOrder->purchase_order_number,
            'link_id' => $purchaseOrder->id,
            'link_type' => PurchaseOrder::class,
        ]);

        $this->assertDatabaseHas(AccountingTransactionLine::class, [
            'type' => AccountingTransactionLine::TYPE_DEBIT,
            'nominal_code_id' => Helpers::setting(Setting::KEY_NC_MAPPING_ACCRUED_PURCHASES),
            'description' => 'Receiving discrepancy for ' . $purchaseOrderLine->product->sku,
            'quantity' => 2,
            'amount' => 100,
            'link_id' => $purchaseOrderLine->id,
            'link_type' => PurchaseOrderLine::class,
        ]);

        $this->assertDatabaseHas(AccountingTransactionLine::class, [
            'type' => AccountingTransactionLine::TYPE_CREDIT,
            'nominal_code_id' => Helpers::setting(Setting::KEY_NC_MAPPING_RECEIVING_DISCREPANCIES),
            'description' => 'Receiving discrepancy for ' . $purchaseOrderLine->product->sku,
            'quantity' => 2,
            'amount' => 100,
            'link_id' => $purchaseOrderLine->id,
            'link_type' => PurchaseOrderLine::class,
        ]);

        // Assert bad request due to already created discrepancy
        $this->postJson(route('purchase-orders.create-receiving-discrepancy', $purchaseOrder->id))
            ->assertBadRequest()
            ->assertJsonPath('message', 'Receiving discrepancy already exists for purchase order ' . $purchaseOrder->purchase_order_number);
    }

    public function test_it_can_delete_shipped_dropship_po(): void
    {
//        app(DropshipManager::class)->ship(CreateDropshipShipmentData::from([
//            'tracking_number' => '123456',
//        ]), $purchaseOrder);
    }
}
