<?php

namespace Tests\Feature\Controllers;

use App\Models\Product;
use App\Models\ProductInventory;
use App\Models\User;
use App\Models\Warehouse;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class ProductControllerTest extends TestCase
{
    use FastRefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        Sanctum::actingAs(User::first());
    }

    public function test_it_can_sort_product_inventory(): void
    {
        $warehouse = Warehouse::factory(1, [
            'name' => 'Test Warehouse',
            'type' => Warehouse::TYPE_DIRECT,
        ])->create()->first();
        $product1 = Product::factory()->create();
        $product2 = Product::factory()->create();
        $product3 = Product::factory()->create();
        ProductInventory::factory()->create([
            'product_id' => $product1->id,
            'warehouse_id' => $warehouse->id,
            'inventory_total' => 10,
        ]);
        ProductInventory::factory()->create([
            'product_id' => $product2->id,
            'warehouse_id' => $warehouse->id,
            'inventory_total' => 33,
        ]);
        ProductInventory::factory()->create([
            'product_id' => $product3->id,
            'warehouse_id' => $warehouse->id,
            'inventory_total' => 2,
        ]);

        $response = $this->getJson(route('products.index').'?sortObjs='.
            json_encode([
                [
                    'column' => 'inventory.'.$warehouse->name,
                    'ascending' => false,
                ],
            ]).'&included=["sku","name","inventory"]'
        )->assertOk();

        $data = $response->json('data');
        $this->assertEquals($product2->id, $data[0]['id']);
        $this->assertEquals($product1->id, $data[1]['id']);
        $this->assertEquals($product3->id, $data[2]['id']);
    }

    public function test_it_can_sort_product_inventory_available(): void
    {
        $warehouse = Warehouse::factory(1, [
            'name' => 'Test Warehouse',
            'type' => Warehouse::TYPE_DIRECT,
        ])->create()->first();
        $product1 = Product::factory()->create();
        $product2 = Product::factory()->create();
        $product3 = Product::factory()->create();
        ProductInventory::factory()->create([
            'product_id' => $product1->id,
            'warehouse_id' => $warehouse->id,
            'inventory_available' => 10,
        ]);
        ProductInventory::factory()->create([
            'product_id' => $product2->id,
            'warehouse_id' => $warehouse->id,
            'inventory_available' => 33,
        ]);
        ProductInventory::factory()->create([
            'product_id' => $product3->id,
            'warehouse_id' => $warehouse->id,
            'inventory_available' => 2,
        ]);

        $response = $this->getJson(route('products.index').'?sortObjs='.
            json_encode([
                [
                    'column' => 'inventory_available_warehouses.'.$warehouse->name,
                    'ascending' => false,
                ],
            ]).'&included=["sku","name","inventory_available_warehouses"]'
        )->assertOk();

        $data = $response->json('data');
        $this->assertEquals($product2->id, $data[0]['id']);
        $this->assertEquals($product1->id, $data[1]['id']);
        $this->assertEquals($product3->id, $data[2]['id']);
    }

    public function test_it_can_sort_products_by_match_score(): void
    {
        Product::factory()->create(['sku' => 'GSL392+400-939', 'updated_at' => now()->addDays(1)]);
        Product::factory()->create(['sku' => 'GSL392', 'name' => 'fuel pump']);

        $response = $this->getJson(route('products.index').'?query=GSL392')->assertOk();

        $data = $response->json('data');

        $this->assertEquals('GSL392', $data[0]['sku']);
        $this->assertEquals('fuel pump', $data[0]['name']);
    }

    public function test_it_can_get_product_by_sku_with_slash(): void
    {
        Product::factory()->create(['sku' => 'GSL392/sku', 'name' => 'fuel pump']);

        $response = $this->getJson(route('products.by-sku', ['sku' => 'GSL392/sku']))->assertOk();

        $data = $response->json('data');

        $this->assertEquals('GSL392/sku', $data['sku']);
    }
}
