<?php

namespace Tests\Feature\Controllers;

use App\Data\BlemishedProductData;
use App\Data\CreateInventoryAdjustmentFromBlemishedProductData;
use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\ProductBlemished;
use App\Models\ProductInventory;
use App\Models\Setting;
use App\Models\User;
use App\Models\Warehouse;
use Exception;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class BlemishedProductControllerTest extends TestCase
{
    use FastRefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        Sanctum::actingAs(User::first());
    }

    public function test_it_can_create_blemished_product_with_inventory_adjustment(): void
    {
        $product = Product::factory()->create([
            'type' => Product::TYPE_STANDARD,
        ]);

        FifoLayer::factory()->create([
            'product_id' => $product->id,
        ]);

        $warehouse = Warehouse::first();

        $inventoryAdjustmentData = CreateInventoryAdjustmentFromBlemishedProductData::from([
            'adjustment_date' => today()->format('Y-m-d'),
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 1,
            'notes' => 'Create Blemished Product',
        ]);

        $response = $this->post(route('products.blemished.store'), BlemishedProductData::from([
            'derived_from_product_id' => $product->id,
            'blemished_sku' => 'BlemishedSKU123',
            'condition' => 'Blemished',
            'reference' => 'Blemish123',
            'adjust_inventory' => true,
            'adjust_inventory_for_original_product' => false,
            'quantity' => 1,
            'adjustment_data' => $inventoryAdjustmentData,
        ])->toArray())->assertOk();

        $blemishedProductId = $response->json()['data']['id'];

        $this->assertDatabaseHas((new Product())->getTable(), [
            'sku' => 'BlemishedSKU123',
            'type' => 'blemished',
        ]);

        $this->assertDatabaseHas((new ProductBlemished())->getTable(), [
            'product_id' => $blemishedProductId,
            'derived_from_product_id' => $product->id,
            'condition' => 'Blemished',
            'reference' => 'Blemish123',
        ]);

        $this->assertDatabaseHas((new InventoryAdjustment())->getTable(), [
            'product_id' => $blemishedProductId,
        ]);
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'product_id' => $blemishedProductId,
        ]);
        $this->assertDatabaseHas((new ProductInventory())->getTable(), [
            'product_id' => $blemishedProductId,
        ]);
    }

    /**
     * @throws Exception
     */
    public function test_sku_pattern_adheres_to_iso8601_date_standard()
    {
        $product = Product::factory()->create(['sku' => 'PROD-01']);
        $today   = now();

        $pattern = 'RETURN-{{sku}}-{{date:YYYYMMDD}}';
        $this->putJson(route('settings.update-blemished-sku-pattern'),
            [
                Setting::KEY_BLEMISHED_SKU_PATTERN => $pattern,
                Setting::KEY_USE_BLEMISHED_SKU_PATTERN => true,
            ]
        )->assertOk();
        $this->assertEquals($pattern, Setting::getBlemishedSkuPattern());

        $generatedSku = $product->generateSkuPattern();
        $this->assertEquals('RETURN-PROD-01-'.$today->format('Ymd'), $generatedSku);

    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_sku_pattern_with_suffix()
    {
        //set the blemished sku pattern
        $pattern = 'RETURN-{{sku}}-{{date:YYYYMMDD}}';
        $this->putJson(route('settings.update-blemished-sku-pattern'),
            [
                Setting::KEY_BLEMISHED_SKU_PATTERN => $pattern,
                Setting::KEY_USE_BLEMISHED_SKU_PATTERN => true,
            ]
        )->assertOk();
        $this->assertEquals($pattern, Setting::getBlemishedSkuPattern());

        $product1 = Product::factory()->create(['sku' => 'PROD-01']);

        FifoLayer::factory()->create([
            'product_id' => $product1->id,
        ]);
        $warehouse = Warehouse::first();

        $product1->setInitialInventory($warehouse->id, 500, 20);
        $inventoryAdjustmentData = CreateInventoryAdjustmentFromBlemishedProductData::from([
            'adjustment_date' => today()->format('Y-m-d'),
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 1,
            'notes' => 'Create Blemished Product',
        ]);

        $generatedSku1 = $product1->generateSkuPattern();
        $this->assertEquals('RETURN-PROD-01-'.now()->format('Ymd'), $generatedSku1);

        //Create blemished product with generated SKU Pattern
        $this->post(route('products.blemished.store'), BlemishedProductData::from([
            'derived_from_product_id' => $product1->id,
            'blemished_sku' => $generatedSku1,
            'condition' => 'Blemished',
            'reference' => 'Blemish123',
            'adjust_inventory' => false,
            'quantity' => 1,
            'adjustment_data' => $inventoryAdjustmentData,
        ])->toArray())->assertOk();

        $generatedSku2 = $product1->generateSkuPattern();
        $this->assertEquals('RETURN-PROD-01-'.now()->format('Ymd').'.1', $generatedSku2);

        //Create blemished product with generated SKU Pattern
        $this->post(route('products.blemished.store'), BlemishedProductData::from([
            'derived_from_product_id' => $product1->id,
            'blemished_sku' => $generatedSku2,
            'condition' => 'Blemished',
            'reference' => 'Blemish123',
            'adjust_inventory' => false,
            'quantity' => 1,
            'adjustment_data' => $inventoryAdjustmentData,
        ])->toArray())->assertOk();

        // Create a third product with the same initial SKU to check incrementing suffix
        $generatedSku3 = $product1->generateSkuPattern();
        $this->assertEquals('RETURN-PROD-01-'.now()->format('Ymd').'.2', $generatedSku3);
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_create_blemished_product_with_original_product_inventory_adjustment(): void
    {
        $warehouse = Warehouse::first();

        $product = Product::factory()->create([
            'type' => Product::TYPE_STANDARD,
        ]);
        $product->setInitialInventory($warehouse->id, 10);

        FifoLayer::factory()->create([
            'product_id' => $product->id,
        ]);

        $inventoryAdjustmentData = CreateInventoryAdjustmentFromBlemishedProductData::from([
            'adjustment_date' => today()->format('Y-m-d'),
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 1,
            'notes' => 'Create Blemished Product',
        ]);

        $response = $this->post(route('products.blemished.store'), BlemishedProductData::from([
            'derived_from_product_id' => $product->id,
            'blemished_sku' => 'BlemishedSKU123',
            'condition' => 'Blemished',
            'reference' => 'Blemish123',
            'adjust_inventory' => true,
            'adjust_inventory_for_original_product' => true,
            'quantity' => 1,
            'adjustment_data' => $inventoryAdjustmentData,
        ])->toArray())->assertOk();

        $blemishedProductId = $response->json()['data']['id'];

        $this->assertDatabaseHas((new Product())->getTable(), [
            'sku' => 'BlemishedSKU123',
            'type' => 'blemished',
        ]);

        $this->assertDatabaseHas((new ProductBlemished())->getTable(), [
            'product_id' => $blemishedProductId,
            'derived_from_product_id' => $product->id,
            'condition' => 'Blemished',
            'reference' => 'Blemish123',
        ]);

        $this->assertDatabaseHas((new InventoryAdjustment())->getTable(), [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => -1,
        ]);
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => -1,
        ]);
        $this->assertDatabaseHas((new ProductInventory())->getTable(), [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'inventory_available' => 9,
        ]);
    }

    public function test_it_can_create_blemished_product_without_parent_inventory_adjustment_and_blemished_product_inventory_adjustment(): void
    {
        $warehouse = Warehouse::first();

        $product = Product::factory()->create([
            'type' => Product::TYPE_STANDARD,
        ]);

        $inventoryAdjustmentData = CreateInventoryAdjustmentFromBlemishedProductData::from([
            'adjustment_date' => today()->format('Y-m-d'),
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 1,
            'notes' => 'Create Blemished Product',
        ]);

        $response = $this->post(route('products.blemished.store'), (array) BlemishedProductData::from([
            'derived_from_product_id' => $product->id,
            'blemished_sku' => 'BlemishedSKU123',
            'condition' => 'Blemished',
            'reference' => 'Blemish123',
            'quantity' => 1,
            'adjustment_data' => $inventoryAdjustmentData,
        ])->toArray())->assertOk();

        $blemishedProductId = $response->json()['data']['id'];

        $this->assertDatabaseHas((new Product())->getTable(), [
            'sku' => 'BlemishedSKU123',
            'type' => 'blemished',
        ]);

        $this->assertDatabaseHas((new ProductBlemished())->getTable(), [
            'product_id' => $blemishedProductId,
            'derived_from_product_id' => $product->id,
            'condition' => 'Blemished',
            'reference' => 'Blemish123',
        ]);

        // Check Original Product Entries
        $this->assertDatabaseMissing((new InventoryAdjustment())->getTable(), [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
        ]);
        $this->assertDatabaseMissing((new InventoryMovement())->getTable(), [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
        ]);
        $this->assertDatabaseMissing((new ProductInventory())->getTable(), [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
        ]);

        //Check Blemished Product Entries
        $this->assertDatabaseMissing((new InventoryAdjustment())->getTable(), [
            'product_id' => $blemishedProductId,
            'warehouse_id' => $warehouse->id,
        ]);
        $this->assertDatabaseMissing((new InventoryMovement())->getTable(), [
            'product_id' => $blemishedProductId,
            'warehouse_id' => $warehouse->id,
        ]);
        $this->assertDatabaseMissing((new ProductInventory())->getTable(), [
            'product_id' => $blemishedProductId,
            'warehouse_id' => $warehouse->id,
        ]);
    }
}
