<?php

namespace Tests\Feature\Controllers;

use App\Jobs\ReleaseBackorderQueuesJob;
use App\Models\BackorderQueue;
use App\Models\BackorderQueueCoverage;
use App\Models\BackorderQueueRelease;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\User;
use App\Models\Warehouse;
use App\Services\InventoryManagement\BulkInventoryManager;
use App\Services\Product\ReleaseBackorderQueues;
use App\Services\PurchaseOrder\ShipmentManager;
use Illuminate\Contracts\Container\BindingResolutionException;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class BackorderQueueControllerTest extends TestCase
{
    use FastRefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();

        Sanctum::actingAs(User::first());
    }

    /**
     * @throws BindingResolutionException
     */
    public function test_it_can_reallocate_inventory_to_another_sales_order_and_back_again()
    {
        /*
         * Set up sales order
         * Set up purchase order
         * Receive purchase order
         * Set up another sales order
         * Reallocate
         * Reallocate back
         */

        $product = Product::factory()->create();
        $warehouse = Warehouse::first();

        $salesOrder1 = SalesOrder::factory()
            ->hasSalesOrderLines(1, [
                'product_id' => $product->id,
                'quantity' => 10,
                'warehouse_id' => $warehouse->id,
            ])->create();
        $salesOrder1Line = $salesOrder1->salesOrderLines->first();

        $salesOrder2 = SalesOrder::factory()
            ->hasSalesOrderLines(1, [
                'product_id' => $product->id,
                'quantity' => 5,
                'warehouse_id' => $warehouse->id,
            ])->create();
        $salesOrder2Line = $salesOrder2->salesOrderLines->first();

        (new BulkInventoryManager())->bulkAllocateNegativeInventoryEvents(collect([$salesOrder1Line, $salesOrder2Line]));

        $backorderQueue1 = BackorderQueue::where('sales_order_line_id', $salesOrder1Line->id)->first();
        $backorderQueue2 = BackorderQueue::where('sales_order_line_id', $salesOrder2Line->id)->first();

        $this->assertDatabaseHas(BackorderQueue::class, [
            'sales_order_line_id' => $salesOrder1Line->id,
            'backordered_quantity' => 10,
        ]);

        $this->assertDatabaseHas(BackorderQueue::class, [
            'sales_order_line_id' => $salesOrder2Line->id,
            'backordered_quantity' => 5,
        ]);

        $this->assertDatabaseEmpty(FifoLayer::class);

        $purchaseOrder = PurchaseOrder::factory()
            ->hasPurchaseOrderLines(1, ['product_id' => $product->id, 'quantity' => 10])
            ->create([
                'destination_warehouse_id' => $warehouse->id,
            ]);
        $purchaseOrderLine = $purchaseOrder->purchaseOrderLines->first();

        $this->assertDatabaseHas(BackorderQueueCoverage::class, [
            'backorder_queue_id' => $backorderQueue1->id,
            'purchase_order_line_id' => $purchaseOrderLine->id,
            'covered_quantity' => 10,
            'released_quantity' => 0,
        ]);

        (new ShipmentManager())->receiveShipment([
            'purchase_order_id' => $purchaseOrder->id,
            'received_at' => now()->toIso8601ZuluString(),
            'receipt_lines' => $purchaseOrder->purchaseOrderLines->map(function ($line) {
                return [
                    'quantity' => 10,
                    'purchase_order_line_id' => $line->id,
                ];
            })->toArray(),
        ]);

        $this->assertDatabaseHas(BackorderQueueCoverage::class, [
            'backorder_queue_id' => $backorderQueue1->id,
            'purchase_order_line_id' => $purchaseOrderLine->id,
            'covered_quantity' => 10,
            'released_quantity' => 10,
        ]);

        $fifoLayer = FifoLayer::first();

        $this->assertDatabaseHas(FifoLayer::class, [
            'product_id' => $product->id,
            'original_quantity' => 10,
        ]);

        $this->assertDatabaseHas(BackorderQueue::class, [
            'sales_order_line_id' => $salesOrder1Line->id,
            'backordered_quantity' => 10,
            'released_quantity' => 10,
        ]);

        $this->assertDatabaseHas(BackorderQueue::class, [
            'sales_order_line_id' => $salesOrder2Line->id,
            'backordered_quantity' => 5,
            'released_quantity' => 0,
        ]);

        $this->assertDatabaseHas(BackorderQueueRelease::class, [
            'backorder_queue_id' => $backorderQueue1->id,
            'released_quantity' => 10,
        ]);
        $this->assertDatabaseCount(BackorderQueueRelease::class, 1);
        $backorderQueueRelease1 = BackorderQueueRelease::first();

        $this->postJson(route('backorder-queues.reallocate-release', $backorderQueueRelease1->id), [
            'sales_order_id' => $salesOrder2->id,
            'quantity' => 2,
        ])->assertOk();

        $backorderQueueRelease2 = BackorderQueueRelease::where('backorder_queue_id', $backorderQueue2->id)->first();

        // Coverage was reduced when reallocated
        $this->assertDatabaseHas(BackorderQueueCoverage::class, [
            'backorder_queue_id' => $backorderQueue1->id,
            'purchase_order_line_id' => $purchaseOrderLine->id,
            'covered_quantity' => 8,
            'released_quantity' => 8,
        ]);

        $this->assertDatabaseHas(BackorderQueue::class, [
            'sales_order_line_id' => $salesOrder1Line->id,
            'backordered_quantity' => 10,
            'released_quantity' => 8,
        ]);

        $this->assertDatabaseHas(BackorderQueueRelease::class, [
            'backorder_queue_id' => $backorderQueue1->first()->id,
            'released_quantity' => 8,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => -2,
            'inventory_status' => 'active',
            'link_type' => SalesOrderLine::class,
            'link_id' => $salesOrder1Line->id,
            'layer_type' => BackorderQueue::class,
            'layer_id' => $backorderQueue1->id,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => 2,
            'inventory_status' => 'reserved',
            'link_type' => SalesOrderLine::class,
            'link_id' => $salesOrder1Line->id,
            'layer_type' => BackorderQueue::class,
            'layer_id' => $backorderQueue1->id,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => -8,
            'inventory_status' => 'active',
            'link_type' => SalesOrderLine::class,
            'link_id' => $salesOrder1Line->id,
            'layer_type' => FifoLayer::class,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => 8,
            'inventory_status' => 'reserved',
            'link_type' => SalesOrderLine::class,
            'link_id' => $salesOrder1Line->id,
            'layer_type' => FifoLayer::class,
        ]);

        // Coverage was added when reallocated
        $this->assertDatabaseHas(BackorderQueueCoverage::class, [
            'backorder_queue_id' => $backorderQueue2->id,
            'purchase_order_line_id' => $purchaseOrderLine->id,
            'covered_quantity' => 2,
            'released_quantity' => 2,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => -2,
            'inventory_status' => 'active',
            'link_type' => SalesOrderLine::class,
            'link_id' => $salesOrder2Line->id,
            'layer_type' => FifoLayer::class,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => 2,
            'inventory_status' => 'reserved',
            'link_type' => SalesOrderLine::class,
            'link_id' => $salesOrder2Line->id,
            'layer_type' => FifoLayer::class,
        ]);

        $this->assertDatabaseHas(BackorderQueue::class, [
            'sales_order_line_id' => $salesOrder2Line->id,
            'backordered_quantity' => 5,
            'released_quantity' => 2,
        ]);

        $this->assertDatabaseHas(BackorderQueueRelease::class, [
            'id' => $backorderQueueRelease2->id,
            'backorder_queue_id' => $backorderQueue2->id,
            'released_quantity' => 2,
        ]);

        $this->postJson(route('backorder-queues.reallocate-release', $backorderQueueRelease2->id), [
            'sales_order_id' => $salesOrder1->id,
            'quantity' => 2,
        ])->assertOk();

        $this->assertDatabaseHas(BackorderQueueCoverage::class, [
            'backorder_queue_id' => $backorderQueue1->id,
            'purchase_order_line_id' => $purchaseOrderLine->id,
            'covered_quantity' => 10,
            'released_quantity' => 10,
        ]);

        $this->assertDatabaseMissing(BackorderQueueRelease::class, [
            'id' => $backorderQueueRelease2->id,
        ]);

        $backorderQueueRelease3 = BackorderQueueRelease::where('backorder_queue_id', $backorderQueue1->id)->where('released_quantity', 2)->first();

        $this->assertDatabaseHas(BackorderQueueRelease::class, [
            'id' => $backorderQueueRelease3->id,
            'backorder_queue_id' => $backorderQueue1->id,
            'released_quantity' => 2,
        ]);

        $this->postJson(route('backorder-queues.reallocate-release', $backorderQueueRelease3->id), [
            'sales_order_id' => $salesOrder2->id,
            'quantity' => 1,
        ])->assertOk();

        $this->assertDatabaseHas(BackorderQueueCoverage::class, [
            'backorder_queue_id' => $backorderQueue1->id,
            'purchase_order_line_id' => $purchaseOrderLine->id,
            'covered_quantity' => 9,
            'released_quantity' => 9,
        ]);

        $this->assertDatabaseHas(BackorderQueueCoverage::class, [
            'backorder_queue_id' => $backorderQueue2->id,
            'purchase_order_line_id' => $purchaseOrderLine->id,
            'covered_quantity' => 1,
            'released_quantity' => 1,
        ]);

        $this->assertDatabaseHas(BackorderQueueRelease::class, [
            'backorder_queue_id' => $backorderQueue1->id,
            'released_quantity' => 8,
        ]);

        $this->assertDatabaseHas(BackorderQueueRelease::class, [
            'backorder_queue_id' => $backorderQueue1->id,
            'released_quantity' => 1,
        ]);

        $this->assertDatabaseHas(BackorderQueueRelease::class, [
            'backorder_queue_id' => $backorderQueue2->id,
            'released_quantity' => 1,
        ]);
    }
}
