<?php

namespace Tests\Feature;

use App\Data\SalesOrderLineData;
use App\Data\UpdateSalesOrderData;
use App\Data\UpdateSalesOrderPayloadData;
use App\Exceptions\InventoryAssemblyComponentException;
use App\Exceptions\InventoryAssemblyStockException;
use App\Exceptions\InventoryAssemblyTypeException;
use App\Exceptions\InventoryMovementTypeException;
use App\Exceptions\KitWithMovementsCantBeChangedException;
use App\Exceptions\ProductBundleException;
use App\Exporters\Jasper\PackingSlipTransformer;
use App\Http\Requests\StoreInventoryAdjustment;
use App\Jobs\GenerateCacheProductListingQuantityJob;
use App\Models\Customer;
use App\Models\InventoryAssembly;
use App\Models\InventoryAssemblyLine;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\ProductComponent;
use App\Models\ProductListing;
use App\Models\ProductListingInventoryLocation;
use App\Models\ProductPricingTier;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Models\User;
use App\Models\Warehouse;
use App\Repositories\ProductRepository;
use App\Services\InventoryAssembly\InventoryAssemblyManager;
use App\Services\SalesOrder\SalesOrderManager;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\Sequence;
use Illuminate\Foundation\Testing\WithFaker;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Queue;
use Tests\TestCase;
use Throwable;

class BundlesAndKitsTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    public function test_bundles_split_into_components(): void
    {
        Queue::fake();

        /** @var Product $bundle */
        $bundle = Product::factory()->create([
            'sku' => 'BUNDLE-1',
            'name' => 'Bundle 1',
            'type' => Product::TYPE_BUNDLE,
        ]);

        $defaultPricingTier = ProductPricingTier::default();

        $bundle->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 100,
            ],
        ]);

        /** @var Product $component1 */
        $component1 = Product::factory()->create([
            'sku' => 'COMPONENT-1',
            'name' => 'Component 1',
            'type' => Product::TYPE_STANDARD,
        ]);

        $component1->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 30,
            ],
        ]);

        /** @var Product $component2 */
        $component2 = Product::factory()->create([
            'sku' => 'COMPONENT-2',
            'name' => 'Component 2',
            'type' => Product::TYPE_STANDARD,
        ]);

        $component2->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 20,
            ],
        ]);

        $bundle->setBundleComponents([
            [
                'id' => $component1->id,
                'quantity' => 2,
            ],
            [
                'id' => $component2->id,
                'quantity' => 3,
            ],
        ]);

        $component1Proration = $component1->getBundlePriceProration($bundle);
        $component2Proration = $component2->getBundlePriceProration($bundle);

        /** @var User $user */
        $user = User::query()->first();

        Sanctum::actingAs($user);

        $response = $this->postJson('/api/sales-orders', [
            'order_status' => 'open',
            'currency_code' => 'USD',
            'order_date' => Carbon::now(),
            'customer_id' => Customer::factory()->create()->id,
            'sales_order_lines' => [
                [
                    'product_id' => $bundle->id,
                    'description' => $bundle->name,
                    'quantity' => 1,
                    'amount' => 150, // Purposely set to different price than bundle product
                ],
            ],
        ])->assertOk();

        $itemInfo = $response->json('data.item_info');

        $this->assertEquals($bundle->id, $itemInfo[0]['bundle']['id']);
        $this->assertEquals($bundle->id, $itemInfo[1]['bundle']['id']);

        $this->assertDatabaseMissing((new SalesOrderLine())->getTable(), [
            'product_id' => $bundle->id,
        ]);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'product_id' => $component1->id,
            'quantity' => 2,
            'amount' => 150 * $component1Proration,
        ]);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'product_id' => $component2->id,
            'quantity' => 3,
            'amount' => 150 * $component2Proration,
        ]);
    }

    /**
     * @throws ProductBundleException
     * @throws KitWithMovementsCantBeChangedException
     */
    public function test_kits_do_not_split_into_components(): void
    {
        Queue::fake();

        /** @var Product $kit */
        $kit = Product::factory()->create([
            'sku' => 'KIT-1',
            'name' => 'Kit 1',
            'type' => Product::TYPE_KIT,
        ]);

        /** @var Product $component1 */
        $component1 = Product::factory()->create([
            'sku' => 'COMPONENT-1',
            'name' => 'Component 1',
            'type' => Product::TYPE_STANDARD,
        ]);

        /** @var Product $component2 */
        $component2 = Product::factory()->create([
            'sku' => 'COMPONENT-2',
            'name' => 'Component 2',
            'type' => Product::TYPE_STANDARD,
        ]);

        $kit->setKitComponents([
            [
                'id' => $component1->id,
                'quantity' => 2,
            ],
            [
                'id' => $component2->id,
                'quantity' => 3,
            ],
        ]);

        $this->assertDatabaseHas((new Product())->getTable(), [
            'sku' => 'KIT-1',
            'name' => 'Kit 1',
            'type' => Product::TYPE_KIT,
        ]);

        $this->assertEquals(
            2,
            $kit->components->count()
        );

        /** @var User $user */
        $user = User::query()->first();

        Sanctum::actingAs($user);

        $this->postJson('/api/sales-orders', [
            'order_status' => 'open',
            'currency_code' => 'USD',
            'order_date' => Carbon::now(),
            'customer_id' => Customer::factory()->create()->id,
            'sales_order_lines' => [
                [
                    'product_id' => $kit->id,
                    'description' => $kit->name,
                    'quantity' => 1,
                    'amount' => 100,
                ],
            ],
        ])->assertOk();

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'product_id' => $kit->id,
            'quantity' => 1,
        ]);

        $this->assertDatabaseCount((new SalesOrderLine())->getTable(), 1);

        try {
            $kit->setKitComponents([
                [
                    'id' => $component1->id,
                    'quantity' => 2,
                ],
            ]);
            $this->fail(); // Exception not thrown
        } catch (KitWithMovementsCantBeChangedException) {
            $this->assertTrue(true);
        }

        $this->assertDatabaseCount((new ProductComponent())->getTable(), 2);
    }

    public function test_can_replace_bundle_with_kit_if_kit_in_stock_and_bundle_is_not(): void
    {
        Queue::fake();
        /**
         * Setup
         */

        /** @var Product $bundle */
        $bundle = Product::factory()->create([
            'sku' => 'BUNDLE-1',
            'name' => 'Bundle 1',
            'type' => Product::TYPE_BUNDLE,
        ]);

        $defaultPricingTier = ProductPricingTier::default();

        $bundle->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 100,
            ],
        ]);

        /** @var Product $component1 */
        $component1 = Product::factory()->create([
            'sku' => 'COMPONENT-1',
            'name' => 'Component 1',
            'type' => Product::TYPE_STANDARD,
        ]);

        $component1->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 30,
            ],
        ]);

        /** @var Product $component2 */
        $component2 = Product::factory()->create([
            'sku' => 'COMPONENT-2',
            'name' => 'Component 2',
            'type' => Product::TYPE_STANDARD,
        ]);

        $component2->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 20,
            ],
        ]);

        $bundle->setBundleComponents([
            [
                'id' => $component1->id,
                'quantity' => 2,
            ],
            [
                'id' => $component2->id,
                'quantity' => 3,
            ],
        ]);

        $kitData = [
            'sku' => 'KIT-1',
            'name' => 'Kit 1',
        ];

        /**
         * Test you can create a kit from a bundle
         */

        /** @var Product $kit */
        $kit = app(ProductRepository::class)->createKitFromBundle($bundle, $kitData);

        $this->assertDatabaseHas((new Product())->getTable(), [
            'id' => $bundle->kit->id, // this is how you access the kit from the bundle
            'sku' => 'KIT-1',
            'name' => 'Kit 1',
            'type' => Product::TYPE_KIT,
        ]);

        $this->assertEquals(
            $bundle->price,
            $kit->price,
        );

        /*
         * Test if kit is in stock but bundle potential stock is not, that it replaces the product in the sales order
         * with the kit instead of the bundle.  The logic is that associating a kit with a bundle is a way to represent
         * that they are physically the same product except one is assembled into a kit and one has its components kept
         * separate.  If you sell a bundle with components left separate and those components are out of stock, but you
         * have an assembled version in stock, it makes sense to use that instead.
         */

        /** @var User $user */
        $user = User::query()->first();

        Sanctum::actingAs($user);

        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::query()->first();

        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => Carbon::now(),
            'product_id' => $kit->id,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 5.00,
            'quantity' => 1,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ])->assertSuccessful();

        $this->postJson('/api/sales-orders', [
            'order_status' => 'open',
            'currency_code' => 'USD',
            'order_date' => Carbon::now(),
            'customer_id' => Customer::factory()->create()->id,
            'sales_order_lines' => [
                [
                    'product_id' => $bundle->id,
                    'description' => $bundle->name,
                    'quantity' => 1,
                    'amount' => 100,
                ],
            ],
        ])->assertOk();

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'product_id' => $kit->id,
            'description' => $bundle->name, // The description for the sales order line shouldn't change in the swap
            'quantity' => 1,
            'amount' => 100,
        ]);

        $this->assertDatabaseMissing((new SalesOrderLine())->getTable(), [
            'product_id' => $bundle->id,
        ]);

        /*
         * Now that the kit has been sold, there is no more in stock, so the next sales order should not swap.
         */

        $this->postJson('/api/sales-orders', [
            'order_status' => 'open',
            'currency_code' => 'USD',
            'order_date' => Carbon::now(),
            'customer_id' => Customer::factory()->create()->id,
            'sales_order_lines' => [
                [
                    'product_id' => $bundle->id,
                    'description' => $bundle->name,
                    'quantity' => 1,
                    'amount' => 100,
                ],
            ],
        ])->assertOk();

        $this->assertDatabaseMissing((new SalesOrderLine())->getTable(), [
            'product_id' => $bundle->id,
        ]);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'product_id' => $component1->id,
            'quantity' => 2,
        ]);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'product_id' => $component2->id,
            'quantity' => 3,
        ]);
    }

    /**
     * @throws InventoryAssemblyComponentException
     * @throws InventoryAssemblyTypeException
     * @throws ProductBundleException
     */
    public function test_assembly_and_disassembly_logic(): void
    {
        Queue::fake();

        /**
         * Setup
         */

        /** @var Product $bundle */
        $bundle = Product::factory()->create([
            'sku' => 'BUNDLE-1',
            'name' => 'Bundle 1',
            'type' => Product::TYPE_BUNDLE,
        ]);

        $defaultPricingTier = ProductPricingTier::default();

        $bundle->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 100,
            ],
        ]);

        /** @var Product $component1 */
        $component1 = Product::factory()->create([
            'sku' => 'COMPONENT-1',
            'name' => 'Component 1',
            'type' => Product::TYPE_STANDARD,
            'unit_cost' => 14.00,
        ]);

        $component1->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 30,
            ],
        ]);

        /** @var Product $component2 */
        $component2 = Product::factory()->create([
            'sku' => 'COMPONENT-2',
            'name' => 'Component 2',
            'type' => Product::TYPE_STANDARD,
            'unit_cost' => 9.00,
        ]);

        $component2->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 20,
            ],
        ]);

        $bundle->setBundleComponents([
            [
                'id' => $component1->id,
                'quantity' => 2,
            ],
            [
                'id' => $component2->id,
                'quantity' => 3,
            ],
        ]);

        $kitData = [
            'sku' => 'KIT-1',
            'name' => 'Kit 1',
        ];

        /** @var Product $kit */
        $kit = app(ProductRepository::class)->createKitFromBundle($bundle, $kitData);

        /** @var User $user */
        $user = User::query()->first();

        Sanctum::actingAs($user);

        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::query()->first();

        /**
         * Test that you can assemble a kit from its components
         *
         * Note current logic for assembly is located in InventoryAssemblyController
         *
         * This should be moved appropriately.  This test reflects the better structure for accessing the logic
         */
        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => Carbon::now(),
            'product_id' => $component1->id,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 5.00, // purposefully different from component unit cost
            'quantity' => 10,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ])->assertSuccessful();

        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => Carbon::now(),
            'product_id' => $component2->id,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 5.00, // purposefully different from component unit cost
            'quantity' => 10,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ])->assertSuccessful();
        $this->assertEquals(3, $bundle->getPotentialInventory());
        $this->assertEquals(3, $kit->getPotentialInventory());

        $assemblyDate = Carbon::now();

        $components = $kit->components;
        $inventoryAssemblyManager = (new InventoryAssemblyManager($kit, $components));

        /*
         * Uses 4 of component 1 and 6 of component 2
         */
        $inventoryAssemblyManager->assemble(2, $assemblyDate); // Assembly date should be optional, if not passed, use current date

        $this->assertDatabaseHas((new Product())->getTable(), [
            'id' => $kit->id,
            //            'unit_cost' => (2 * 14.00) + (3 * 9.00),
        ]);

        $this->assertDatabaseHas((new InventoryAssembly())->getTable(), [
            'warehouse_id' => $warehouse->id,
            'action' => InventoryAssembly::INVENTORY_ASSEMBLY_ACTION_ASSEMBLE,
            'action_date' => $assemblyDate,
        ]);

        $this->assertDatabaseHas((new InventoryAssemblyLine())->getTable(), [
            'product_id' => $kit->id,
            'quantity' => 2,
            'product_type' => InventoryAssemblyLine::INVENTORY_ASSEMBLY_LINE_PRODUCT_TYPE_KIT,
        ]);
        $this->assertDatabaseHas((new InventoryAssemblyLine())->getTable(), [
            'product_id' => $component1->id,
            'quantity' => -4,
            'product_type' => InventoryAssemblyLine::INVENTORY_ASSEMBLY_LINE_PRODUCT_TYPE_COMPONENT,
        ]);

        $this->assertDatabaseHas((new InventoryAssemblyLine())->getTable(), [
            'product_id' => $component2->id,
            'quantity' => -6,
            'product_type' => InventoryAssemblyLine::INVENTORY_ASSEMBLY_LINE_PRODUCT_TYPE_COMPONENT,
        ]);

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'product_id' => $kit->id,
            'quantity' => 2,
            'warehouse_id' => $warehouse->id,
            'type' => InventoryMovement::TYPE_ASSEMBLY,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'inventory_movement_date' => $assemblyDate,
        ]);

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'product_id' => $component1->id,
            'quantity' => -4,
            'warehouse_id' => $warehouse->id,
            'type' => InventoryMovement::TYPE_ASSEMBLY,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'inventory_movement_date' => $assemblyDate,
        ]);

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'product_id' => $component2->id,
            'quantity' => -6,
            'warehouse_id' => $warehouse->id,
            'type' => InventoryMovement::TYPE_ASSEMBLY,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'inventory_movement_date' => $assemblyDate,
        ]);

        /*
         * Test you get an exception if you try to assemble more than you have in stock
         */

        try {
            $inventoryAssemblyManager->assemble(10, $assemblyDate);
        } catch (InventoryAssemblyStockException $e) {
            $this->assertEquals('Not enough component inventory to assemble 10 kits of KIT-1.', $e->getMessage());
        }

        /*
         * Test that you can disassemble a kit into its components
         */

        $inventoryAssemblyManager->disassemble(1, $assemblyDate);

        $this->assertDatabaseHas((new InventoryAssembly())->getTable(), [
            'warehouse_id' => $warehouse->id,
            'action' => InventoryAssembly::INVENTORY_ASSEMBLY_ACTION_DISASSEMBLE,
            'action_date' => $assemblyDate,
        ]);

        $this->assertDatabaseHas((new InventoryAssemblyLine())->getTable(), [
            'product_id' => $kit->id,
            'quantity' => -1,
            'product_type' => InventoryAssemblyLine::INVENTORY_ASSEMBLY_LINE_PRODUCT_TYPE_KIT,
        ]);

        $this->assertDatabaseHas((new InventoryAssemblyLine())->getTable(), [
            'product_id' => $component1->id,
            'quantity' => 2,
            'product_type' => InventoryAssemblyLine::INVENTORY_ASSEMBLY_LINE_PRODUCT_TYPE_COMPONENT,
        ]);

        $this->assertDatabaseHas((new InventoryAssemblyLine())->getTable(), [
            'product_id' => $component2->id,
            'quantity' => 3,
            'product_type' => InventoryAssemblyLine::INVENTORY_ASSEMBLY_LINE_PRODUCT_TYPE_COMPONENT,
        ]);

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'product_id' => $kit->id,
            'quantity' => -1,
            'warehouse_id' => $warehouse->id,
            'type' => InventoryMovement::TYPE_ASSEMBLY,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'inventory_movement_date' => $assemblyDate,
        ]);

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'product_id' => $component1->id,
            'quantity' => 2,
            'warehouse_id' => $warehouse->id,
            'type' => InventoryMovement::TYPE_ASSEMBLY,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'inventory_movement_date' => $assemblyDate,
        ]);

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'product_id' => $component2->id,
            'quantity' => 3,
            'warehouse_id' => $warehouse->id,
            'type' => InventoryMovement::TYPE_ASSEMBLY,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'inventory_movement_date' => $assemblyDate,
        ]);

        /*
         * Test that you cannot assemble or disassemble products of type other than kit
         */

        try {
            $inventoryAssemblyManager->assemble(1, $assemblyDate, $component1);
        } catch (InventoryAssemblyTypeException $e) {
            $this->assertEquals('Only products of type kit can be assembled/disassembled.', $e->getMessage());
        }

        try {
            $inventoryAssemblyManager->disassemble(1, $assemblyDate, $component1);
        } catch (InventoryAssemblyTypeException $e) {
            $this->assertEquals('Only products of type kit can be assembled/disassembled.', $e->getMessage());
        }

        try {
            $inventoryAssemblyManager->assemble(1, $assemblyDate, $bundle);
        } catch (InventoryAssemblyTypeException $e) {
            $this->assertEquals('Only products of type kit can be assembled/disassembled.', $e->getMessage());
        }

        try {
            $inventoryAssemblyManager->disassemble(1, $assemblyDate, $bundle);
        } catch (InventoryAssemblyTypeException $e) {
            $this->assertEquals('Only products of type kit can be assembled/disassembled.', $e->getMessage());
        }

        $matrix = Product::factory()->create([
            'sku' => 'MATRIX-1',
            'type' => Product::TYPE_MATRIX,
        ]);

        try {
            $inventoryAssemblyManager->assemble(1, $assemblyDate, $matrix);
        } catch (InventoryAssemblyTypeException $e) {
            $this->assertEquals('Only products of type kit can be assembled/disassembled.', $e->getMessage());
        }

        try {
            $inventoryAssemblyManager->disassemble(1, $assemblyDate, $matrix);
        } catch (InventoryAssemblyTypeException $e) {
            $this->assertEquals('Only products of type kit can be assembled/disassembled.', $e->getMessage());
        }

        $kitWithoutComponents = Product::factory()->create([
            'sku' => 'KIT-2',
            'type' => Product::TYPE_KIT,
        ]);

        try {
            $inventoryAssemblyManager->assemble(1, $assemblyDate, $kitWithoutComponents);
        } catch (InventoryAssemblyComponentException $e) {
            $this->assertEquals('KIT-2 does not have any components.', $e->getMessage());
        }
    }

    public function test_only_standard_products_be_purchased(): void
    {
        Queue::fake();
        $supplier = Supplier::factory()->withWarehouse()->create();
        /** @var Product $standard */
        $standard = Product::factory()->has(
            SupplierProduct::factory(1, [
                'supplier_id' => $supplier->id,
            ])
        )->create([
            'sku' => 'STANDARD-1',
            'type' => Product::TYPE_STANDARD,
        ]);
        /** @var Product $bundle */
        $bundle = Product::factory()->has(
            SupplierProduct::factory(1, [
                'supplier_id' => $supplier->id,
            ])
        )->create([
            'sku' => 'BUNDLE-1',
            'type' => Product::TYPE_BUNDLE,
        ]);

        /** @var Product $kit */
        $kit = Product::factory()->has(
            SupplierProduct::factory(1, [
                'supplier_id' => $supplier->id,
            ])
        )->create([
            'sku' => 'KIT-1',
            'type' => Product::TYPE_KIT,
        ]);

        /** @var Product $matrix */
        $matrix = Product::factory()->has(
            SupplierProduct::factory(1, [
                'supplier_id' => $supplier->id,
            ])
        )->create([
            'sku' => 'MATRIX-1',
            'type' => Product::TYPE_MATRIX,
        ]);

        /** @var User $user */
        $user = User::query()->first();

        Sanctum::actingAs($user);
        $response = $this->postJson('/api/purchase-orders', [
            'supplier_id' => $supplier->id,
            'store_id' => 1,
            'currency_code' => 'USD',
            'destination_warehouse_id' => 1,
            'purchase_order_date' => Carbon::now(),
            'purchase_order_lines' => [
                [
                    'product_id' => $standard->id,
                    'quantity' => 1,
                    'amount' => '5',
                    'is_product' => true,
                ],
            ],
        ]);
        $response->assertOk();

        $purchaseOrderId = $response->json('data.id');

        /**
         * Test that you cannot only purchase a standard product
         */
        $response = $this->postJson('/api/purchase-orders', [
            'supplier_id' => $supplier->id,
            'store_id' => 1,
            'currency_code' => 'USD',
            'destination_warehouse_id' => 1,
            'purchase_order_date' => Carbon::now(),
            'purchase_order_lines' => [
                [
                    'product_id' => $bundle->id,
                    'quantity' => 1,
                    'amount' => '5',
                    'is_product' => true,
                ],
            ],
        ]);
        $responseData = $response->json('errors');
        $errorMessages = $responseData['purchase_order_lines.0.product_id'];
        $this->assertEquals('Only standard products can be purchased.', $errorMessages[0]);

        $response = $this->postJson('/api/purchase-orders', [
            'supplier_id' => $supplier->id,
            'store_id' => 1,
            'currency_code' => 'USD',
            'destination_warehouse_id' => 1,
            'purchase_order_date' => Carbon::now(),
            'purchase_order_lines' => [
                [
                    'product_id' => $kit->id,
                    'quantity' => 1,
                    'amount' => '5',
                    'is_product' => true,
                ],
            ],
        ]);
        $responseData = $response->json('errors');
        $errorMessages = $responseData['purchase_order_lines.0.product_id'];
        $this->assertEquals('Only standard products can be purchased.', $errorMessages[0]);
        // Access the validator instance

        $response = $this->putJson('/api/purchase-orders/'.$purchaseOrderId, [
            'purchase_order_lines' => [
                [
                    'product_id' => $bundle->id,
                    'quantity' => 1,
                    'amount' => '5',
                    'is_product' => true,
                ],
            ],
        ]);

        $responseData = $response->json('errors');
        $errorMessages = $responseData['purchase_order_lines.0.product_id'];
        $this->assertEquals('Only standard products can be purchased.', $errorMessages[0]);
    }

    public function test_bundles_and_matrix_cant_have_inventory_movements(): void
    {
        Queue::fake();

        /** @var Product $bundle */
        $bundle = Product::factory()->create([
            'sku' => 'BUNDLE-1',
            'type' => Product::TYPE_BUNDLE,
        ]);

        /** @var Product $matrix */
        $matrix = Product::factory()->create([
            'sku' => 'MATRIX-1',
            'type' => Product::TYPE_MATRIX,
        ]);

        try {
            InventoryMovement::query()->create([
                'product_id' => $bundle->id,
                'warehouse_id' => 1,
                'quantity' => 1,
                'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
                'type' => InventoryMovement::TYPE_ADJUSTMENT,
                'created_at' => Carbon::now(),
            ]);
        } catch (InventoryMovementTypeException $e) {
            $this->assertEquals('Only standard/blemished/kit products can have inventory movements.', $e->getMessage());
        }

        try {
            InventoryMovement::query()->create([
                'product_id' => $matrix->id,
                'warehouse_id' => 1,
                'quantity' => 1,
                'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
                'type' => InventoryMovement::TYPE_ADJUSTMENT,
                'created_at' => Carbon::now(),
            ]);
        } catch (InventoryMovementTypeException $e) {
            $this->assertEquals('Only standard/blemished/kit products can have inventory movements.', $e->getMessage());
        }
    }

    public function test_packing_slip_payload_returns_the_bundle_product_info(): void
    {
        Queue::fake();

        /** @var Product $bundle */
        $bundle = Product::factory()->create([
            'sku' => 'BUNDLE-1',
            'name' => 'Bundle 1',
            'type' => Product::TYPE_BUNDLE,
        ]);

        $defaultPricingTier = ProductPricingTier::default();

        $bundle->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 100,
            ],
        ]);

        /** @var Product $component1 */
        $component1 = Product::factory()->create([
            'sku' => 'COMPONENT-1',
            'name' => 'Component 1',
            'type' => Product::TYPE_STANDARD,
        ]);

        $component1->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 30,
            ],
        ]);

        /** @var Product $component2 */
        $component2 = Product::factory()->create([
            'sku' => 'COMPONENT-2',
            'name' => 'Component 2',
            'type' => Product::TYPE_STANDARD,
        ]);

        $component2->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 20,
            ],
        ]);

        $bundle->setBundleComponents([
            [
                'id' => $component1->id,
                'quantity' => 2,
            ],
            [
                'id' => $component2->id,
                'quantity' => 3,
            ],
        ]);

        /** @var User $user */
        $user = User::query()->first();

        Sanctum::actingAs($user);

        $response = $this->postJson('/api/sales-orders', [
            'order_status' => 'open',
            'currency_code' => 'USD',
            'order_date' => Carbon::now(),
            'customer_id' => Customer::factory()->create()->id,
            'sales_order_lines' => [
                [
                    'product_id' => $bundle->id,
                    'description' => $bundle->name,
                    'quantity' => 1,
                    'amount' => 150, // Purposely set to different price than bundle product
                ],
            ],
        ])->assertOk();

        $salesOrderLine = SalesOrderLine::first();
        $responseArray = PackingSlipTransformer::salesOrderLine($salesOrderLine);
        $this->assertEquals($bundle->id, $responseArray[0]['bundle']['id']);
    }

    /**
     * @throws \Exception
     */
    public function test_sales_order_lines_can_split_bundle_later(): void
    {
        Queue::fake();

        /** @var User $user */
        $user = User::query()->first();

        Sanctum::actingAs($user);

        $defaultPricingTier = ProductPricingTier::default();

        $bundle = Product::factory()->create([
            'sku' => 'bundle',
            'name' => 'Bundle 1',
            'type' => Product::TYPE_STANDARD,
        ]);

        $this->postJson('/api/sales-orders', [
            'order_status' => 'open',
            'currency_code' => 'USD',
            'order_date' => Carbon::now(),
            'customer_id' => Customer::factory()->create()->id,
            'sales_order_lines' => [
                [
                    'product_id' => $bundle->id,
                    'description' => $bundle->name,
                    'quantity' => 1,
                    'amount' => 100,
                ],
            ],
        ])->assertOk();

        $bundle->update(['type' => Product::TYPE_BUNDLE]);

        /** @var Product $component1 */
        $component1 = Product::factory()->create([
            'sku' => 'COMPONENT-1',
            'name' => 'Component 1',
            'type' => Product::TYPE_STANDARD,
        ]);

        $component1->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 30,
            ],
        ]);

        /** @var Product $component2 */
        $component2 = Product::factory()->create([
            'sku' => 'COMPONENT-2',
            'name' => 'Component 2',
            'type' => Product::TYPE_STANDARD,
        ]);

        $component2->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 20,
            ],
        ]);

        $bundle->setBundleComponents([
            [
                'id' => $component1->id,
                'quantity' => 1,
            ],
            [
                'id' => $component2->id,
                'quantity' => 2,
            ],
        ]);

        $salesOrder = SalesOrder::first();
        $salesOrderLine = $salesOrder->salesOrderLines()->first();

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_type' => SalesOrderLine::class,
            'product_id' => $bundle->id,
        ]);

        app(SalesOrderManager::class)->fixBundlesForSalesOrder($salesOrder);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'product_id' => $component1->id,
            'quantity' => 1,
            'bundle_id' => $bundle->id,
        ]);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'product_id' => $component2->id,
            'quantity' => 2,
            'bundle_id' => $bundle->id,
        ]);

        $this->assertDatabaseMissing((new InventoryMovement())->getTable(), [
            'link_type' => SalesOrderLine::class,
            'product_id' => $bundle->id,
        ]);

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_type' => SalesOrderLine::class,
            'product_id' => $component1->id,
        ]);

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_type' => SalesOrderLine::class,
            'product_id' => $component2->id,
        ]);
    }

    public function test_product_listing_quantity_for_bundle_products_updated_when_component_inventory_updated(): void
    {
        $this->markTestSkipped('This test is not working due to commenting out seed in CacheProductListingQuantity.  @Ahmad needs fixing');
        Queue::fake();

        /** @var Product $bundle */
        $bundle = Product::factory()->create([
            'sku' => 'BUNDLE-1',
            'name' => 'Bundle 1',
            'type' => Product::TYPE_BUNDLE,
        ]);

        $defaultPricingTier = ProductPricingTier::default();

        $bundle->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 100,
            ],
        ]);

        /** @var Product $component1 */
        $component1 = Product::factory()->create([
            'sku' => 'COMPONENT-1',
            'name' => 'Component 1',
            'type' => Product::TYPE_STANDARD,
        ]);

        $component1->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 30,
            ],
        ]);

        /** @var Product $component2 */
        $component2 = Product::factory()->create([
            'sku' => 'COMPONENT-2',
            'name' => 'Component 2',
            'type' => Product::TYPE_STANDARD,
        ]);

        $component2->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => 20,
            ],
        ]);

        $bundle->setBundleComponents([
            [
                'id' => $component1->id,
                'quantity' => 2,
            ],
            [
                'id' => $component2->id,
                'quantity' => 3,
            ],
        ]);
        $productListing = ProductListing::factory()->create([
            'product_id' => $bundle->id,
        ]);
        $component1Listing = ProductListing::factory()->create([
            'product_id' => $component1->id,
            'sales_channel_id' => $productListing->sales_channel_id,

        ]);
        $component2Listing = ProductListing::factory()->create([
            'product_id' => $component2->id,
            'sales_channel_id' => $productListing->sales_channel_id,

        ]);

        $this->assertEquals(0, $productListing->quantity);

        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::query()->first();

        /** @var User $user */
        $user = User::query()->first();

        Sanctum::actingAs($user);
        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => Carbon::now(),
            'product_id' => $component1->id,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 5.00,
            'quantity' => 10,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ])->assertSuccessful();

        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => Carbon::now(),
            'product_id' => $component2->id,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 5.00,
            'quantity' => 10,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ])->assertSuccessful();
        dispatch_sync(new GenerateCacheProductListingQuantityJob($productListing->salesChannel->integrationInstance));

        $this->assertDatabaseHas((new ProductListingInventoryLocation())->getTable(), [
            'product_listing_id' => $productListing->id,
            'quantity' => 3,
        ]);
        $this->assertDatabaseHas((new ProductListingInventoryLocation())->getTable(), [
            'product_listing_id' => $component1Listing->id,
            'quantity' => 10,
        ]);
        $this->assertDatabaseHas((new ProductListingInventoryLocation())->getTable(), [
            'product_listing_id' => $component2Listing->id,
            'quantity' => 10,
        ]);
    }

    /**
     * See SKU-6414
     * @throws ProductBundleException
     * @throws Throwable
     */
    public function test_sales_order_lines_dont_get_deleted_on_order_update_for_changed_bundle_components(): void
    {
        #$this->markTestSkipped('this logic needs further consideration as it is causing other bugs');
        /** @var Product $bundle */
        $bundle = Product::factory()->create([
            'sku' => 'BUNDLE-1',
            'name' => 'Bundle 1',
            'type' => Product::TYPE_BUNDLE,
        ]);

        /** @var Product $component1 */
        $component1 = Product::factory()->create([
            'sku' => 'COMPONENT-1',
            'name' => 'Component 1',
            'type' => Product::TYPE_STANDARD,
        ]);

        /** @var Product $component2 */
        $component2 = Product::factory()->create([
            'sku' => 'COMPONENT-2',
            'name' => 'Component 2',
            'type' => Product::TYPE_STANDARD,
        ]);

        /** @var Product $component2 */
        $component3 = Product::factory()->create([
            'sku' => 'COMPONENT-3',
            'name' => 'Component 3',
            'type' => Product::TYPE_STANDARD,
        ]);

        $bundle->setBundleComponents([
            [
                'id' => $component1->id,
                'quantity' => 1,
            ],
            [
                'id' => $component2->id,
                'quantity' => 1,
            ],
        ]);

        $salesOrder = SalesOrder::factory()
            ->has(
                SalesOrderLine::factory()
                    ->count(2)
                    ->state(new Sequence(
                        [
                            'product_id' => $component1->id,
                            'bundle_id' => $bundle->id,
                            'sales_channel_line_id' => 'salesChannelLineId',
                            'product_listing_id' => null,
                            'description' => $component1->name,
                            'bundle_quantity_cache' => 1,
                            'quantity' => 1,
                        ],
                        [
                            'product_id' => $component2->id,
                            'bundle_id' => $bundle->id,
                            'sales_channel_line_id' => 'salesChannelLineId',
                            'product_listing_id' => null,
                            'description' => $component2->name,
                            'bundle_quantity_cache' => 1,
                            'quantity' => 1,
                        ],
                    ))
            )
            ->create([
                'order_status' => SalesOrder::STATUS_OPEN,
            ]);

        $salesOrderLines = $salesOrder->salesOrderLines;
        $salesOrderLine1 = $salesOrderLines[0];
        $salesOrderLine2 = $salesOrderLines[1];

        $manager = app(SalesOrderManager::class);

        $updateOrderData = UpdateSalesOrderData::from([
            'salesOrder' => $salesOrder,
            'payload' => UpdateSalesOrderPayloadData::from([
                'salesOrderLines' => SalesOrderLineData::collection([
                    SalesOrderLineData::from([
                        'product_id' => $bundle->id,
                        'sales_channel_line_id' => 'salesChannelLineId',
                        'product_listing_id' => null,
                        'quantity' => 2,
                        'amount' => 5.00
                    ]),
                ]),
            ])
        ]);

        $manager->updateOrder($updateOrderData);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'id' => $salesOrderLine1->id,
            'product_id' => $component1->id,
        ]);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'id' => $salesOrderLine2->id,
            'product_id' => $component2->id,
        ]);

        $bundle->setBundleComponents([
            [
                'id' => $component3->id,
                'quantity' => 1,
            ],
            [
                'id' => $component2->id,
                'quantity' => 1,
            ],
        ]);

        $manager->updateOrder($updateOrderData);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'id' => $salesOrderLine1->id,
            'product_id' => $component1->id,
        ]);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'id' => $salesOrderLine2->id,
            'product_id' => $component2->id,
        ]);
    }

    /**
     * @throws ProductBundleException
     * @throws KitWithMovementsCantBeChangedException
     */
    public function test_duplicate_components_cant_be_added_to_bundle_or_kit(): void
    {
        /** @var Product $bundle */
        $bundle = Product::factory()->create([
            'sku' => 'BUNDLE-1',
            'name' => 'Bundle 1',
            'type' => Product::TYPE_BUNDLE,
        ]);

        /** @var Product $component1 */
        $component1 = Product::factory()->create([
            'sku' => 'COMPONENT-1',
            'name' => 'Component 1',
            'type' => Product::TYPE_STANDARD,
        ]);

        $bundle->setBundleComponents([
            [
                'id' => $component1->id,
                'quantity' => 1,
            ],
            [
                'id' => $component1->id,
                'quantity' => 1,
            ],
        ]);

        $this->assertDatabaseCount((new ProductComponent())->getTable(), 1);

        $kit = Product::factory()->create([
            'sku' => 'KIT-1',
            'name' => 'Kit 1',
            'type' => Product::TYPE_KIT,
        ]);

        $kit->setKitComponents([
            [
                'id' => $component1->id,
                'quantity' => 1,
            ],
            [
                'id' => $component1->id,
                'quantity' => 1,
            ],
        ]);

        $this->assertDatabaseCount((new ProductComponent())->getTable(), 2);
    }

    /*
     * TODO: For amazon FBA products, test that they cannot be associated to bundles (merchant fulfilled products can be associated to bundles), delegated to SKU-5763
     * TODO: Test that when a setting is enabled (true by default), the bundle sku is sent to the shipping provider, when the setting is disabled, the component skus are sent to the shipping provider, delegated to SKU-5764
     * TODO: Test that components forecasting considers sales of kits that contain them.  Delegated to SKU-5765
     *
     */
}
