<?php

namespace Tests\Feature;

use App\Exceptions\NegativeInventoryFulfilledSalesOrderLinesException;
use App\Exceptions\PurchaseOrder\NotOpenPurchaseOrderException;
use App\Exceptions\PurchaseOrder\ReceivePurchaseOrderLineException;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\BackorderQueue;
use App\Models\BackorderQueueRelease;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\PurchaseOrderShipment;
use App\Models\PurchaseOrderShipmentLine;
use App\Models\PurchaseOrderShipmentReceipt;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineLayer;
use App\Models\Supplier;
use App\Models\Warehouse;
use App\Repositories\PurchaseOrderRepository;
use App\Services\InventoryManagement\BulkInventoryManager;
use App\Services\PurchaseOrder\ShipmentManager;
use Carbon\Carbon;
use Database\Factories\FactoryDataRecycler;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Queue;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\Concerns\UseSimpleSkuFactories;
use Tests\TestCase;
use Throwable;

class BulkInventoryManagerTest extends TestCase
{
    use FastRefreshDatabase;
    use UseSimpleSkuFactories;
    use WithFaker;

    /**
     * @throws NegativeInventoryFulfilledSalesOrderLinesException
     * @throws Throwable
     */
    private function deletePurchaseOrder(PurchaseOrder $purchaseOrder): void
    {
        app(PurchaseOrderRepository::class)->bulkDelete(collect()->push($purchaseOrder));
    }

    /**
     * @throws BindingResolutionException
     * @throws NotOpenPurchaseOrderException
     * @throws ReceivePurchaseOrderLineException
     */
    public function test_it_can_bulk_allocate_negative_events()
    {

        $this->setUpProduct();
        $this->setUpWarehouse();
        $this->receivePurchaseOrder($this->setUpPurchaseOrders(5)->first());
        $this->setUpSalesOrders(5);
        (new BulkInventoryManager())->bulkAllocateNegativeInventoryEvents(SalesOrderLine::all());

        $this->assertTrue(
            InventoryMovement::query()
                ->where('link_type', SalesOrderLine::class)
                ->where('type', InventoryMovement::TYPE_SALE)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                ->whereNotNull('inventory_movement_date')
                ->exists()
        );

        $this->assertDatabaseHas((new SalesOrderLineLayer())->getTable(), [
            'sales_order_line_id' => SalesOrderLine::query()->first()->id,
            'layer_type' => FifoLayer::class,
        ]);
    }

    /**
     * @throws NotOpenPurchaseOrderException
     * @throws ReceivePurchaseOrderLineException
     * @throws BindingResolutionException
     */
    public function test_it_can_bulk_allocate_negative_event_with_backorders()
    {
        $this->setUpProduct();
        $this->setUpWarehouse();
        // Only 2 in stock
        $this->receivePurchaseOrder($this->setUpPurchaseOrders(2)->first());
        // 2 sales orders, totaling quantity 11 needed
        $this->setUpSalesOrders([1, 10]);
        (new BulkInventoryManager())->bulkAllocateNegativeInventoryEvents(SalesOrderLine::all());

        $this->assertCount(
            2,
            InventoryMovement::query()
                ->where('link_type', SalesOrderLine::class)
                ->where('type', InventoryMovement::TYPE_SALE)
                ->where('quantity', -1)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                ->where('layer_type', FifoLayer::class)
                ->get()
        );

        $this->assertCount(
            1,
            InventoryMovement::query()
                ->where('link_type', SalesOrderLine::class)
                ->where('type', InventoryMovement::TYPE_SALE)
                ->where('quantity', -9)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                ->where('layer_type', BackorderQueue::class)
                ->get()
        );

        $this->assertDatabaseHas((new FifoLayer())->getTable(), [
            'fulfilled_quantity' => 2,
        ]);
    }

    /**
     * @throws NegativeInventoryFulfilledSalesOrderLinesException
     * @throws NotOpenPurchaseOrderException
     * @throws ReceivePurchaseOrderLineException
     * @throws Throwable
     */
    public function test_it_can_bulk_delete_purchase_orders_receipt_lines()
    {
        $bulkInventoryManger = app(BulkInventoryManager::class);

        $this->setUpProduct();
        $this->setUpWarehouse();
        $purchaseOrders = $this->setUpPurchaseOrders([
            5, // to delete
            10, // unallocated
        ]);
        $po1 = $purchaseOrders[0];
        $po2 = $purchaseOrders[1];

        $receipt1 = $this->receivePurchaseOrder($po1);
        $positiveInventoryEventPo1 = $receipt1->purchaseOrderShipmentReceiptLines->first();
        $fifoLayerPo1 = $positiveInventoryEventPo1->fifoLayers->first();

        $receipt2 = $this->receivePurchaseOrder($po2);
        $positiveInventoryEventPo2 = $receipt2->purchaseOrderShipmentReceiptLines->first();
        $fifoLayerPo2 = $positiveInventoryEventPo2->fifoLayers->first();

        $salesOrder = $this->setUpSalesOrders(5)->first(); // no backorder
        $salesOrderLine = $salesOrder->salesOrderLines->first();

        $bulkInventoryManger->bulkAllocateNegativeInventoryEvents(SalesOrderLine::all());

        $salesOrderReservationMovementsFromPo1 = $salesOrderLine->inventoryMovements()
            ->where('layer_id', $positiveInventoryEventPo1->getFifoLayer())
            ->where('layer_type', FifoLayer::class)
            ->get();

        $bulkInventoryManger->bulkDeletePositiveInventoryEvents(collect()->add($positiveInventoryEventPo1));

        $this->assertDatabaseMissing((new FifoLayer())->getTable(), [
            'link_id' => $positiveInventoryEventPo1->id,
            'link_type' => PurchaseOrderShipmentReceipt::class,
        ]);

        $this->assertDatabaseMissing((new InventoryMovement())->getTable(), [
            'link_id' => $positiveInventoryEventPo1->id,
            'link_type' => PurchaseOrderShipmentReceipt::class,
            'layer_id' => $fifoLayerPo1->id,
            'layer_type' => FifoLayer::class,
        ]);

        // Original movements belonging to deleted layer are deleted
        $this->assertEquals(
            0,
            InventoryMovement::whereIn('id', $salesOrderReservationMovementsFromPo1->pluck('id'))->count()
        );
        // Sales order line layers are deleted with the FifoLayer->delete() method

        $this->assertDatabaseMissing((new SalesOrderLineLayer())->getTable(), [
            'layer_id' => $fifoLayerPo1->id,
            'layer_type' => FifoLayer::class,
        ]);

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_id' => $positiveInventoryEventPo2->id,
            'link_type' => PurchaseOrderShipmentReceiptLine::class,
            'layer_type' => FifoLayer::class,
            'layer_id' => $fifoLayerPo2->id,
            'quantity' => $positiveInventoryEventPo2->quantity,
        ]);

        $this->assertDatabaseHas((new SalesOrderLineLayer())->getTable(), [
            'sales_order_line_id' => $salesOrder->salesOrderLines->first()->id,
            'layer_id' => $fifoLayerPo2->id,
            'layer_type' => FifoLayer::class,
            'quantity' => $positiveInventoryEventPo1->quantity,
        ]);

        // Usages
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_id' => $fifoLayerPo2->id,
            'layer_type' => FifoLayer::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'quantity' => $salesOrderLine->quantity,
        ]);
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_id' => $fifoLayerPo2->id,
            'layer_type' => FifoLayer::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'quantity' => -$salesOrderLine->quantity,
        ]);

        // Now delete PO 2, no unallocated

        $salesOrderReservationMovementsFromPo2 = $salesOrderLine->inventoryMovements()
            ->where('layer_id', $positiveInventoryEventPo2->getFifoLayer())
            ->where('layer_type', FifoLayer::class)
            ->get();

        $bulkInventoryManger->bulkDeletePositiveInventoryEvents(collect()->add($positiveInventoryEventPo2));

        $this->assertDatabaseMissing((new FifoLayer())->getTable(), [
            'link_id' => $positiveInventoryEventPo2->id,
            'link_type' => PurchaseOrderShipmentReceipt::class,
        ]);

        $this->assertDatabaseMissing((new InventoryMovement())->getTable(), [
            'link_id' => $positiveInventoryEventPo2->id,
            'link_type' => PurchaseOrderShipmentReceipt::class,
            'layer_id' => $fifoLayerPo2->id,
            'layer_type' => FifoLayer::class,
        ]);

        // Original movements belonging to deleted layer are deleted
        $this->assertEquals(
            0,
            InventoryMovement::whereIn('id', $salesOrderReservationMovementsFromPo2->pluck('id'))->count()
        );
        // Sales order line layers are deleted with the FifoLayer->delete() method

        $this->assertDatabaseMissing((new SalesOrderLineLayer())->getTable(), [
            'layer_id' => $fifoLayerPo2->id,
            'layer_type' => FifoLayer::class,
        ]);

        // Usages (Backorder)
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_type' => BackorderQueue::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'quantity' => $salesOrderLine->quantity,
        ]);
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_type' => BackorderQueue::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'quantity' => -$salesOrderLine->quantity,
        ]);

        $this->assertDatabaseHas((new BackorderQueue())->getTable(), [
            'sales_order_line_id' => $salesOrderLine->id,
            'backordered_quantity' => $salesOrderLine->quantity,
        ]);

        // now left with just the sales order, backordered

        // Release scenarios

        $receipt1 = $this->receivePurchaseOrder($po1, 2);
        $receipt2 = $this->receivePurchaseOrder($po1, 3);
        $positiveInventoryEventPo1 = $receipt1->purchaseOrderShipmentReceiptLines->first();
        $positiveInventoryEventPo2 = $receipt2->purchaseOrderShipmentReceiptLines->first();

        $receipt3 = $this->receivePurchaseOrder($po2);
        $positiveInventoryEventPo3 = $receipt3->purchaseOrderShipmentReceiptLines->first();

        $this->assertDatabaseHas((new BackorderQueueRelease())->getTable(), [
            'backorder_queue_id' => $salesOrderLine->backorderQueue->id,
            'link_id' => $positiveInventoryEventPo1->id,
            'link_type' => PurchaseOrderShipmentReceiptLine::class,
            'released_quantity' => 2,
        ]);
        $this->assertDatabaseHas((new BackorderQueueRelease())->getTable(), [
            'backorder_queue_id' => $salesOrderLine->backorderQueue->id,
            'link_id' => $positiveInventoryEventPo2->id,
            'link_type' => PurchaseOrderShipmentReceiptLine::class,
            'released_quantity' => 3,
        ]);

        $bulkInventoryManger->bulkDeletePositiveInventoryEvents(collect()->add($positiveInventoryEventPo1));

        $this->assertDatabaseHas((new BackorderQueueRelease())->getTable(), [
            'backorder_queue_id' => $salesOrderLine->backorderQueue->id,
            'link_id' => $positiveInventoryEventPo2->id,
            'link_type' => PurchaseOrderShipmentReceiptLine::class,
            'released_quantity' => 3,
        ]);
        $this->assertDatabaseHas((new BackorderQueueRelease())->getTable(), [
            'backorder_queue_id' => $salesOrderLine->backorderQueue->id,
            'link_id' => $positiveInventoryEventPo3->id,
            'link_type' => PurchaseOrderShipmentReceiptLine::class,
            'released_quantity' => 2,
        ]);

        $bulkInventoryManger->bulkDeletePositiveInventoryEvents(collect([$positiveInventoryEventPo2, $positiveInventoryEventPo3]));

        $this->assertDatabaseEmpty((new BackorderQueueRelease())->getTable());

        $this->assertDatabaseHas((new BackorderQueue())->getTable(), [
            'sales_order_line_id' => $salesOrderLine->id,
            'backordered_quantity' => $salesOrderLine->quantity,
            'released_quantity' => 0,
        ]);

        // Now what if we have only some unallocated fifo layers

        $receipt1 = $this->receivePurchaseOrder($po1, 3);
        $receipt2 = $this->receivePurchaseOrder($po2, 1);
        $positiveInventoryEventPo1 = $receipt1->purchaseOrderShipmentReceiptLines->first();
        $positiveInventoryEventPo2 = $receipt2->purchaseOrderShipmentReceiptLines->first();

        $bulkInventoryManger->bulkDeletePositiveInventoryEvents(collect([$positiveInventoryEventPo1]));

        $this->assertDatabaseHas((new BackorderQueueRelease())->getTable(), [
            'backorder_queue_id' => $salesOrderLine->backorderQueue->id,
            'link_id' => $positiveInventoryEventPo2->id,
            'link_type' => PurchaseOrderShipmentReceiptLine::class,
            'released_quantity' => 1,
        ]);

        // original movement for backorder
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_type' => BackorderQueue::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'quantity' => 1,
        ]);
        // new movement for backorder
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_type' => BackorderQueue::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'quantity' => 3,
        ]);
        // original movement for fifo layer
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_id' => $positiveInventoryEventPo2->fifoLayers->first()->id,
            'layer_type' => FifoLayer::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'quantity' => 1,
        ]);

        // Reset
        $this->deletePurchaseOrder($po1);
        $this->deletePurchaseOrder($po2);
        InventoryMovement::query()->delete();
        FifoLayer::query()->delete();
        BackorderQueue::query()->delete();
        BackorderQueueRelease::query()->delete();

        // Partial scenario for inventory movements

        $purchaseOrders = $this->setUpPurchaseOrders([1, 1]);
        $po1 = $purchaseOrders[0];
        $po2 = $purchaseOrders[1];
        $receipt1 = $this->receivePurchaseOrder($po1);
        $positiveInventoryEventPo1 = $receipt1->purchaseOrderShipmentReceiptLines->first();
        $this->receivePurchaseOrder($po2);
        $bulkInventoryManger->bulkAllocateNegativeInventoryEvents(SalesOrderLine::all());
        $bulkInventoryManger->bulkDeletePositiveInventoryEvents(collect([$positiveInventoryEventPo1]));

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_type' => BackorderQueue::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'quantity' => 1,
        ]);
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_type' => BackorderQueue::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'quantity' => 3,
        ]);

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_type' => FifoLayer::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'quantity' => 1,
        ]);

        // Reset
        $this->deletePurchaseOrder($po1);
        $this->deletePurchaseOrder($po2);
        InventoryMovement::query()->delete();
        FifoLayer::query()->delete();
        BackorderQueue::query()->delete();
        BackorderQueueRelease::query()->delete();

        // Release history + unallocated

        $bulkInventoryManger->bulkAllocateNegativeInventoryEvents(SalesOrderLine::all());
        $purchaseOrders = $this->setUpPurchaseOrders([2, 1]);
        $po1 = $purchaseOrders[0];
        $po2 = $purchaseOrders[1];
        $receipt1 = $this->receivePurchaseOrder($po1);
        $receipt1->purchaseOrderShipmentReceiptLines->first();
        $receipt2 = $this->receivePurchaseOrder($po2);
        $positiveInventoryEventPo2 = $receipt2->purchaseOrderShipmentReceiptLines->first();

        $this->assertDatabaseHas((new BackorderQueue())->getTable(), [
            'sales_order_line_id' => $salesOrderLine->id,
            'backordered_quantity' => 5,
            'released_quantity' => 3,
        ]);

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_type' => FifoLayer::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'quantity' => 2,
        ]);
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_type' => FifoLayer::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'quantity' => 1,
        ]);
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_type' => BackorderQueue::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'quantity' => 2,
        ]);

        $bulkInventoryManger->bulkDeletePositiveInventoryEvents(collect([$positiveInventoryEventPo2]));

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_type' => FifoLayer::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'quantity' => 2,
        ]);
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_type' => BackorderQueue::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'quantity' => 1,
        ]);
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_id' => $salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_type' => BackorderQueue::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'quantity' => 2,
        ]);

        $this->assertDatabaseHas((new BackorderQueue())->getTable(), [
            'sales_order_line_id' => $salesOrderLine->id,
            'backordered_quantity' => 5,
            'released_quantity' => 2,
        ]);
    }

    /**
     * @throws NegativeInventoryFulfilledSalesOrderLinesException
     * @throws Throwable
     * @throws ReceivePurchaseOrderLineException
     * @throws BindingResolutionException
     * @throws NotOpenPurchaseOrderException
     */
    public function test_backorder_queue_mappings()
    {
        Queue::fake([SyncBackorderQueueCoveragesJob::class, UpdateProductsInventoryAndAvgCost::class]);
        /*
         * This tests that newly created backorders as a result of deletion of a purchase order are correctly mapped.
         */
        $bulkInventoryManager = (new BulkInventoryManager());

        // Create direct warehouses
        $warehouse = Warehouse::factory(1)->direct()->create();

        // Create suppliers
        $supplier = Supplier::factory(1)->create();

        // Create products, all associated with suppliers
        $products = Product::factory(1)
            ->withProductPricing()
            ->withSupplierProduct(
                (new FactoryDataRecycler([
                    $supplier,
                ]))
            )
            ->create();

        // Create sales orders
        SalesOrder::factory(1)->withLines(
            1,
            (new FactoryDataRecycler())
                ->addRecycledData($products, null, false)
                ->addRecycledData($warehouse)
        )->open()
            ->create([
                'sales_channel_id' => SalesChannel::query()->first()->id,
                'order_date' => $this->faker->dateTimeThisYear(),
            ]);
        SalesOrderLine::query()->update(['quantity' => 1]);

        $bulkInventoryManager->bulkAllocateNegativeInventoryEvents(SalesOrderLine::all());

        $originalBackorderQueue = BackorderQueue::query()->first();

        $purchaseOrder = PurchaseOrder::factory(1)
            ->withLines(
                1,
                (new FactoryDataRecycler())->addRecycledData($products, null, false)
            )
            /*
             * This is Laravel's built-in recycle method.  We need to explore the differences in functionality between
             * my FactoryDataRecycler class and Laravel's recycle method.
             */
            //->recycle($suppliers)
            ->factoryDataRecycler(
                (new FactoryDataRecycler())
                    ->addRecycledData($warehouse, 'destination_warehouse_id')
                    ->addRecycledData($supplier)
            )
            //->receiveAll()
            ->create()->first();

        $purchaseOrderLine = PurchaseOrderLine::query()->first();
        $purchaseOrderLine->update(['quantity' => 1]);

        (new ShipmentManager())->receiveShipment([
            'received_at' => Carbon::parse($purchaseOrder->purchase_order_date)->addDays($this->faker->numberBetween(5, 30))->toIso8601ZuluString(),
            'receipt_lines' => $purchaseOrder->purchaseOrderLines->map(function (PurchaseOrderLine $line) {
                return [
                    'quantity' => $line->quantity,
                    'purchase_order_line_id' => $line->id,
                ];
            })->toArray(),
        ]);

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_type' => SalesOrderLine::class,
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);

        app(PurchaseOrderRepository::class)->bulkDelete(PurchaseOrder::all());

        $this->assertEquals(
            1,
            BackorderQueue::query()
                ->where('sales_order_line_id', SalesOrderLine::query()->first()->id)
                ->count()
        );

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_type' => SalesOrderLine::class,
            'layer_type' => BackorderQueue::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_type' => SalesOrderLine::class,
            'layer_type' => BackorderQueue::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
        ]);

        $this->assertDatabaseEmpty((new PurchaseOrder())->getTable());
        $this->assertDatabaseEmpty((new PurchaseOrderLine())->getTable());
        $this->assertDatabaseEmpty((new PurchaseOrderShipment())->getTable());
        $this->assertDatabaseEmpty((new PurchaseOrderShipmentLine())->getTable());
        $this->assertDatabaseEmpty((new PurchaseOrderShipmentReceipt())->getTable());
        $this->assertDatabaseEmpty((new PurchaseOrderShipmentReceiptLine())->getTable());
        $this->assertDatabaseEmpty((new FifoLayer())->getTable());
        $this->assertDatabaseMissing((new InventoryMovement())->getTable(), [
            'layer_type' => FifoLayer::class,
        ]);
    }

    // TODO: Test deleting a purchase order line (maybe in purchase order manager test) - can't delete if shipped/received
    // TODO: Test that coverages works
    // TODO: Test that releases get deleted with positive event deletion
    // TODO: Test if there is a backorder release with initial count... mabye this shouldn't be a possibililty.
}
