<?php

namespace Tests\Feature;

use App\Http\Requests\StoreInventoryAdjustment;
use App\Http\Requests\StorePurchaseOrder;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Models\BackorderQueue;
use App\Models\Currency;
use App\Models\Customer;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\SalesOrder;
use App\Models\Supplier;
use App\Models\User;
use App\Models\Warehouse;
use App\Services\SalesOrder\WarehouseRoutingMethod;
use Illuminate\Support\Facades\Queue;
use Illuminate\Testing\TestResponse;
use JetBrains\PhpStorm\NoReturn;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class BackorderQueueCoveragesTest extends TestCase
{
    use FastRefreshDatabase;

    private function createSaleAndPurchaseOrders(bool $createSalesOrder = true): TestResponse
    {
        Queue::fake();

        Sanctum::actingAs(User::factory()->create());

        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::query()->first();

        /** @var Product $product */
        $product = Product::factory()->create();
        if($createSalesOrder){
            $this->postJson('/api/sales-orders', [
                'order_date' => now(),
                'order_status' => SalesOrder::STATUS_OPEN,
                'currency_id' => Currency::query()->first()?->id,
                'customer_id' => Customer::factory()->create()->id,
                'sales_order_lines' => [
                    [
                        'product_id' => $product->id,
                        'quantity' => 5,
                        'description' => $product->name,
                        'amount' => 10,
                        'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                    ],
                ],
            ])->assertOk();

            // Backorder queue should be created with no coverages
            $this->assertDatabaseHas('backorder_queues', [
                'backordered_quantity' => 5,
                'released_quantity' => 0,
            ]);
            $this->assertDatabaseEmpty('backorder_queue_coverages');

        }

        // Create PO
        /** @var Supplier $supplier */
        $supplier = Supplier::query()->first();
        $supplier->products()->attach($product->id, ['is_default' => 1]);

        $response = $this->postJson('/api/purchase-orders', [
            'purchase_order_date' => now(),
            'order_status' => PurchaseOrder::STATUS_DRAFT,
            'currency_id' => Currency::query()->first()?->id,
            'supplier_id' => $supplier->id,
            'destination_warehouse_id' => $warehouse->id,
            'purchase_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 3,
                    'amount' => 8,
                ],
            ],
        ])->assertOk();
        // No coverage should be created
        $this->assertDatabaseEmpty('backorder_queue_coverages');

        // Approve PO
        $response = $this->putJson('/api/purchase-orders/'.$response->json('data.id'), [
            'approval_status' => StorePurchaseOrder::APPROVAL_STATUS_APPROVED,
        ])->assertOk();

        // Sync backorder queues
        (new SyncBackorderQueueCoveragesJob(
            purchaseOrderLineIds: collect($response->json('data.items'))->pluck('id')->toArray()
        ))->handle();

        return $response;
    }

    public function test_it_updates_backorder_queue_coverage_when_po_quantity_changes(): void
    {
        $response = $this->createSaleAndPurchaseOrders();

        // Backorder should be covered.
        $this->assertDatabaseCount('backorder_queue_coverages', 1);
        $this->assertDatabaseHas('backorder_queue_coverages', [
            'covered_quantity' => 3,
            'released_quantity' => 0,
        ]);

        // Increase PO Line quantity
        $this->putJson('/api/purchase-orders/'.$response->json('data.id'), [
            'purchase_order_lines' => [
                [
                    'id' => $response->json('data.items.0.id'),
                    'quantity' => 4,
                    'amount' => 8,
                ],
            ],
        ])->assertOk();

        // Sync backorder queues
        (new SyncBackorderQueueCoveragesJob(
            purchaseOrderLineIds: collect($response->json('data.items'))->pluck('id')->toArray()
        ))->handle();

        // Backorder coverage should be updated
        $this->assertDatabaseCount('backorder_queue_coverages', 1);
        $this->assertDatabaseHas('backorder_queue_coverages', [
            'covered_quantity' => 4,
            'released_quantity' => 0,
        ]);

        // Decrease PO Line quantity
        $this->putJson('/api/purchase-orders/'.$response->json('data.id'), [
            'purchase_order_lines' => [
                [
                    'id' => $response->json('data.items.0.id'),
                    'quantity' => 2,
                    'amount' => 8,
                ],
            ],
        ])->assertOk();

        // Sync backorder queues
        (new SyncBackorderQueueCoveragesJob(
            purchaseOrderLineIds: collect($response->json('data.items'))->pluck('id')->toArray()
        ))->handle();

        // Backorder coverage should be updated
        $this->assertDatabaseCount('backorder_queue_coverages', 1);
        $this->assertDatabaseHas('backorder_queue_coverages', [
            'covered_quantity' => 2,
            'released_quantity' => 0,
        ]);
    }

    #[NoReturn] public function test_it_update_backorder_queue_coverage_when_the_backordered_line_is_added_to_an_already_approved_po(): void{

        Queue::fake();

        $response = $this->createSaleAndPurchaseOrders(createSalesOrder: false);

        $this->assertEquals(PurchaseOrder::STATUS_OPEN, $response->json('data.order_status'));

        // Create a new sales order with a different product not currently on the PO
        $product = Product::factory()->create();
        $this->postJson('/api/sales-orders', [
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_OPEN,
            'currency_id' => Currency::query()->first()?->id,
            'customer_id' => Customer::factory()->create()->id,
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 5,
                    'description' => 'New Product',
                    'amount' => 10,
                    'warehouse_id' => $response->json('data.destination_warehouse_id'),
                ],
            ],
        ])->assertOk();

        // Backorder queue should be created with no coverages
        $this->assertDatabaseHas('backorder_queues', [
            'backordered_quantity' => 5,
            'released_quantity' => 0,
        ]);
        $this->assertDatabaseEmpty('backorder_queue_coverages');

        // Now add the new product to the already approved PO
        $supplier = Supplier::query()->findOrFail($response->json('data.supplier_id'));
        $supplier->products()->attach($product->id);

        $this->putJson('/api/purchase-orders/'.$response->json('data.id'), [
            'purchase_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 3,
                    'amount' => 8,
                ],
                [
                    'id' => $response->json('data.items.0.id'),
                    'quantity' => 3,
                    'amount' => 8,
                ],
            ],
        ])->assertOk();

        // Sync backorder queue coverages job should be dispatched, one for each line
        Queue::assertPushed(SyncBackorderQueueCoveragesJob::class, 2);
    }

    public function test_it_can_release_backorder_queue_with_purchase_receipt_and_not_set_created_at_to_null(): void
    {
        $response = $this->createSaleAndPurchaseOrders();

        // Receive PO to release backorder queue.
        $this->postJson('/api/purchase-order-shipments/receive', [
            'purchase_order_id' => $response->json('data.id'),
            'received_at' => now(),
            'receipt_lines' => [
                [
                    'purchase_order_line_id' => $response->json('data.items.0.id'),
                    'quantity' => 2,
                ],
            ],
        ])->assertOk();

        // Backorder queue should be updated
        $this->assertDatabaseHas('backorder_queues', [
            'backordered_quantity' => 5,
            'released_quantity' => 2,
        ]);

        // Sync backorder coverages
        (new SyncBackorderQueueCoveragesJob())->withGlobalCoverage()->handle();

        // Coverage should be updated
        $this->assertDatabaseHas('backorder_queue_coverages', [
            'covered_quantity' => 3,
            'released_quantity' => 2,
            'purchase_order_line_id' => $response->json('data.items.0.id'),
        ]);

        $this->assertDatabaseMissing('backorder_queue_coverages', [
            'created_at' => null,
        ]);
    }

    public function test_it_does_not_clear_inventory_movements_when_deleting_receipt(): void
    {
        $response = $this->createSaleAndPurchaseOrders();

        // Receive PO to release backorder queue.
        $receiptResponse = $this->postJson('/api/purchase-order-shipments/receive', [
            'purchase_order_id' => $response->json('data.id'),
            'received_at' => now(),
            'receipt_lines' => [
                [
                    'purchase_order_line_id' => $response->json('data.items.0.id'),
                    'quantity' => 2,
                ],
            ],
        ])->assertOk();

        // Inventory movements should exist with sales order movements
        // split into those with fifo layers and those with backorder layers.
        $this->assertDatabaseCount('inventory_movements', 5);

        // Delete receipt
        $this->deleteJson('/api/purchase-order-shipments/receipts/'.$receiptResponse->json('data.id'))
            ->assertOk();

        (new SyncBackorderQueueCoveragesJob)->withGlobalCoverage()->handle();

        // Sales order should be put in the backorder queue
        $this->assertDatabaseCount('inventory_movements', 4);
        $this->assertDatabaseHas('inventory_movements', [
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'quantity' => 2,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'quantity' => 3,
        ]);

        // Backorder queue should be covered
        $this->assertDatabaseCount('backorder_queues', 1);
        $this->assertDatabaseHas('backorder_queues', [
            'backordered_quantity' => 5,
            'released_quantity' => 0,
        ]);

        $this->assertDatabaseCount('backorder_queue_coverages', 1);
        $this->assertDatabaseHas('backorder_queue_coverages', [
            'covered_quantity' => 3,
            'released_quantity' => 0,
        ]);
        $this->assertDatabaseEmpty('backorder_queue_releases');
    }

    public function test_it_updates_coverage_quantity_when_backorder_queue_is_partially_released_by_adjustment(): void
    {
        $response = $this->createSaleAndPurchaseOrders();

        // Create positive adjustment.
        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => now(),
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'product_id' => $response->json('data.items.0.sku.id'),
            'warehouse_id' => $response->json('data.destination_warehouse_id'),
            'quantity' => 3,
            'unit_cost' => 10,
        ])->assertOk();

        (new SyncBackorderQueueCoveragesJob(
            purchaseOrderLineIds: collect($response->json('data.items'))->pluck('id')->toArray()
        ))->withGlobalCoverage()->handle();

        // Backorder queue should be released.
        $this->assertDatabaseCount('backorder_queue_releases', 1);
        $this->assertDatabaseHas('backorder_queue_releases', [
            'released_quantity' => 3,
        ]);
        $this->assertDatabaseHas('backorder_queues', [
            'backordered_quantity' => 5,
            'released_quantity' => 3,
        ]);

        // Coverages should be updated
        $this->assertDatabaseHas('backorder_queue_coverages', [
            'covered_quantity' => 2,
            'released_quantity' => 0,
        ]);
    }

    public function test_it_does_not_reduce_backordered_quantity_when_positive_adjustment_is_deleted()
    {

        Queue::fake();
        $response = $this->createSaleAndPurchaseOrders();

        // Create positive adjustment.
        $adjustmentResponse = $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => now(),
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'product_id' => $response->json('data.items.0.sku.id'),
            'warehouse_id' => $response->json('data.destination_warehouse_id'),
            'quantity' => 3,
            'unit_cost' => 10,
        ])->assertOk();

        (new SyncBackorderQueueCoveragesJob)->withGlobalCoverage()->handle();
        $this->assertDatabaseHas('backorder_queue_coverages', [
            'covered_quantity' => 2,
            'released_quantity' => 0,
        ]);

        // Delete the adjustment
        $this->deleteJson('/api/inventory-adjustments/'.$adjustmentResponse->json('data.id'))->assertOk();

        // Run coverages script
        (new SyncBackorderQueueCoveragesJob)->withGlobalCoverage()->handle();

        // Backorder queue should not have release
        $this->assertDatabaseHas('backorder_queues', [
            'backordered_quantity' => 5,
            'released_quantity' => 0,
        ]);
        // There should be no releases
        $this->assertDatabaseEmpty('backorder_queue_releases');

        $this->assertDatabaseHas('backorder_queue_coverages', [
            'covered_quantity' => 3,
            'released_quantity' => 0,
        ]);
    }

    public function test_it_does_not_reduce_backordered_quantity_when_reducing_receipt_quantity(): void
    {
        $response = $this->createSaleAndPurchaseOrders();

        // Receive PO to release backorder queue.
        $receiptResponse = $this->postJson('/api/purchase-order-shipments/receive', [
            'purchase_order_id' => $response->json('data.id'),
            'received_at' => now(),
            'receipt_lines' => [
                [
                    'purchase_order_line_id' => $response->json('data.items.0.id'),
                    'quantity' => 2,
                ],
            ],
        ])->assertOk();

        // Run coverages script
        (new SyncBackorderQueueCoveragesJob)->withGlobalCoverage()->handle();

        // Coverage must exist
        $this->assertDatabaseCount('backorder_queue_coverages', 1);
        $this->assertDatabaseHas('backorder_queue_coverages', [
            'covered_quantity' => 3,
            'released_quantity' => 2
        ]);

        // Inventory movements should exist with sales order movements
        // split into those with fifo layers and those with backorder layers.
        $this->assertDatabaseCount('inventory_movements', 5);

        // Shipments and receipt lines must exist
        $this->assertDatabaseCount('purchase_order_shipment_lines', 1);
        $this->assertDatabaseHas('purchase_order_shipment_lines', [
            'quantity' => 2,
        ]);
        $this->assertDatabaseCount('purchase_order_shipment_receipt_lines', 1);
        $this->assertDatabaseHas('purchase_order_shipment_receipt_lines', [
            'quantity' => 2,
        ]);

        // Reduce receipt quantity
        $this->putJson('/api/purchase-order-shipments/receipts/'.$receiptResponse->json('data.id'), [
            'received_at' => now(),
            'receipt_lines' => [
                [
                    'purchase_order_line_id' => $response->json('data.items.0.id'),
                    'quantity' => 1,
                ],
            ],
        ])->assertOk();

        // Run coverages script
        (new SyncBackorderQueueCoveragesJob)->withGlobalCoverage()->handle();

        // Sales order movements should be split.
        $this->assertDatabaseCount('inventory_movements', 5);
        $this->assertDatabaseHas('inventory_movements', [
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'quantity' => -4,
            'layer_type' => BackorderQueue::class,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'quantity' => 4,
            'layer_type' => BackorderQueue::class,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'quantity' => -1,
            'layer_type' => FifoLayer::class,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'quantity' => 1,
            'layer_type' => FifoLayer::class,
        ]);

        // Backorder queue should be covered
        $this->assertDatabaseCount('backorder_queues', 1);
        $this->assertDatabaseHas('backorder_queues', [
            'backordered_quantity' => 5,
            'released_quantity' => 1,
        ]);

        $this->assertDatabaseCount('backorder_queue_coverages', 1);
        $this->assertDatabaseHas('backorder_queue_coverages', [
            'covered_quantity' => 3,
            'released_quantity' => 1,
        ]);
        $this->assertDatabaseCount('backorder_queue_releases', 1);
        $this->assertDatabaseHas('backorder_queue_releases', [
            'link_type' => PurchaseOrderShipmentReceiptLine::class,
            'released_quantity' => 1,
        ]);

        // Shipments and receipt lines must be updated
        $this->assertDatabaseCount('purchase_order_shipment_lines', 1);
        $this->assertDatabaseHas('purchase_order_shipment_lines', [
            'quantity' => 1,
        ]);
        $this->assertDatabaseCount('purchase_order_shipment_receipt_lines', 1);
        $this->assertDatabaseHas('purchase_order_shipment_receipt_lines', [
            'quantity' => 1,
        ]);
    }

    public function test_it_can_fully_receive_po_and_release_backorder_queue(): void
    {
        $response = $this->createSaleAndPurchaseOrders();

        // Receive PO to release backorder queue.
        $this->postJson('/api/purchase-order-shipments/receive', [
            'purchase_order_id' => $response->json('data.id'),
            'received_at' => now(),
            'receipt_lines' => [
                [
                    'purchase_order_line_id' => $response->json('data.items.0.id'),
                    'quantity' => 3,
                ],
            ],
        ])->assertOk();

        // Run coverages script
        (new SyncBackorderQueueCoveragesJob)->withGlobalCoverage()->handle();

        // Inventory movements should exist with sales order movements
        // split into those with fifo layers and those with backorder layers.
        $this->assertDatabaseCount('inventory_movements', 5);

        // Sales order movements should be split.
        $this->assertDatabaseCount('inventory_movements', 5);
        $this->assertDatabaseHas('inventory_movements', [
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'quantity' => -2,
            'layer_type' => BackorderQueue::class,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'quantity' => 2,
            'layer_type' => BackorderQueue::class,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'quantity' => -3,
            'layer_type' => FifoLayer::class,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'quantity' => 3,
            'layer_type' => FifoLayer::class,
        ]);

        // Backorder queue should be covered
        $this->assertDatabaseCount('backorder_queues', 1);
        $this->assertDatabaseHas('backorder_queues', [
            'backordered_quantity' => 5,
            'released_quantity' => 3,
        ]);

        $this->assertDatabaseCount('backorder_queue_coverages', 1);
        $this->assertDatabaseHas('backorder_queue_coverages', [
            'covered_quantity' => 3,
            'released_quantity' => 3,
        ]);
        $this->assertDatabaseCount('backorder_queue_releases', 1);
        $this->assertDatabaseHas('backorder_queue_releases', [
            'link_type' => PurchaseOrderShipmentReceiptLine::class,
            'released_quantity' => 3,
        ]);
    }
}
