<?php

namespace Tests\Feature;

use App\Exceptions\NegativeInventoryFulfilledSalesOrderLinesException;
use App\Exceptions\PurchaseOrder\NotOpenPurchaseOrderException;
use App\Exceptions\PurchaseOrder\ReceivePurchaseOrderLineException;
use App\Managers\ProductInventoryManager;
use App\Models\Customer;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\PurchaseOrderShipment;
use App\Models\PurchaseOrderShipmentLine;
use App\Models\PurchaseOrderShipmentReceipt;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\SalesOrderLineLayer;
use App\Models\User;
use App\Models\Warehouse;
use App\Services\InventoryManagement\BulkInventoryManager;
use App\Services\PurchaseOrder\ShipmentManager;
use Carbon\Carbon;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class AlternativeBulkImportTest extends TestCase
{
    use FastRefreshDatabase;

    /**
     * @throws NotOpenPurchaseOrderException
     * @throws ReceivePurchaseOrderLineException
     * @throws NegativeInventoryFulfilledSalesOrderLinesException
     * @throws Throwable
     */
    public function test_it_can_bulk_receive_purchase_orders(): void
    {
        Queue::fake();

        $bulkInventoryManager = new BulkInventoryManager();

        $products = Product::factory(5)->create();
        $warehouse = Warehouse::first();

        $salesOrderLines = [];
        foreach ($products as $product) {
            $salesOrderLines[] = [
                'product_id' => $product->id,
                'description' => $product->name,
                'quantity' => 1,
                'amount' => 100,
            ];
        }

        Sanctum::actingAs(User::factory()->create());

        $this->postJson('/api/sales-orders', [
            'order_status' => 'open',
            'currency_code' => 'USD',
            'order_date' => Carbon::now(),
            'customer_id' => Customer::factory()->create()->id,
            'sales_order_lines' => $salesOrderLines,
        ])->assertOk();

        $purchaseOrder = PurchaseOrder::factory()
            ->has(PurchaseOrderLine::factory()->count(5)->sequence(
                fn ($sequence) => ['product_id' => $products[$sequence->index]->id]
            ))
            ->create(['destination_warehouse_id' => $warehouse->id]);

        (new ProductInventoryManager($products->pluck('id')->toArray()))->updateProductInventoryAndAvgCost();

        $purchaseOrderLines = $purchaseOrder->purchaseOrderLines;

        $receipt_lines = [];

        foreach ($purchaseOrderLines as $line) {
            $receipt_lines[] = ['purchase_order_line_id' => $line->id, 'quantity' => $line->quantity];
        }

        $payload = [
            'purchase_order_id' => $purchaseOrder->id,
            'received_at' => now(),
        ];

        (new ShipmentManager())->newReceiveLines($receipt_lines, $payload);

        $this->assertDatabaseCount((new PurchaseOrderShipment())->getTable(), 1);
        $this->assertDatabaseCount((new PurchaseOrderShipmentLine())->getTable(), 5);
        $this->assertDatabaseCount((new PurchaseOrderShipmentReceipt())->getTable(), 1);
        $this->assertDatabaseCount((new PurchaseOrderShipmentReceiptLine())->getTable(), 5);
        $this->assertDatabaseCount((new FifoLayer())->getTable(), 5);
        $this->assertDatabaseCount((new SalesOrderLineLayer())->getTable(), 5);
        $this->assertDatabaseCount((new InventoryMovement())->getTable(), 15);
        $this->assertEquals($receipt_lines[0]['quantity'], PurchaseOrderLine::query()->first()->received_quantity);
        $this->assertEquals(0, PurchaseOrderLine::query()->first()->unreceived_quantity);
    }
}
