<?php

namespace Tests\Feature;

use App\Data\AccountingTransactionData;
use App\Data\AccountingTransactionLineData;
use App\Data\AccountingTransactionLineNominalCodeUpdateData;
use App\Data\AccountingTransactionUpdateData;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Enums\AccountingTransactionTypeEnum;
use App\Helpers;
use App\Models\AccountingTransaction;
use App\Models\AccountingTransactionLine;
use App\Models\NominalCode;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\User;
use App\Repositories\SettingRepository;
use App\Services\Accounting\AccountingTransactionManager;
use Laravel\Sanctum\Sanctum;
use Modules\Xero\Tests\XeroTest;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;

class AccountingTransactionControllerTest extends XeroTest
{
    use FastRefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();

        Sanctum::actingAs(User::first());
    }

    public function test_it_can_sync_with_xero_service(): void
    {
        SalesOrder::factory()->hasSalesOrderLines(3)->create([
            'order_date' => now(),
        ]);
        app(AccountingTransactionManager::class)->sync();

        $this->mockConnectionRequests();
        $this->mockEndpoint(
            endpoint: '/Invoices?*',
            response: [
                'Invoices' => [],
            ]
        );

        $this->postJson('/api/accounting/transactions/sync-external', [
            'ids' => AccountingTransaction::query()->pluck('id')->toArray(),
        ])->assertSuccessful();
    }

    public function test_it_can_store_accounting_transaction(): void
    {
        $nc = NominalCode::create([
            'name' => 'Lost In Transit Expense',
            'code' => '601',
            'type' => NominalCode::TYPE_EXPENSE,
        ]);

        app(SettingRepository::class)->set(Setting::KEY_NC_MAPPING_RECEIVING_DISCREPANCIES, $nc->id);

        $purchaseOrder = PurchaseOrder::factory()
            ->hasPurchaseOrderLines(3)
            ->create([
            'purchase_order_date' => now(),
        ]);

        // Under-received
        $accountingTransactionLines = collect();
        $underReceivedPurchaseOrderLine = $purchaseOrder->purchaseOrderLines->first();

        $accountingTransactionLines->add(AccountingTransactionLineData::from([
            'type' => AccountingTransactionLineTypeEnum::DEBIT,
            'nominal_code_id' => Helpers::setting(Setting::KEY_NC_MAPPING_RECEIVING_DISCREPANCIES),
            'description' => 'Receiving discrepancy for ' . $underReceivedPurchaseOrderLine->product?->sku ?? $underReceivedPurchaseOrderLine->description,
            // Completely unreceived for this item
            'quantity' => $underReceivedPurchaseOrderLine->quantity,
            'amount' => $underReceivedPurchaseOrderLine->amount,
            'link_id' => $underReceivedPurchaseOrderLine->id,
            'link_type' => PurchaseOrderLine::class,
        ]));

        $accountingTransactionLines->add(AccountingTransactionLineData::from([
            'type' => AccountingTransactionLineTypeEnum::CREDIT,
            'nominal_code_id' => Helpers::setting(Setting::KEY_NC_MAPPING_ACCRUED_PURCHASES),
            'description' => 'Receiving discrepancy for ' . $underReceivedPurchaseOrderLine->product?->sku ?? $underReceivedPurchaseOrderLine->description,
            'quantity' => $underReceivedPurchaseOrderLine->quantity,
            'amount' => $underReceivedPurchaseOrderLine->amount,
            'link_id' => $underReceivedPurchaseOrderLine->id,
            'link_type' => PurchaseOrderLine::class,
        ]));

        $this->post(route('accounting.transactions.store', AccountingTransactionData::from([
            'transaction_date' => $purchaseOrder->purchase_order_date,
            'type' => AccountingTransactionTypeEnum::RECEIVING_DISCREPANCY,
            'name' => 'Test Invoice',
            'reference' => 'Receiving discrepancy for ' . $purchaseOrder->purchase_order_number,
            'link_id' => $purchaseOrder->id,
            'link_type' => PurchaseOrder::class,
            'accounting_transaction_lines' => $accountingTransactionLines,
        ])->toArray()))->assertSuccessful();

        $this->assertDatabaseHas(AccountingTransaction::class, [
            'type' => AccountingTransactionTypeEnum::RECEIVING_DISCREPANCY,
            'name' => 'Test Invoice',
            'reference' => 'Receiving discrepancy for ' . $purchaseOrder->purchase_order_number,
            'link_id' => $purchaseOrder->id,
            'link_type' => PurchaseOrder::class,
            'total' => $underReceivedPurchaseOrderLine->amount * $underReceivedPurchaseOrderLine->quantity,
        ]);

        $this->assertDatabaseHas(AccountingTransactionLine::class, [
            'type' => AccountingTransactionLineTypeEnum::DEBIT,
            'nominal_code_id' => Helpers::setting(Setting::KEY_NC_MAPPING_RECEIVING_DISCREPANCIES),
            'description' => 'Receiving discrepancy for ' . $underReceivedPurchaseOrderLine->product?->sku ?? $underReceivedPurchaseOrderLine->description,
            'quantity' => $underReceivedPurchaseOrderLine->quantity,
            'amount' => $underReceivedPurchaseOrderLine->amount,
            'link_id' => $underReceivedPurchaseOrderLine->id,
            'link_type' => PurchaseOrderLine::class,
        ]);

        $this->assertDatabaseHas(AccountingTransactionLine::class, [
            'type' => AccountingTransactionLineTypeEnum::CREDIT,
            'nominal_code_id' => Helpers::setting(Setting::KEY_NC_MAPPING_ACCRUED_PURCHASES),
            'description' => 'Receiving discrepancy for ' . $underReceivedPurchaseOrderLine->product?->sku ?? $underReceivedPurchaseOrderLine->description,
            'quantity' => $underReceivedPurchaseOrderLine->quantity,
            'amount' => $underReceivedPurchaseOrderLine->amount,
            'link_id' => $underReceivedPurchaseOrderLine->id,
            'link_type' => PurchaseOrderLine::class,
        ]);
    }

    public function test_it_can_update_nominal_codes_for_accounting_transaction(): void
    {
        $oldNominalCode = NominalCode::factory()->create();
        $newNominalCode = NominalCode::factory()->create();

        $salesOrder = SalesOrder::factory()->hasSalesOrderLines()->create();
        $saleOrderLine = $salesOrder->salesOrderLines->first();

        $transaction = AccountingTransaction::factory([
            'type' => AccountingTransactionTypeEnum::SALES_ORDER_INVOICE,
            'link_id' => $salesOrder->id,
            'link_type' => SalesOrder::class,
        ])->hasAccountingTransactionLines(1, [
            'type' => AccountingTransactionLineTypeEnum::SALES_INVOICE_LINE,
            'nominal_code_id' => $oldNominalCode,
            'link_id' => $saleOrderLine->id,
            'link_type' => SalesOrderLine::class,
        ])
            ->create();

        $this->putJson(route('accounting.transactions.update', $transaction->id), AccountingTransactionUpdateData::from([
            'is_locked' => true,
            'lines' => [
                AccountingTransactionLineNominalCodeUpdateData::from([
                    'id' => $transaction->accountingTransactionLines->first()->id,
                    'nominal_code_id' => $newNominalCode->id,
                ])->toArray()
            ]
        ])->toArray())->assertSuccessful();

        $this->assertDatabaseHas(AccountingTransaction::class, [
            'id' => $transaction->id,
            'is_locked' => true,
        ]);

        $this->assertDatabaseHas(AccountingTransactionLine::class, [
            'id' => $transaction->accountingTransactionLines->first()->id,
            'nominal_code_id' => $newNominalCode->id,
        ]);
    }

    public function test_it_can_bulk_replace_nominal_codes_for_accounting_transactions(): void
    {
        $oldNominalCode = NominalCode::factory()->create();
        $newNominalCode = NominalCode::factory()->create();

        $salesOrder = SalesOrder::factory()->hasSalesOrderLines()->create();
        $saleOrderLine = $salesOrder->salesOrderLines->first();

        $transaction = AccountingTransaction::factory([
            'type' => AccountingTransactionTypeEnum::SALES_ORDER_INVOICE,
            'link_id' => $salesOrder->id,
            'link_type' => SalesOrder::class,
        ])->hasAccountingTransactionLines(1, [
            'type' => AccountingTransactionLineTypeEnum::SALES_INVOICE_LINE,
            'nominal_code_id' => $oldNominalCode,
            'link_id' => $saleOrderLine->id,
            'link_type' => SalesOrderLine::class,
        ])
            ->create();

        $this->postJson(route('accounting.transactions.bulkReplaceNominalCodes', [
            'ids' => [$transaction->id],
            'is_locked' => true,
            'old_nominal_code_id' => $oldNominalCode->id,
            'new_nominal_code_id' => $newNominalCode->id,
        ]))->assertSuccessful();

        $this->assertDatabaseHas(AccountingTransaction::class, [
            'id' => $transaction->id,
            'is_locked' => true,
        ]);

        $this->assertDatabaseHas(AccountingTransactionLine::class, [
            'id' => $transaction->accountingTransactionLines->first()->id,
            'nominal_code_id' => $newNominalCode->id,
        ]);
    }
}
