<?php

namespace Tests\Feature\API;

use App\Http\Requests\StoreInventoryAdjustment;
use App\Models\Address;
use App\Models\Customer;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\User;
use App\Models\Warehouse;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\DatabaseTransactions;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Http;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Tests\TestCaseOld;

/**
 * @group manual
 */
class StarshipitTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    /**
     * A basic unit test example.
     */
    public function test_can_connect_to_starshipit(): void
    {
        $this->markTestSkipped('should not hit starship it endpoint');
        $url = config('starshipit.endpoint').'/orders/unshipped';
        $headers = [
            'Content-Type' => 'application/json',
            'StarShipIT-Api-Key' => config('starshipit.test_api_key'),
            'Ocp-Apim-Subscription-Key' => config('starshipit.test_subscription_key'),
        ];
        $response = Http::withHeaders($headers)->get($url);

        $this->assertEquals(200, $response->status());
    }

    public function test_it_can_fulfill(): void
    {
        $this->markTestSkipped('Need to provide api credentials');
        $integration = Integration::query()->where('name', Integration::NAME_STARSHIPIT)->firstOrFail();

        IntegrationInstance::factory()->create([
            'integration_id' => $integration->id,
            'name' => 'Starshipit',
            'connection_settings' => [
                'apiKey' => config('starshipit.test_api_key'),
                'subscriptionKey' => config('starshipit.test_subscription_key'),
            ],
        ]);

        /** @var Customer $customer */
        $customer = Customer::factory()->create();
        /** @var Product $product */
        $product = Product::factory()->create([
            'dimension_unit' => 'in',
            'weight_unit' => 'lb',
            'weight' => 3,
            'length' => 5,
            'width' => 5,
            'height' => 5,
        ]);
        /** @var Address $address */
        $address = Address::factory()->create();
        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::factory()->create([
            'address_id' => $address->id,
        ]);

        Sanctum::actingAs(User::factory()->create());

        $timeOfOrder = Carbon::now()->format('Y-m-d H:i:s');

        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => $timeOfOrder,
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 5.00,
            'quantity' => 50,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ])->assertSuccessful();

        $salesOrderResponse = $this->postJson('/api/sales-orders', [
            'sales_order_number' => 'SSI_test'.$this->faker->randomNumber(5),
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => $timeOfOrder,
            'customer_id' => $customer->id,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'description' => $product->name,
                    'warehouse_id' => $warehouse->id,
                    'amount' => 5.00,
                    'quantity' => 10,
                ],
            ],
        ]);

        $salesOrderResponse->assertSuccessful();

        $salesOrderId = $salesOrderResponse->json('data.id');

        $response = $this->postJson("/api/sales-orders/$salesOrderId/fulfill", [
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrderResponse->json('data.item_info.0.sales_order_line_id'),
                    'quantity' => 2,
                ],
            ],
            'fulfillment_type' => SalesOrderFulfillment::TYPE_STARSHIPIT,
            'metadata' => ['signature_required' => false],
            'requested_shipping_method' => 'UPS Ground',
            'requested_shipping_method_id' => '',
            'warehouse' => $warehouse->id,
        ]);

        $response->assertSuccessful();
    }
}
