import { isEmpty, isArray } from 'lodash'

/**
 * Values must be unique
 * @param {*} values
 */
export const uniqueValues = (values) => {
  if (values === null) return true
  const parsed = parseOptionValues(values).map(c => c.value.toLowerCase())
  return [...new Set(parsed)].length === parsed.length
}

export const maxLength = (values, vm) => {
  if (!values) return true
  const validations = vm.getValidationsByType(true)
  if (isEmpty(validations)) return true // No validation specified

  // We validate based on the type
  const parsed = parseOptionValues(values)
  if (!parsed) return true

  if (vm.isTextValidation(vm.attributeType) && validations.limit !== undefined &&
    !isNaN(parseInt(validations.limit))) {
    // Check character limit
    return validate(parsed, (val) => val.length <= parseInt(validations.limit))
  }
  return true // All good
}

export const numeric = (values) => {
  if (!values) return true
  const parsed = parseOptionValues(values)
  if (!parsed) return true
  return validate(parsed, (value) => !isNaN(Number(value)))
}

export const min = (values, vm) => {
  if (!values) return true

  // Get the validations
  const validations = vm.getValidationsByType(true)
  if (isEmpty(validations)) return true // No validation specified

  // We validate based on the type
  const parsed = parseOptionValues(values)
  if (!parsed) return true

  if (validations.min !== undefined && !isNaN(parseFloat(validations.min))) {
    return validate(parsed, (val) => val >= parseFloat(validations.min))
  }
  return true
}

export const precision = (values, vm) => {
  if (!values) return true

  // Get the validations
  const validations = vm.getValidationsByType(true)
  if (isEmpty(validations)) return true // No validation specified

  // We validate based on the type
  const parsed = parseOptionValues(values)
  if (!parsed) return true

  if (validations.precision !== undefined && !isNaN(parseFloat(validations.precision))) {
    return validate(parsed, (val) => {
      const valString = val.toString()
      const index = valString.indexOf('.')
      if (index === -1) return true
      const places = valString.length - index - 1
      return places <= validations.precision
    })
  }
  return true
}

export const max = (values, vm) => {
  if (!values) return true

  // Get the validations
  const validations = vm.getValidationsByType(true)
  if (isEmpty(validations)) return true // No validation specified

  // We validate based on the type
  const parsed = parseOptionValues(values)
  if (!parsed) return true

  if (validations.max !== undefined && !isNaN(parseFloat(validations.max))) {
    return validate(parsed, (val) => val <= parseFloat(validations.max))
  }
  return true
}

/**
 * Checks custom values for validation
 * @param {*} values
 * @param {*} vm
 */
export const checkValues = (values, vm) => {
  if (!vm.isValidationApplicable(vm.attributeType) ||
    !vm.hasOptions()) return true // No validations necessary

  // Get the validations
  const validations = vm.getValidationsByType(true)
  if (isEmpty(validations)) return true // No validation specified

  // We validate based on the type
  const parsed = parseOptionValues(values)
  if (!parsed) return true

  // Validate based on type
  if (vm.isTextValidation(vm.attributeType) && validations.limit !== undefined &&
    !isNaN(parseInt(validations.limit))) {
    // Check character limit
    return validate(parsed, (val) => val.length <= parseInt(validations.limit))
  } else if (vm.isNumberValidation(vm.attributeType)) {
    // Precision, Min & Max
    let min = true
    let max = true
    let numeric = validate(parsed, (val) => !isNaN(parseFloat(val)))
    if (validations.min !== undefined && !isNaN(parseFloat(validations.min))) {
      min = validate(parsed, (val) => val >= parseFloat(validations.min))
    }
    if (validations.max !== undefined && !isNaN(parseFloat(validations.max))) {
      max = validate(parsed, (val) => val <= parseFloat(validations.max))
    }

    return numeric && min && max
  }

  return true // All good
}

/**
 * All values must pass the custom validation
 * @param {*} values
 * @param {*} passes
 */
const validate = (values, passes) => !values.some(current => !passes(current.value))

/**
 * Builds validations object based on type
 * @param {*} type
 * @param {*} rules
 * @param {*} vm
 */
export const buildValidationsByType = (type, rules, vm) => {
  // set empty object later because sometimes 'validation' should remain undefined
  // to compare the difference between initial and changed attribute correctly
  let validation
  if (vm.isTextValidation(type)) {
    validation = {}
    validation.limit = rules && rules.limit !== undefined && rules.limit !== null ? rules.limit : ''
  }
  if (vm.isNumberValidation(type)) {
    validation = {}
    validation.min = rules && rules.min !== undefined && rules.min !== null ? rules.min : ''
    validation.max = rules && rules.max !== undefined && rules.max !== null ? rules.max : ''
  }
  if (type === 'numeric') {
    validation = {}
    validation.precision = rules && rules.precision !== undefined && rules.precision !== null ? rules.precision : ''
  }

  return validation
}

/**
 * Parses option values to string if it's array
 * and to array if it's string
 *
 * @param values
 * @returns {string|array}
 */
export const parseOptionValues = (values) => {
  if (values === null) return null

  if (!isArray(values)) {
    // Convert to array
    return values.toString().trim('\n').trim().split('\n').map((value, index) => {
      return { value: value, sort_order: index }
    })
  }

  let text = values.reduce((acc, item) => {
    acc += `${item.value}\n`
    return acc
  }, '')
  if (text === '') return null
  return text.trim('\n')
}

export const parseStringToArray = (stringValue) => {
  if (stringValue === null || stringValue === '') return null

  // Convert to array
  return stringValue.toString().trim().split('\n').map((value, index) => {
    return { value: value, sort_order: index }
  })
}

export const parseArrayToString = (arr) => {
  if (arr === null) return null

  let text = arr.reduce((acc, item) => {
    acc += `${item.value}\n`
    return acc
  }, '')

  if (text === '') return null

  return text.trim('\n')
}
