import { legasyDrawers as legacyDrawerIds } from '@/assets/legasyDrawers'

export function LayerData () {
  this.id = ''
  this.data = {}

  this.fromObject = function (obj) {
    this.data = obj

    return this
  }

  this.setId = function (idString) {
    this.id = idString
    return this
  }

  this.fromQueryString = function (query) {
    query = query || ''
    this.data = {}
    let valuePairs = query.split(';')

    valuePairs.map(vp => {
      let v = vp.split(':')
      if (v[0] !== '') this.data[v[0]] = (typeof (v[1]) === 'undefined' ? true : v[1])
    }).filter(x => x)

    return this
  }

  this.toQueryString = function () {
    let query = ''
    for (const prop in this.data) {
      if (this.data.hasOwnProperty(prop)) {
        query += (query === '' ? '' : ';')
        query += prop + (this.data[prop] === true ? '' : ':' + this.data[prop])
      }
    }
    return query
  }

  this.getId = function () {
    return this.id
  }

  this.getData = function () {
    return this.data
  }

  this.get = function (key, defaultValue) {
    if (typeof (this.data[key]) === 'undefined') {
      if (typeof (defaultValue) === 'undefined') {
        return null
      } else {
        return defaultValue || null
      }
    }
    return this.data[key]
  }

  this.set = function (key, newValue) {
    if (this.data[key] && newValue === null) {
      delete this.data[key]
    } else {
      this.data[key] = newValue
    }
  }
}

export function LayerManager (Vue, router) {
  const self = this
  this.layers = []
  this.data = {}

  const removeElement = function (array, elem) {
    var index = array.indexOf(elem)
    if (index > -1) {
      array.splice(index, 1)
    }
    return self
  }

  const randomUid = function () {
    return Math.random().toString(36).substring(2, 7)
  }

  const openLayer = function (layerUid, layerData) {
    // do the actual opening of the drawer component
    // (do not change route here)
    // use this function in

    let dataItem
    if (layerData && layerData.getData) dataItem = layerData.getData()

    self.layers.push(layerUid)

    Vue.set(self.data, layerUid, dataItem)
  }

  const closeLayer = function (layerUid) {
    // do the actual closing of the drawer component
    // (do not change route here)

    removeElement(self.layers, layerUid)
    // delete self.data[layerUid]
    Vue.delete(self.data, layerUid)
  }

  const toLayerDataInstance = function (data) {
    var dataObject = {}

    if (data instanceof LayerData) {
      dataObject = data
    } else if (typeof (data) === 'string') {
      dataObject = (new LayerData()).fromQueryString(data)
    } else if (typeof (data) === 'object') {
      dataObject = (new LayerData()).fromObject(data)
    } else {
      dataObject = (new LayerData())
    }

    return dataObject
  }

  this.editLayer = (layerId, changes) => {
    // use router here
    // obtain layers array from router, then addToLayersArray(changes)
    // and then update router with that new array

    // Example changes:
    //    {
    //       mode: "DashboardFinancials",
    //       hideNavigator: false
    //    }

    if (!this.layers.includes(layerId)) {
      return this
    }

    var dataObject = this.data[layerId]

    dataObject = toLayerDataInstance(dataObject)

    for (const prop in changes) {
      if (changes.hasOwnProperty(prop)) {
        dataObject.set(prop, changes[prop])
      }
    }

    let newQuery = JSON.parse(JSON.stringify(router.currentRoute.query)) // must clone, not reference

    newQuery[layerId] = dataObject.toQueryString()

    router.push({ query: newQuery }).catch(() => {})
    return this
  }

  this.zIndex = function (layerId, offset, multiplier) {
    if (!this.layers.includes(layerId)) {
      return false
    }

    offset = (offset === undefined ? 5000000 : offset)
    multiplier = (multiplier === undefined ? 10 : multiplier)

    return offset + (this.layers.indexOf(layerId) + 1) * multiplier
  }

  this.updateLayers = (queryTo, queryFrom) => {
    // check the contents of the arrays and open/close layers to match it;
    // use openLayer() and closeLayer() here

    let newLayersString = (queryTo && queryTo.layers ? queryTo.layers : '')
    let oldLayersString = (queryFrom && queryFrom.layers ? queryFrom.layers : '')

    let newLayers = newLayersString.split(',')
    let oldLayers = oldLayersString.split(',')

    // Each layer must have an uid, and we must find the instance in the
    // layers array of this object.

    // let layerData = {}

    // We must detect any layer that exists in the layers array but is no longer
    // mentioned in the layers received in queryTo, remove them from this.layers
    // and run the closing fuction.

    for (let i in oldLayers) {
      if (oldLayers[i] === '' || newLayers.includes(oldLayers[i])) {
        continue
      }
      closeLayer(oldLayers[i])
    }

    // We must detect any layer that is not in the this.layers array, add it and
    // run the opening function.

    for (let i in newLayers) {
      if (newLayers[i] === '' ||
        (
          oldLayers.includes(newLayers[i]) &&
          !isLegacyType(newLayers[i]) // never skip legacy, so data can be updated
        )
      ) {
        continue
      }
      let layerName = newLayers[i]
      openLayer(layerName, (new LayerData()).setId(layerName).fromQueryString(queryTo[layerName]))
    }

    this.reorderLayers(newLayers)
  }

  this.reorderLayers = (ordered) => {
    this.layers = JSON.parse(JSON.stringify(ordered))
  }

  let isLegacyType = drawerType => legacyDrawerIds.includes(drawerType)

  this.open = function (layerData) {
    // use router here
    // obtain layers array from router, then addToLayersArray(layerData)
    // and then update router with that new array

    // Example layerData:
    //    {
    //       t: "ProductDrawer",  // the drawer's type
    //       name: "ProductDrawer",
    //       id: 3,
    //       mode: "DashboardFinancials",
    //       hideNavigator: false
    //    }
    // Can have UID or not.  Can have ID or not.

    var dataObject = toLayerDataInstance(layerData)

    let layerName

    // if it's a legacy drawer:
    //      use layerData.t as UID
    //      append it to the end of the layers list unless it's already there
    //      in which case, move it to the end
    // else:
    //      create an UID and append it to the end of the layers list

    let type = dataObject.get('t', dataObject.get('name'))

    layerName = isLegacyType(type) ? type : randomUid()

    dataObject.setId(layerName)

    let currentLayers = router.currentRoute.query.layers ? router.currentRoute.query.layers.split(',') : []
    removeElement(currentLayers, layerName)
    currentLayers.push(layerName)
    let newQuery = JSON.parse(JSON.stringify(router.currentRoute.query)) // must clone, not reference
    newQuery.layers = currentLayers.join(',')
    let queryString = dataObject.toQueryString()
    if (queryString !== '') {
      newQuery[layerName] = queryString
    }

    router.push({ query: newQuery }).catch(() => {})
    return this
  }

  this.close = function (layerId) {
    // use router here
    // obtain layers array from router, then removeFromLayersArray(layerId)
    // and then update router with that new array

    let currentLayers = router.currentRoute.query.layers ? router.currentRoute.query.layers.split(',') : []
    removeElement(currentLayers, layerId)
    let newQuery = JSON.parse(JSON.stringify(router.currentRoute.query)) // must clone, not reference
    newQuery.layers = currentLayers.join(',')
    delete newQuery[layerId]

    router.push({ query: newQuery }).catch(() => {})
    return this
  }

  this.getLayers = function () {
    return this.layers
  }

  return this
}

export default {
  install (Vue, options) {
    top.$lm = Vue.prototype.$layerManager = Vue.observable(new LayerManager(Vue, options.router))
  }
}
